﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/gpio/gpio.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_BatteryLot.h>
#include <nn/settings/factory/settings_Result.h>
#include <nn/settings/system/settings_Ptm.h>
#include <nne/max17050/max17050.h>

#include <nn/psm/detail/psm_Log.h>

#include "psm_ErrorReporter.h"
#include "psm_FuelGaugeCustomParameters-hardware.nx.h"
#include "psm_FuelGaugeDriver.h"
#include "psm_MacroForDriver.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

const double SenseResistorOhm = 0.005;

::nn::Result ClearInterruptImpl(uint16_t mask) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(
        ~(::nne::max17050::StatusField_VoltageTooLow
          | ::nne::max17050::StatusField_TemperatureTooLow
          | ::nne::max17050::StatusField_StateOfChargeTooLow
          | ::nne::max17050::StatusField_BatteryInsertion
          | ::nne::max17050::StatusField_VoltageTooHigh
          | ::nne::max17050::StatusField_TemperatureTooHigh
          | ::nne::max17050::StatusField_StateOfChargeTooHigh
          | ::nne::max17050::StatusField_BatteryRemoval)
        & mask, 0);

    uint16_t status;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetStatus(&status));
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetStatus(status & ~mask));
    NN_RESULT_SUCCESS;
}

const ::nne::max17050::CustomParameters& GetCustomParameters() NN_NOEXCEPT
{
    ::nn::settings::factory::BatteryLot factoryLot;

    if ((::nn::settings::factory::GetBatteryLot(&factoryLot)).IsFailure())
    {
        // エラーの場合パラメータは ATL を使用する。
        return CustomParametersForAtl;
    }

    const int batteryLotVendorIndex = 7;

    switch ( factoryLot.string[batteryLotVendorIndex] )
    {
    case 'A':
        return CustomParametersForAtl;
    case 'M':
        return CustomParametersForMaxell;
    default:
        // 未定義の文字の場合パラメータは ATL を使用する。
        return CustomParametersForAtl;
    }
}

// TODO: FuelGaugeParameterManager と実装を共有できればより良いです。
// 別途 BatteryLotAccessor といったドライバに相当するモジュールを作成して処理を集約させるべきです。

//! 電池情報が有効か否かを返します。
bool IsValidBatteryLot(::nn::settings::factory::BatteryLot& lot) NN_NOEXCEPT
{
    for ( auto i = 0; i < sizeof(lot.string); ++i )
    {
        if ( lot.string[i] != 0 )
        {
            return true;
        }
    }

    // ゼロ埋めされている場合は無効として扱います。（SIGLO-39573）
    return false;
}

//! 電池情報が完全に一致するかを返します。
bool IsSameBatteryLot(::nn::settings::factory::BatteryLot& lot1, ::nn::settings::factory::BatteryLot& lot2) NN_NOEXCEPT
{
    if ( ::std::strncmp(lot1.string, lot2.string, sizeof(lot1.string)) == 0 )
    {
        return true;
    }
    else
    {
        return false;
    }
}

} // namespace

const int FuelGaugeDriver::TemperatureThresholdCelsiusMin = -128;
const int FuelGaugeDriver::TemperatureThresholdCelsiusMax = 127;
const int FuelGaugeDriver::VoltageThresholdMillivoltMax = 5100;
const int FuelGaugeDriver::CyclesMin = 96;
const int FuelGaugeDriver::CyclesClearThreshold = 296;
const int FuelGaugeDriver::InvalidLowChargeThresholdPercentage = 0x00;
const int FuelGaugeDriver::InvalidHighChargeThresholdPercentage = 0xFF;

void FuelGaugeDriver::Initialize() NN_NOEXCEPT
{
    ::nn::gpio::Initialize();

    ::nn::gpio::OpenSession(&m_FuelGaugeInterruptSession, ::nn::gpio::GpioPadName_BattMgicIrq);
    ::nn::gpio::SetDirection(&m_FuelGaugeInterruptSession, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(&m_FuelGaugeInterruptSession, ::nn::gpio::InterruptMode_LowLevel);
    ::nn::gpio::SetInterruptEnable(&m_FuelGaugeInterruptSession, true);

    ::nn::settings::factory::BatteryLot factoryLot;
    ::nn::settings::factory::BatteryLot savedLot;

    auto result = ::nn::settings::factory::GetBatteryLot(&factoryLot);

    // ハードウェア NX 以外で較正情報の取得に失敗した場合、エラーを返さず処理を継続します。
#if (defined NN_BUILD_CONFIG_HARDWARE_NX)
    if ( result.IsFailure()
        && !nn::settings::factory::ResultCalibrationDataCrcError::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
#endif

    if ( result.IsSuccess() && IsValidBatteryLot(factoryLot) )
    {
        ::nn::settings::system::GetPtmBatteryLot(&savedLot);

        // 本体 NAND 上の電池情報と較正情報上の電池情報が異なる場合 SoftPowerOnReset を発行します。
        if ( !IsSameBatteryLot(factoryLot, savedLot) )
        {
            NN_DETAIL_PSM_INFO("Issue SoftPOR.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(::nne::max17050::IssueSoftPowerOnReset());
        }
    }

    ::nne::max17050::Initialize(GetCustomParameters());
}

void FuelGaugeDriver::Finalize() NN_NOEXCEPT
{
    ::nne::max17050::Finalize();

    ::nn::gpio::UnbindInterrupt(&m_FuelGaugeInterruptSession);
    ::nn::gpio::CloseSession(&m_FuelGaugeInterruptSession);

    ::nn::gpio::Finalize();
}

::nn::Result FuelGaugeDriver::EnableInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SwitchFuelGaugeAlerts(true));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SwitchFuelGaugeAlerts(false));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowTemperatureThreshold(
    int thresholdCelsius) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(
        thresholdCelsius,
        TemperatureThresholdCelsiusMin,
        TemperatureThresholdCelsiusMax);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMinTemperatureThresholdCelsius(
                     static_cast<int16_t>(thresholdCelsius)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighTemperatureThreshold(
    int thresholdCelsius) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(
        thresholdCelsius,
        TemperatureThresholdCelsiusMin,
        TemperatureThresholdCelsiusMax);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMaxTemperatureThresholdCelsius(
                     static_cast<int16_t>(thresholdCelsius)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowVoltageThreshold(
    int thresholdMillivolt) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(thresholdMillivolt, 0, VoltageThresholdMillivoltMax);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMinVoltageThreshold(
                     static_cast<uint16_t>(thresholdMillivolt)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighVoltageThreshold(
    int thresholdMillivolt) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(thresholdMillivolt, 0, VoltageThresholdMillivoltMax);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMaxVoltageThreshold(
                     static_cast<uint16_t>(thresholdMillivolt)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetLowChargeThreshold(
    int thresholdPercentage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(thresholdPercentage, 0, 100);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMinSocThreshold(
                     static_cast<uint8_t>(thresholdPercentage)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetHighChargeThreshold(
    int thresholdPercentage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(thresholdPercentage, 0, 100);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMaxSocThreshold(
                     static_cast<uint8_t>(thresholdPercentage)));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableLowChargeInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMinSocThreshold(InvalidLowChargeThresholdPercentage));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::DisableHighChargeInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SetMaxSocThreshold(InvalidHighChargeThresholdPercentage));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pOutInterruptEvent, &m_FuelGaugeInterruptSession));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
                     ::nne::max17050::StatusField_VoltageTooLow
                     | ::nne::max17050::StatusField_TemperatureTooLow
                     | ::nne::max17050::StatusField_StateOfChargeTooLow
                     | ::nne::max17050::StatusField_BatteryInsertion
                     | ::nne::max17050::StatusField_VoltageTooHigh
                     | ::nne::max17050::StatusField_TemperatureTooHigh
                     | ::nne::max17050::StatusField_StateOfChargeTooHigh
                     | ::nne::max17050::StatusField_BatteryRemoval));

    // 連続で割り込みを受け付けないように最後に有効化
    ::nn::gpio::ClearInterruptStatus(&m_FuelGaugeInterruptSession);
    ::nn::os::ClearSystemEvent(pInterruptEvent);
    ::nn::gpio::SetInterruptEnable(&m_FuelGaugeInterruptSession, true);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearTemperatureInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
                     ::nne::max17050::StatusField_TemperatureTooLow
                     | ::nne::max17050::StatusField_TemperatureTooHigh));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearVoltageInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
                     ::nne::max17050::StatusField_VoltageTooLow
                     | ::nne::max17050::StatusField_VoltageTooHigh));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearChargeInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
                     ::nne::max17050::StatusField_StateOfChargeTooLow
                     | ::nne::max17050::StatusField_StateOfChargeTooHigh));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearBatteryInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(ClearInterruptImpl(
                     ::nne::max17050::StatusField_BatteryInsertion
                     | ::nne::max17050::StatusField_BatteryRemoval));
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetTemperatureCelsius(
    double* pTemperature) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pTemperature);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetTemperatureCelsius(pTemperature));

    GetErrorReporter().SetBatteryTemperature(*pTemperature);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetMaxTemperatureCelsius(int* pOutMaxTemperatureCelsius) NN_NOEXCEPT
{
    int8_t maxTemperatureCelsius;
    NN_RESULT_DO(::nne::max17050::GetMaxTemperatureCelsius(&maxTemperatureCelsius));
    *pOutMaxTemperatureCelsius = maxTemperatureCelsius;
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetStatus(Status* pStatus) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStatus);

    uint16_t status;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetStatus(&status));

    pStatus->Clear();

    pStatus->isLowTemperatureInterruptGenerated = (status & ::nne::max17050::StatusField_TemperatureTooLow) != 0;
    pStatus->isHighTemperatureInterruptGenerated = (status & ::nne::max17050::StatusField_TemperatureTooHigh) != 0;

    pStatus->isLowVoltageInterruptGenerated = (status & ::nne::max17050::StatusField_VoltageTooLow) != 0;
    pStatus->isHighVoltageInterruptGenerated = (status & ::nne::max17050::StatusField_VoltageTooHigh) != 0;

    pStatus->isLowChargeInterruptGenerated = (status & ::nne::max17050::StatusField_StateOfChargeTooLow) != 0;
    pStatus->isHighChargeInterruptGenerated = (status & ::nne::max17050::StatusField_StateOfChargeTooHigh) != 0;

    pStatus->isBatteryRemovalInterruptGenerated = (status & ::nne::max17050::StatusField_BatteryRemoval) != 0;
    pStatus->isBatteryInsertionInterruptGenerated = (status & ::nne::max17050::StatusField_BatteryInsertion) != 0;

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryChargePercentage(
    double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetBatteryChargePercentage(pOutValue));

    GetErrorReporter().SetBatteryChargePercent(*pOutValue);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryChargeMilliVoltage(
    int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    uint32_t batteryCharge;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetBatteryAverageChargeMilliVoltage(&batteryCharge));
    *pOutValue = static_cast<int>(batteryCharge);

    GetErrorReporter().SetBatteryChargeVoltage(static_cast<int>(batteryCharge));

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetBatteryAgePercentage(
    double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetAgePercentage(pOutValue));

    GetErrorReporter().SetBatteryAge(*pOutValue);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SaveParameter(
    Parameter* pOutParameter) NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::SaveParameters());

    ::nne::max17050::ExportParameters(
        &pOutParameter->rcomp0,
        &pOutParameter->tempCo,
        &pOutParameter->fullCap,
        &pOutParameter->cycles,
        &pOutParameter->fullCapNom,
        &pOutParameter->iAvgEmpty,
        &pOutParameter->qrTable00,
        &pOutParameter->qrTable10,
        &pOutParameter->qrTable20,
        &pOutParameter->qrTable30);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::RestoreParameter(
    const Parameter& parameter) NN_NOEXCEPT
{
    ::nne::max17050::OverwriteParameters(
        parameter.rcomp0,
        parameter.tempCo,
        parameter.fullCap,
        parameter.cycles,
        parameter.fullCapNom,
        parameter.iAvgEmpty,
        parameter.qrTable00,
        parameter.qrTable10,
        parameter.qrTable20,
        parameter.qrTable30);

    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::RestoreParameters());

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::SetNeedToRestoreParameters(bool needToRestoreParameters) NN_NOEXCEPT
{
    return ::nne::max17050::SetNeedToRestoreParameters(needToRestoreParameters);
}

::nn::Result FuelGaugeDriver::GetNeedToRestoreParameters(bool* pNeedToRestoreParameters) NN_NOEXCEPT
{
    return ::nne::max17050::GetNeedToRestoreParameters(pNeedToRestoreParameters);
}

::nn::Result FuelGaugeDriver::GetCycles(int* pOutCycles) NN_NOEXCEPT
{
    uint16_t cycles;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetCycles(&cycles));

    *pOutCycles = cycles;

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::ClearCycles() NN_NOEXCEPT
{
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::ClearCycles());

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetAverageCurrentMilliAmpere(int* pOutValue) NN_NOEXCEPT
{
    double milliAmpere = 0.0;
    NN_DETAIL_PSM_DRIVER_RESULT_DO_RETRY(::nne::max17050::GetAverageChargeRateMilliAmp(SenseResistorOhm, &milliAmpere));
    *pOutValue = static_cast<int>(milliAmpere);
    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetVoltageFuelGaugePercentage(double* pOutPercentage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPercentage);

    return ::nne::max17050::GetSocVfPercentage(pOutPercentage);
}

::nn::Result FuelGaugeDriver::GetFullCapacityMilliAmpereH(int* pOutMilliAmpereH) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutMilliAmpereH);

    double milliAmpereH = 0.0;
    NN_RESULT_DO(::nne::max17050::GetFullCapacityMilliAmp(SenseResistorOhm, &milliAmpereH));
    *pOutMilliAmpereH = static_cast<int>(milliAmpereH);

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeDriver::GetRemainingCapacityMilliAmpereH(int* pOutMilliAmpereH) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutMilliAmpereH);

    double milliAmpereH = 0.0;
    NN_RESULT_DO(::nne::max17050::GetRemainingCapacityMilliAmp(SenseResistorOhm, &milliAmpereH));
    *pOutMilliAmpereH = static_cast<int>(milliAmpereH);

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
