﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_Mutex.h>
#include <nn/psm/detail/psm_Log.h>
#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_ChargeArbiter.h"
#include "psm_ChargeParameterRule.h"
#include "psm_IChargerDriver.h"
#include "psm_ErrorReporter.h"
#include "psm_PowerState.h"
#include "psm_SettingsHolder-spec.nx.h"

namespace nn { namespace psm { namespace driver { namespace detail {

// 方針：Arbiter（現在の名称。要改称） はどの環境でも同一のクラスを使用し内部で Spec, PlatformConfiguration で分岐する実装をする考えです。
//
// Arbiter 外側に出したいもの（Spec 依存のもの）
// - fast モード時の cap の値（別クラスに追い出して静的分岐）
// - 20% モード（別クラスに追い出して静的分岐）
// - fullcharge と 20% モードの切り替え条件（別クラスに追い出して静的分岐）
// - OTG 関連処理（システム全体の給電・放電管理マネージャを作成し InputCurrentLimit、Hi-Z と一緒にそちらで管理する）

ChargeArbiter::ChargeArbiter() NN_NOEXCEPT
    : m_PowerState(PowerState::FullAwake)
    , m_IsChargeRequested(true)
    , m_IsOtgRequested(false)
    , m_pChargerDriver()
    , m_Mutex(false)
    , m_IsFullCharge(false)
    , m_ChargeParameterRule()
    , m_ChargeCurrentLimitCapMilliAmpere(FastChargeCurrentLimitMilliAmpereAwakeStateDefault)
{
    // 何もしない
}

ChargeArbiter::~ChargeArbiter() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result ChargeArbiter::Initialize(IChargerDriver* pChargerDriver, const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pChargerDriver);
    NN_SDK_ASSERT_NOT_NULL(pSettingsHolder);

    // コンストラクタと同じ初期値の保証
    m_PowerState = PowerState::FullAwake;
    m_IsChargeRequested = true;
    m_IsOtgRequested = false;
    m_pChargerDriver = pChargerDriver;
    m_IsFullCharge = false;
    m_ChargeCurrentLimitCapMilliAmpere = FastChargeCurrentLimitMilliAmpereAwakeStateDefault;
    m_pSettingsHolder = pSettingsHolder;

    GetErrorReporter().SetFastBatteryChargingEnabled(true);

    // 充電を有効化しておく
    UpdateCharger();

    NN_RESULT_SUCCESS;
}

void ChargeArbiter::Finalize() NN_NOEXCEPT
{
    // 何もしません。
}

bool ChargeArbiter::IsBatteryCharging() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    return IsBatteryChargingInternal();
}

bool ChargeArbiter::IsChargeRequested() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    return m_IsChargeRequested;
}

bool ChargeArbiter::IsOtgRequested() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    return m_IsOtgRequested;
}

::nn::Result ChargeArbiter::SetBatteryTemperatureLevel(BatteryTemperatureLevel batteryTemperatureLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_ChargeParameterRule.SetBatteryTemperatureLevel(batteryTemperatureLevel);
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::SetBatteryVoltageLevel(BatteryVoltageLevelForCharge batteryVoltageLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_ChargeParameterRule.SetBatteryVoltageLevel(batteryVoltageLevel);
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::SetPowerState(PowerState powerState) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if ( m_PowerState != powerState )
    {
        m_ChargeParameterRule.SetPowerState(powerState);
        m_PowerState = powerState;
        NN_RESULT_DO(UpdateCharger());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::StartChargeRequest() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_IsChargeRequested = true;
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::StopChargeRequest() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_IsChargeRequested = false;
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::StartOtgRequest() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    if ( m_pSettingsHolder->GetUsbTypeCPowerSourceCircuitVersion() == 1 )
    {
        NN_RESULT_SUCCESS;
    }

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_IsOtgRequested = true;
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::StopOtgRequest() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    if ( m_pSettingsHolder->GetUsbTypeCPowerSourceCircuitVersion() == 1 )
    {
        NN_RESULT_SUCCESS;
    }

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_IsOtgRequested = false;
    NN_RESULT_DO(UpdateCharger());

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::NotifyFullCharge() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    if ( m_IsFullCharge != true )
    {
        m_IsFullCharge = true;
        NN_RESULT_DO(UpdateCharger());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::NotifyNotFullCharge() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    if ( m_IsFullCharge != false )
    {
        m_IsFullCharge = false;
        NN_RESULT_DO(UpdateCharger());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::EnableFastBatteryCharging() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if ( m_ChargeCurrentLimitCapMilliAmpere != FastChargeCurrentLimitMilliAmpereAwakeStateDefault )
    {
        m_ChargeCurrentLimitCapMilliAmpere = FastChargeCurrentLimitMilliAmpereAwakeStateDefault;
        NN_RESULT_DO(UpdateCharger());
        GetErrorReporter().SetFastBatteryChargingEnabled(true);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ChargeArbiter::DisableFastBatteryCharging() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if ( m_ChargeCurrentLimitCapMilliAmpere != FastChargeCurrentLimitMilliAmpereAwakeStateLow )
    {
        m_ChargeCurrentLimitCapMilliAmpere = FastChargeCurrentLimitMilliAmpereAwakeStateLow;
        NN_RESULT_DO(UpdateCharger());
        GetErrorReporter().SetFastBatteryChargingEnabled(false);
    }

    NN_RESULT_SUCCESS;
}

bool ChargeArbiter::IsBatteryChargingInternal() NN_NOEXCEPT
{
    // 満充電による充電停止の有効・無効はシステム電源状態に依存して変化する。
    bool fullCharge = m_IsFullCharge;

    switch (m_PowerState)
    {
    case PowerState::FullAwake:
        fullCharge = m_IsFullCharge;
        break;
    case PowerState::MinimumAwake:
    case PowerState::SleepCharge:
    case PowerState::SleepDischarge:
    case PowerState::ShutdownChargeMain:
        fullCharge = false;
        break;
    } // NOLINT(style/switch_default)

    return m_ChargeParameterRule.GetChargeAllowed() && m_IsChargeRequested && !fullCharge;
}

::nn::Result ChargeArbiter::UpdateCharger() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pChargerDriver);

    // 20 パーセントモードの有効・無効はシステム電源状態に依存して変化する。
    bool chargeCurrentLimitOption20Percent = true;

    switch (m_PowerState)
    {
    case PowerState::FullAwake:
        chargeCurrentLimitOption20Percent = true;
        break;
    case PowerState::MinimumAwake:
    case PowerState::SleepCharge:
    case PowerState::SleepDischarge:
    case PowerState::ShutdownChargeMain: // シャットダウン時に充電 IC による充電停止を有効にするため。
        chargeCurrentLimitOption20Percent = false;
        break;
    } // NOLINT(style/switch_default)

    // 本モジュールの関心対象は充電池の充電制御であり、システムの全体の給電・放電の制御ではありません。
    // TODO: EnableWatchdogTimer と EnableOnTheGo はプラットフォームに依存の強いの機能なので別モジュールに移したいです。
    if (m_IsOtgRequested)
    {
        NN_RESULT_DO(m_pChargerDriver->EnableWatchdogTimer());
        NN_RESULT_DO(m_pChargerDriver->EnableOnTheGo());
    }
    else
    {
        if ( IsBatteryChargingInternal() )
        {
            m_pChargerDriver->SetChargeEnable(true);
            GetErrorReporter().SetChargeEnabled(true);
        }
        else
        {
            m_pChargerDriver->SetChargeEnable(false);
            GetErrorReporter().SetChargeEnabled(false);
        }
        // 電源断で充電できない状態にならないように GPIO に関係なく I2C レジスタは充電許可にしておく
        NN_RESULT_DO(m_pChargerDriver->EnableBatteryCharging());
        NN_RESULT_DO(m_pChargerDriver->DisableWatchdogTimer());
    }

    // 充電電流上限の更新
    auto chargeCurrentLimit = ::std::min(::std::min(m_ChargeCurrentLimitCapMilliAmpere,
        m_ChargeParameterRule.GetChargeCurrentLimitMilliAmpere()),
        m_pSettingsHolder->GetChargeCurrentLimitMilliAmpere());

    if ( chargeCurrentLimitOption20Percent )
    {
        NN_RESULT_DO(m_pChargerDriver->SetFastChargeCurrentLimit(chargeCurrentLimit, IChargerDriver::FastChargeCurrentLimitOption::Current20Percent));
    }
    else
    {
        NN_RESULT_DO(m_pChargerDriver->SetFastChargeCurrentLimit(chargeCurrentLimit));
    }

    // 充電電圧の更新
    NN_RESULT_DO(m_pChargerDriver->SetChargeVoltageLimit(m_ChargeParameterRule.GetChargeVoltageLimitMilliVolt()));

    NN_DETAIL_PSM_TRACE("OTG: %s.\n", m_IsOtgRequested ? "Requested" : "Not requested");
    NN_DETAIL_PSM_TRACE("BatteryCharging: %s.\n", IsBatteryChargingInternal() ? "Enabled" : "Disabled");
    NN_DETAIL_PSM_TRACE("ChargeCurrentLimit: %d mA.\n", chargeCurrentLimit);
    NN_DETAIL_PSM_TRACE("ChargeVoltageLimit: %d mV.\n", m_ChargeParameterRule.GetChargeVoltageLimitMilliVolt());

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
