﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/psm/detail/psm_Log.h>
#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_BatteryTemperatureManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

const int ThresholdsCelsius[] =
{
    BatteryTemperatureCelsiusThresholdTooLow,
    BatteryTemperatureCelsiusThresholdLow,
    BatteryTemperatureCelsiusThresholdHigh,
    BatteryTemperatureCelsiusThresholdTooHigh,
};

// 連続での割り込みを防ぐために温度範囲にヒステリシスを持たせる
// 電池の保証温度範囲を守れる方向にのみヒステリシスを持たせるために以下の仕様とする
// - BatteryTemperatureLevel::High, BatteryTemperatureLevel::TooHigh に遷移するとき下限温度にヒステリシスを減算
// - BatteryTemperatureLevel::Low, BatteryTemperatureLevel::TooLow に遷移するとき上界温度にヒステリシスを加算
const int HysteresisCelsius = 3;

} // namespace

::nn::Result BatteryTemperatureManager::Initialize(
    IFuelGaugeDriver* pFuelGaugeDriver,
    ChargeArbiter* pChargeArbiter) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeDriver);
    NN_SDK_ASSERT_NOT_NULL(pChargeArbiter);

    m_pFuelGaugeDriver = pFuelGaugeDriver;
    m_pChargeArbiter = pChargeArbiter;

    double temperature;
    NN_RESULT_DO(m_pFuelGaugeDriver->GetTemperatureCelsius(&temperature));
    NN_DETAIL_PSM_TRACE(
        "Initial battery temperature: %d deg C.\n",
        static_cast<int>(temperature));

    m_TemperatureLevel = BatteryTemperatureLevel::Medium;

    // 下層のモジュールの持つ温度レベルおよびスレッショルドの更新
    UpdateInternal();

    // 温度レベルが中温でない場合は即座に割り込みで更新されます。
    NN_RESULT_DO(m_pFuelGaugeDriver->ClearTemperatureInterrupt());

    NN_RESULT_SUCCESS;
}

void BatteryTemperatureManager::Finalize() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result BatteryTemperatureManager::Update() NN_NOEXCEPT
{
    double temperature;
    NN_RESULT_DO(m_pFuelGaugeDriver->GetTemperatureCelsius(&temperature));
    auto temperatureLevel = QuantizeTemperature(temperature);

    // 割り込みクリア直後に連続で割り込みが発生した場合は何もしない
    if ( temperatureLevel == m_TemperatureLevel )
    {
        NN_DETAIL_PSM_TRACE("Double interrupt.\n");
        NN_RESULT_SUCCESS;
    }

    m_TemperatureLevel = temperatureLevel;

    return UpdateInternal();
}

::nn::Result BatteryTemperatureManager::UpdateInternal() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pChargeArbiter->SetBatteryTemperatureLevel(m_TemperatureLevel));

    auto lowThreshold = GetLowTemperatureThresholdCelsius(m_TemperatureLevel);
    auto highThreshold = GetHighTemperatureThresholdCelsius(m_TemperatureLevel);

    NN_RESULT_DO(
        m_pFuelGaugeDriver->SetLowTemperatureThreshold(lowThreshold));
    NN_RESULT_DO(
        m_pFuelGaugeDriver->SetHighTemperatureThreshold(highThreshold));
    NN_DETAIL_PSM_TRACE("Low temperature threshold: %d deg C.\n", lowThreshold);
    NN_DETAIL_PSM_TRACE("High temperature threshold: %d deg C.\n", highThreshold);

    NN_RESULT_SUCCESS;
}

BatteryTemperatureLevel BatteryTemperatureManager::QuantizeTemperature(
    double celsius) NN_NOEXCEPT
{
    const BatteryTemperatureLevel levels[] =
    {
        BatteryTemperatureLevel::TooLow,
        BatteryTemperatureLevel::Low,
        BatteryTemperatureLevel::Medium,
        BatteryTemperatureLevel::High,
    };
    const auto size = sizeof(levels) / sizeof(levels[0]);
    NN_STATIC_ASSERT(
        size == sizeof(ThresholdsCelsius) / sizeof(ThresholdsCelsius[0]));

    for (auto i = 0; i < size; ++i)
    {
        if (celsius < ThresholdsCelsius[i])
        {
            return levels[i];
        }
    }
    return BatteryTemperatureLevel::TooHigh;
}

int BatteryTemperatureManager::GetLowTemperatureThresholdCelsius(
    BatteryTemperatureLevel level) NN_NOEXCEPT
{
    if (level == BatteryTemperatureLevel::TooLow)
    {
        return FuelGaugeDriver::TemperatureThresholdCelsiusMin;
    }

    auto threshold = ThresholdsCelsius[static_cast<int>(level) - 1];
    // 電池の保証温度範囲を守れる方向にのみヒステリシスを持たせる
    if (level == BatteryTemperatureLevel::High || level == BatteryTemperatureLevel::TooHigh)
    {
        threshold -= HysteresisCelsius;
    }
    return threshold;
}

int BatteryTemperatureManager::GetHighTemperatureThresholdCelsius(
    BatteryTemperatureLevel level) NN_NOEXCEPT
{
    if (level == BatteryTemperatureLevel::TooHigh)
    {
        return FuelGaugeDriver::TemperatureThresholdCelsiusMax;
    }

    auto threshold = ThresholdsCelsius[static_cast<int>(level)];
    // 電池の保証温度範囲を守れる方向にのみヒステリシスを持たせる
    if (level == BatteryTemperatureLevel::Low || level == BatteryTemperatureLevel::TooLow)
    {
        threshold += HysteresisCelsius;
    }
    return threshold;
}

}}}} // namespace nn::psm::driver::detail
