﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/psm/driver/detail/psm_BatteryChargePercentage.h>
#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_BatteryState.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

BatteryVoltageState ConvertToBatteryVoltageState(int milliVoltage) NN_NOEXCEPT
{
    auto shutdownThreshold = BatteryState::ShutdownVoltageThresholdMilliVolt;
    auto sleepThreshold = 3200;
    auto boostPerformanceModeThreshold = 3320;

    BatteryVoltageState state = BatteryVoltageState_Good;

    if ( milliVoltage < shutdownThreshold )
    {
        state = BatteryVoltageState_ShutdownRequired;
    }
    else if ( milliVoltage < sleepThreshold )
    {
        state = BatteryVoltageState_SleepRequired;
    }
    else if ( milliVoltage < boostPerformanceModeThreshold )
    {
        state = BatteryVoltageState_BoostPerformanceModeProhibited;
    }

    return state;
}

BatteryVoltageLevelForCharge ConvertToBatteryVoltageLevelForCharge(int milliVoltage) NN_NOEXCEPT
{
    BatteryVoltageLevelForCharge level = BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature;

    if ( milliVoltage < DisallowFastSpeedChargeThresholdMilliVoltage )
    {
        level = BatteryVoltageLevelForCharge::DisallowFastSpeedCharge;
    }
    else if ( milliVoltage < AllowChargeOnHighTemperatureThresholdMilliVoltage )
    {
        level = BatteryVoltageLevelForCharge::AllowChargeOnHighTemperature;
    }
    else
    {
        level = BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature;
    }

    return level;
}

} // namespace

const int BatteryState::ShutdownVoltageThresholdMilliVolt = 3100;

BatteryState::BatteryState() NN_NOEXCEPT
    : m_RawBatteryChargePercentage(0)
    , m_ChargerType(ChargerType_Unconnected)
    , m_RawBatteryChargeMilliVoltage(3200)
    , m_IsEnoughPowerSupplied(false)
    , m_BatteryAgePercentage(100)
    , m_AverageCurrentMilliAmpere(0)
    , m_VoltageFuelGaugePercentage(0)
    , m_RawBatteryChargeOffsetPercentage(0)
    , m_BatteryChargeCalibratedEvent(nn::os::EventClearMode::EventClearMode_ManualClear, true)
{
    // 何もしない
}

BatteryState::~BatteryState() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result BatteryState::Initialize() NN_NOEXCEPT
{
    m_RawBatteryChargePercentage = 0;
    m_ChargerType = ChargerType_Unconnected;
    m_RawBatteryChargeMilliVoltage = 3200;
    m_IsEnoughPowerSupplied = false;
    m_RawBatteryChargeOffsetPercentage = 0;
    m_BatteryAgePercentage = 100;
    m_AverageCurrentMilliAmpere = 0;
    m_VoltageFuelGaugePercentage = 0;
    m_RawBatteryChargeOffsetPercentage = 0;

    NN_RESULT_SUCCESS;
}

void BatteryState::Finalize() NN_NOEXCEPT
{
    // 何もしない
}

int BatteryState::GetBatteryChargePercentage() NN_NOEXCEPT
{
    return ConvertBatteryChargePercentage(m_RawBatteryChargePercentage, MaxRawDefaultPercentage, m_RawBatteryChargeOffsetPercentage);
}

double BatteryState::GetRawBatteryChargePercentage() NN_NOEXCEPT
{
    return m_RawBatteryChargePercentage;
}

double BatteryState::GetBatteryAgePercentage() NN_NOEXCEPT
{
    return m_BatteryAgePercentage;
}

ChargerType BatteryState::GetChargerType() NN_NOEXCEPT
{
    return m_ChargerType;
}

int BatteryState::GetRawBatteryChargeMilliVoltage() NN_NOEXCEPT
{
    return m_RawBatteryChargeMilliVoltage;
}

BatteryVoltageState BatteryState::GetBatteryVoltageState() NN_NOEXCEPT
{
    return ConvertToBatteryVoltageState(m_RawBatteryChargeMilliVoltage);
}

bool BatteryState::IsEnoughPowerSupplied() NN_NOEXCEPT
{
    return m_IsEnoughPowerSupplied;
}

void BatteryState::SetRawBatteryChargePercentage(double percentage) NN_NOEXCEPT
{
    m_RawBatteryChargePercentage = percentage;
}

void BatteryState::SetBatteryAgePercentage(double percentage) NN_NOEXCEPT
{
    m_BatteryAgePercentage = percentage;
}

void BatteryState::SetChargerType(ChargerType chargerType) NN_NOEXCEPT
{
    m_ChargerType = chargerType;
}

void BatteryState::SetRawBatteryChargeMilliVoltage(int milliVoltage) NN_NOEXCEPT
{
    m_RawBatteryChargeMilliVoltage = milliVoltage;
}

void BatteryState::SetEnoughPowerSupplied(bool isEnoughPowerSupplied) NN_NOEXCEPT
{
    m_IsEnoughPowerSupplied = isEnoughPowerSupplied;
}

BatteryVoltageLevelForCharge BatteryState::GetBatteryVoltageLevelForCharge() NN_NOEXCEPT
{
    return ConvertToBatteryVoltageLevelForCharge(m_RawBatteryChargeMilliVoltage);
}

void BatteryState::SetAverageCurrentMilliAmpere(int averageCurrentMilliAmpere) NN_NOEXCEPT
{
    m_AverageCurrentMilliAmpere = averageCurrentMilliAmpere;
}

void BatteryState::SetVoltageFuelGaugePercentage(double voltageFuelGaugePercentage) NN_NOEXCEPT
{
    m_VoltageFuelGaugePercentage = voltageFuelGaugePercentage;
}

bool BatteryState::IsFullCharge() NN_NOEXCEPT
{
    const double FullChargeThresholdPercentage = 99.0;
    const int FullChargeThresholdMilliAmpere = 128;

    return (m_VoltageFuelGaugePercentage >= FullChargeThresholdPercentage)
        && (m_AverageCurrentMilliAmpere <= FullChargeThresholdMilliAmpere);
}

void BatteryState::UpdateRawBatteryChargeOffsetPercentage() NN_NOEXCEPT
{
    m_RawBatteryChargeOffsetPercentage = GetRawBatteryChargePercentage();
}

void BatteryState::TryDecreaseRawBatteryChargeOffsetPercentage() NN_NOEXCEPT
{
    m_RawBatteryChargeOffsetPercentage = ::std::min<double>(GetRawBatteryChargePercentage(), m_RawBatteryChargeOffsetPercentage);
}

nn::os::SystemEventType* BatteryState::GetBatteryChargeCalibratedEvent() NN_NOEXCEPT
{
    return m_BatteryChargeCalibratedEvent.GetBase();
}

}}}} // namespace nn::psm::driver::detail
