﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/result/result_HandlingUtility.h>

#include "../psm_SettingsHolder-spec.nx.h"
#include "psm_SystemPowerStateEventHandlerLow.h"

namespace nn { namespace psm { namespace driver { namespace detail {

SystemPowerStateEventHandlerLow::SystemPowerStateEventHandlerLow() NN_NOEXCEPT
    : m_PmModule()
    , m_PmModuleMultiWaitHolder()
    , m_pSettingsHolder(nullptr)
    , m_Vdd5v3EnSession()
    , m_VConnEnSession()
{
    // 何もしません。
}

SystemPowerStateEventHandlerLow::~SystemPowerStateEventHandlerLow() NN_NOEXCEPT
{
    // 何もしません。
}

::nn::Result SystemPowerStateEventHandlerLow::Initialize(const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    m_pSettingsHolder = pSettingsHolder;

    // 複雑化を防ぐため（例：Windows 環境のテストで本オブジェクトの PmModule に対して遷移を通知しないと PmModuleId_Psm による遷移が正常に行われない等）
    // Low の処理はハンドラ内から他マネージャを経由せず直接外部のライブラリの機能を呼び出す。
    // 処理内容が少ない現在故の処置
    ::nn::psc::PmModuleId dependencies[] =
    {
        ::nn::psc::PmModuleId_Gpio,
    };

    NN_RESULT_DO(m_PmModule.Initialize(::nn::psc::PmModuleId_PsmLow,
        dependencies, NN_ARRAY_SIZE(dependencies), ::nn::os::EventClearMode_ManualClear));

    ::nn::os::InitializeMultiWaitHolder(&m_PmModuleMultiWaitHolder, m_PmModule.GetEventPointer()->GetBase());

    if ( GetUsbTypeCPowerSourceCircuitVersion() == 1 )
    {
        InitializeVdd5v3EnAndVconnEn();
        PowerOnVdd5v3EnAndVconnEn();
    }

    NN_RESULT_SUCCESS;
}

void SystemPowerStateEventHandlerLow::Finalize() NN_NOEXCEPT
{
    if ( GetUsbTypeCPowerSourceCircuitVersion() == 1 )
    {
        PowerOffVdd5v3EnAndVconnEn();
        FinalizeVdd5v3EnAndVconnEn();
    }

    ::nn::os::FinalizeMultiWaitHolder(&m_PmModuleMultiWaitHolder);

    m_PmModule.Finalize();
}

void SystemPowerStateEventHandlerLow::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_PmModuleMultiWaitHolder);
}

void SystemPowerStateEventHandlerLow::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_PmModuleMultiWaitHolder);
}

bool SystemPowerStateEventHandlerLow::HandleEvent() NN_NOEXCEPT
{
    if ( m_PmModule.GetEventPointer()->TryWait() )
    {
        m_PmModule.GetEventPointer()->Clear();

        ::nn::psc::PmState pmState;
        ::nn::psc::PmFlagSet pmFlagSet;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.GetRequest(&pmState, &pmFlagSet));

        HandlePowerStateEvent(pmState, pmFlagSet);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PmModule.Acknowledge(pmState, ::nn::ResultSuccess()));

        return true;
    }

    return false;
}

void SystemPowerStateEventHandlerLow::HandlePowerStateEvent(::nn::psc::PmState pmState, ::nn::psc::PmFlagSet pmFlagSet) NN_NOEXCEPT
{
    NN_UNUSED(pmFlagSet);

    if ( GetUsbTypeCPowerSourceCircuitVersion() != 1 )
    {
        return;
    }

    switch ( pmState )
    {
    case ::nn::psc::PmState_MinimumAwake:
        PowerOnVdd5v3EnAndVconnEn();
        break;
    case ::nn::psc::PmState_SleepReady:
    case ::nn::psc::PmState_ShutdownReady:
        PowerOffVdd5v3EnAndVconnEn();
        break;
    default:
        break;
    }
}

int SystemPowerStateEventHandlerLow::GetUsbTypeCPowerSourceCircuitVersion() NN_NOEXCEPT
{
    auto usbTypeCPowerSourceCircuitVersion = m_pSettingsHolder->GetUsbTypeCPowerSourceCircuitVersion();

    return usbTypeCPowerSourceCircuitVersion;
}

void SystemPowerStateEventHandlerLow::InitializeVdd5v3EnAndVconnEn() NN_NOEXCEPT
{
    ::nn::gpio::Initialize();
    ::nn::gpio::OpenSession(&m_VConnEnSession, gpio::GpioPadName_PdVconnEn);
    ::nn::gpio::SetDirection(&m_VConnEnSession, gpio::Direction_Output);
    ::nn::gpio::OpenSession(&m_Vdd5v3EnSession, gpio::GpioPadName_Vdd5V3En);
    ::nn::gpio::SetDirection(&m_Vdd5v3EnSession, gpio::Direction_Output);
}

void SystemPowerStateEventHandlerLow::FinalizeVdd5v3EnAndVconnEn() NN_NOEXCEPT
{
    ::nn::gpio::CloseSession(&m_VConnEnSession);
    ::nn::gpio::CloseSession(&m_Vdd5v3EnSession);
    ::nn::gpio::Finalize();
}

void SystemPowerStateEventHandlerLow::PowerOnVdd5v3EnAndVconnEn() NN_NOEXCEPT
{
    ::nn::gpio::SetValue(&m_Vdd5v3EnSession, gpio::GpioValue_High);
    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(10));
    ::nn::gpio::SetValue(&m_VConnEnSession, gpio::GpioValue_High);
}

void SystemPowerStateEventHandlerLow::PowerOffVdd5v3EnAndVconnEn() NN_NOEXCEPT
{
    ::nn::gpio::SetValue(&m_VConnEnSession, gpio::GpioValue_Low);
    ::nn::gpio::SetValue(&m_Vdd5v3EnSession, gpio::GpioValue_Low);
}

}}}} // namespace nn::psm::driver::detail
