﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_ThreadTypes.h>

#include "profiler_ThreadListItem.h"
#include "profiler_UniqueSortedList.h"

namespace nn { namespace profiler {

typedef UniqueSortedList<::nn::os::ThreadId, ::nn::profiler::ThreadListItem> ThreadList;

class TargetApplication
{
    NN_DISALLOW_COPY(TargetApplication);

public:
    static void Initialize();
    static void Finalize();

    static TargetApplication* GetCurrent();

    static uint32_t GetCoreMask();
    static void SetCoreMask(uint32_t mask);
    static int GetCoreCount();

    static nn::os::ThreadId GetThreadId(nn::os::ThreadType* thread);
    static nn::os::ThreadType* GetThreadType(nn::os::ThreadId thread);
    static const char* GetThreadName(nn::os::ThreadType* thread);
    static int GetThreadPriority(nn::os::ThreadType* thread);
    static void GetThreadCoreMask(nn::os::ThreadType* thread, int* idealCore, int* coreMask);

    static ThreadListItem* GetThreadData(nn::os::ThreadType* thread);

    static int GetMaximumThreadPriority();

    static bool FindModuleName(
        uintptr_t baseaddr,
        uintptr_t endaddr,
        const char*& pOutName,
        size_t& pOutNameLength);
    static bool FindModuleBuildId(uintptr_t baseaddr, uintptr_t endaddr, void* gnuBuildId);

public:
    void GetApplicationName(const char*& pName, size_t& pLength) const;

    void FindCodeRegions();
    uintptr_t GetMinCodeAddress() const;
    uintptr_t GetMaxCodeAddress() const;
    int GetCodeRegionCount() const;

    bool GetStackStartFromThreadId(nn::os::ThreadType* thread, uintptr_t* stackBase) const;
    void ClearThreadList();
    ThreadListItem* RegisterThread(nn::os::ThreadType* thread);
    void RegisterAllThreads();
    ThreadList* GetThreadList();

    uint32_t GetSdkVersion() const;
    void SetSdkVersion(uint32_t version);

protected:

private:
    static uint32_t s_CoreMask;
    static int s_CoreCount;
    static TargetApplication* s_CurrentApplication;
    static int s_MaximumThreadPriority;

private:
    uint32_t m_sdkVersion;

    uint64_t m_minCodeAddress;
    uint64_t m_maxCodeAddress;
    uint64_t m_minStaticCodeAddress;
    uint64_t m_maxStaticCodeAddress;
    int m_codeRegionCount;
    int m_staticCodeRegionCount;

    ThreadList m_threadList;

private:
    TargetApplication();
    ~TargetApplication();

    void FindStaticCodeRegions();
    void FindDynamicCodeRegions();
};


}} // nn::profiler
