﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nvn/nvn.h>
#include <nvn/nvn_Cpp.h>

#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_Nvn.h"
#include "profiler_NvnNameTable.autogen.h"
#include "profiler_RecordMethods.h"
#include "profiler_WriteToBuffer.h"

/*--------------------------------------------------------------------------------*/
// Some hackery to allow for us to work even if NVN isn't linked
extern "C" {
    PFNNVNDEVICEGETPROCADDRESSPROC _LIBCPP_WEAK pfnc_nvnDeviceGetProcAddress = nullptr;
    extern PFNNVNDEVICEGETPROCADDRESSPROC pfnc_nvnDeviceGetProcAddress;
}
namespace nvn {
    DeviceGetProcAddressFunc _LIBCPP_WEAK pfncpp_nvnDeviceGetProcAddress = nullptr;
    extern DeviceGetProcAddressFunc pfncpp_nvnDeviceGetProcAddress;
    namespace objects {
        DeviceGetProcAddressFunc _LIBCPP_WEAK pfncpp_nvnDeviceGetProcAddress = nullptr;
        extern DeviceGetProcAddressFunc pfncpp_nvnDeviceGetProcAddress;
    }
}
/*--------------------------------------------------------------------------------*/

namespace nn { namespace profiler {

char *gNvnFuncTable = nullptr;

namespace /*anonymous*/ {
    const size_t NvnFunctionCount = sizeof(nvn::gNvnFunctionNames) / sizeof(*nvn::gNvnFunctionNames);
    const size_t NvnFishingCount = 20;

    size_t gNvnFuncTableWriteLocation = 0;
    size_t gNvnFuncTableSize = 0;
    bool gFoundFunctions;
} // anonymous


bool InitializeNvnSupport() NN_NOEXCEPT
{
    if (gNvnFuncTable == nullptr)
    {
        // Allow for additional NVN functions to be found.
        gNvnFuncTableSize = ((NvnFunctionCount + NvnFishingCount) * (48 + sizeof(uintptr_t))) +
            NvnFunctionPointersTableHeaderSize;
        gNvnFuncTable = reinterpret_cast<char*>(Memory::GetInstance()->Allocate(gNvnFuncTableSize));
        gNvnFuncTableWriteLocation = 0;
        gFoundFunctions = false;
    }
    return (gNvnFuncTable != nullptr);
}


void FinalizeNvnSupport() NN_NOEXCEPT
{
    Memory::GetInstance()->Free(gNvnFuncTable);
    gNvnFuncTable = nullptr;
    gNvnFuncTableWriteLocation = 0;
    gNvnFuncTableSize = 0;
    gFoundFunctions = false;
}


bool GetNvnFunctionPointers(const char** outPtr, size_t* outSize) NN_NOEXCEPT
{
    if (!gFoundFunctions)
    {
        char* writePtr = gNvnFuncTable;
        writePtr = WriteToBuffer(writePtr, static_cast<uint32_t>(1)); // version
        char* nvnFunctionCountPtr = writePtr;
        writePtr = WriteToBuffer(writePtr, static_cast<uint32_t>(NvnFunctionCount)); // function count

        *outPtr = gNvnFuncTable;
        *outSize = static_cast<size_t>(writePtr - gNvnFuncTable);
        NN_SDK_ASSERT(*outSize == NvnFunctionPointersTableHeaderSize);

        PFNNVNDEVICEGETPROCADDRESSPROC getProc = nullptr;
        if (getProc == nullptr)
        {
            getProc = pfnc_nvnDeviceGetProcAddress;
        }
        if (getProc == nullptr)
        {
            getProc = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(::nvn::pfncpp_nvnDeviceGetProcAddress);
        }
        if (getProc == nullptr)
        {
            getProc = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(::nvn::objects::pfncpp_nvnDeviceGetProcAddress);
        }
        if (getProc == nullptr)
        {
            return false;
        }

        /* NVN Function Table
         *
         * Version 1:
         * Version number (32 bits)
         * Function count (32 bits)
         * Array of 'Function count' function addresses (sizeof(uintptr_t) * 'Function count')
         * Array of 'Function count' strings (null-terminated string table
         */

        // Lookup and write addresses of known NVN functions
        const char *lastName = nullptr;
        for (size_t i = 0; i < NvnFunctionCount; ++i)
        {
            const char* name = nvn::gNvnFunctionNames[i];
            lastName = std::max(name, lastName);
            uintptr_t func = reinterpret_cast<uintptr_t>(getProc(nullptr, name));
            writePtr = WriteToBuffer(writePtr, func);
        }
        size_t foundFunctionCount = NvnFunctionCount;

        /* Look for additional NVN Functions
         *
         * This is attempted by making use of the linker's bundling of strings.
         * Two strings that are exactly the same should use the same element in the string table.
         * Because the NVN strings for loading procedures are all present in the same header, the linker
         *  should put them all one after the other inside the RO section.
         * So, we continue walking past the last known NVN function name, to see if there are more.
         * All NVN functions seems to match regex "^nvn[A-Z][A-Za-z0-9]*".
         * As they are C-style exports, we don't need to worry about mangling or spaces.
         *
         * Potential Issues:
         *  - It is possible that, if the strings were the last thing in the RO section, that this could
         *    walk off the end of the RO section. If the memory is not mapped, it would be an illegal access.
         *    The likelihood of this occurring seems low.
         */
        const char *fishingNames[NvnFishingCount] = { 0 };
        {
            const char *nextName = lastName += strlen(lastName) + 1;
            size_t fishingIndex = 0;
            bool foundFunction = true;
            while (foundFunction && fishingIndex < NvnFishingCount)
            {
                foundFunction = false;
                const char *whereNull = nullptr;
                if (nextName[0] == 'n' && nextName[1] == 'v' && nextName[2] == 'n' && isupper(nextName[3]))
                {
                    const char *c = &nextName[3];
                    while (!foundFunction)
                    {
                        if (*c == '\0')
                        {
                            foundFunction = true;
                            whereNull = c;
                        }
                        else if (isalnum(*c))
                        {
                            ++c;
                        }
                        else
                        {
                            // Not a valid NVN function. Stop looking.
                            break;
                        }
                    }
                }
                if (foundFunction)
                {
                    // Write to buffer
                    const char* name = nextName;
                    uintptr_t func = reinterpret_cast<uintptr_t>(getProc(nullptr, name));
                    writePtr = WriteToBuffer(writePtr, func);
                    ++foundFunctionCount;

                    fishingNames[fishingIndex] = nextName;
                    ++fishingIndex;
                    nextName += strlen(nextName) + 1;
                }
            }

            // Retroactively go back and update the number of NVN functions in the table.
            WriteToBuffer(nvnFunctionCountPtr, static_cast<uint32_t>(foundFunctionCount));
        }


        // Write the NVN function names.
        // First write known function names.
        for (size_t i = 0; i < NvnFunctionCount; ++i)
        {
            const char* name = nvn::gNvnFunctionNames[i];
            strcpy(writePtr, name);
            writePtr += strlen(writePtr) + 1;
            NN_SDK_ASSERT(size_t(writePtr - gNvnFuncTable) < gNvnFuncTableSize);
        }

        // Then write any additional NVN function names.
        for (size_t i = NvnFunctionCount; i < foundFunctionCount; ++i)
        {
            const char* name = fishingNames[i - NvnFunctionCount];
            strcpy(writePtr, name);
            writePtr += strlen(writePtr) + 1;
            NN_SDK_ASSERT(size_t(writePtr - gNvnFuncTable) < gNvnFuncTableSize);
        }

        gFoundFunctions = true;
        NN_SDK_ASSERT(writePtr > gNvnFuncTable);
        gNvnFuncTableWriteLocation = static_cast<size_t>(writePtr - gNvnFuncTable);
    }

    *outPtr = gNvnFuncTable;
    *outSize = gNvnFuncTableWriteLocation;
    return true;
}


}} // nn::profiler
