﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/service/prepo_Server.h>
#include <nn/prepo/detail/ipc/prepo_ServiceConfig.h>
#include <nn/prepo/detail/ipc/prepo_ServiceName.h>
#include <nn/prepo/detail/service/prepo_Capability.h>
#include <nn/prepo/detail/service/prepo_Service.h>
#include <nn/prepo/detail/service/core/prepo_WorkerThread.h>
#include <nn/prepo/detail/service/core/prepo_UploadThread.h>
#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>

namespace nn { namespace prepo { namespace service {

namespace
{
    const int ParallelExecutionCountMax = 1;

    nn::os::ThreadType g_Threads[ParallelExecutionCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStacks[ParallelExecutionCountMax][16 * 1024];

    struct PrepoServiceHipcSimpleAllInOneServerManagerOption
    {
        // これ以上のサイズのデータは、[BufferTransferMode(BufferTransferMode.MapAlias)] でマップ転送に切り替えること。
        static const size_t PointerTransferBufferSize = 128;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = 64;
        static const int ObjectInSubDomainCountMax = 16;
    };

    class PrepoServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<64, detail::ipc::ServiceCount,
            PrepoServiceHipcSimpleAllInOneServerManagerOption>
    {
    };

    PrepoServiceServerManager* g_ServerManager = nullptr;
    std::aligned_storage<sizeof (PrepoServiceServerManager)>::type g_ServerManagerStorage;
}

namespace
{
    void RegisterService(int index, const char* serviceName, Bit32 capability) NN_NOEXCEPT
    {
        detail::service::InitializeService(index, serviceName, capability);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager->RegisterObjectForPort(detail::service::GetPrepoService(index),
            detail::ipc::SessionCountMax, serviceName));
    }

    void IpcDispatcher(void*) NN_NOEXCEPT
    {
        g_ServerManager->LoopAuto();
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_PREPO_INFO("[prepo] Start service...\n");

    detail::service::core::FileSystem::MountAll();

    g_ServerManager = new (&g_ServerManagerStorage) PrepoServiceServerManager;

    // prepo:u
    RegisterService(0, detail::ipc::ServiceNameForGeneric,
        detail::service::Capability::Flag_Generic);

    // prepo:s
    RegisterService(1, detail::ipc::ServiceNameForSystemUser,
        detail::service::Capability::Flag_System);

    // prepo:m
    RegisterService(2, detail::ipc::ServiceNameForManager,
        detail::service::Capability::Flag_System | detail::service::Capability::Flag_Manage);

    // prepo:a
    RegisterService(3, detail::ipc::ServiceNameForAdministrator,
        detail::service::Capability::Flag_All);

    g_ServerManager->Start();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i],
            IpcDispatcher, nullptr, g_ThreadStacks[i], sizeof (g_ThreadStacks[i]), NN_SYSTEM_THREAD_PRIORITY(prepo, IpcServer)));

        nn::os::SetThreadNamePointer(&g_Threads[i], NN_SYSTEM_THREAD_NAME(prepo, IpcServer));
        nn::os::StartThread(&g_Threads[i]);
    }

    detail::service::core::UploadThread::Start();
    detail::service::core::WorkerThread::Start();

    NN_DETAIL_PREPO_INFO("[prepo] Start service... done!\n");
}

void StopServer() NN_NOEXCEPT
{
    // g_ServerManager->RequestStop();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        nn::os::DestroyThread(&g_Threads[i]);
    }

    g_ServerManager->~PrepoServiceServerManager();
    g_ServerManager = nullptr;

    detail::service::FinalizeServices();

    // detail::service::core::UploadThread::Stop();
    // detail::service::core::WorkerThread::Stop();

    detail::service::core::FileSystem::UnmountAll();

    NN_DETAIL_PREPO_INFO("[prepo] End service.\n");
}

bool GetHalfAwakeScheduleInterval(nn::bgtc::Interval* outSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSeconds);
    NN_UNUSED(outSeconds);

    // TODO: 対応するかも。

    return false;
}

void NotifySystemHalfAwaked() NN_NOEXCEPT
{
}

void NotifySystemFullAwaked() NN_NOEXCEPT
{
    detail::service::core::UploadThread::NotifySystemFullAwaked();
}

void NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
{
    detail::service::core::UploadThread::NotifySystemSleepReady(isFromFullAwake);
}

void NotifySystemShutdownReady() NN_NOEXCEPT
{
    g_ServerManager->RequestStop();
}

void WaitForSleepReady() NN_NOEXCEPT
{
}

void WaitForShutdownReady() NN_NOEXCEPT
{
    detail::service::core::WorkerThread::Stop();
    detail::service::core::UploadThread::Stop();
}

void NotifyNetworkConnected() NN_NOEXCEPT
{
    // TODO: 対応するかも。
}

void NotifyUpdateNintendoAccountCacheResult(const nn::account::Uid& uid, nn::Result result) NN_NOEXCEPT
{
    detail::service::core::UserAgreementChecker::GetInstance().NotifyUpdateNintendoAccountCacheResult(uid, result);
}

}}}
