﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/util/prepo_HttpErrorHandler.h>
#include <nn/prepo/prepo_ResultPrivate.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace util {

nn::Result HandleHttpError(CURLcode code) NN_NOEXCEPT
{
    if (code == CURLE_OK)
    {
        return nn::ResultSuccess();
    }

    switch (code)
    {
    case CURLE_UNSUPPORTED_PROTOCOL:
        return ResultHttpErrorUnsupportedProtocol();
    case CURLE_FAILED_INIT:
        return ResultHttpErrorFailedInit();
    case CURLE_URL_MALFORMAT:
        return ResultHttpErrorUrlMalformat();
    case CURLE_NOT_BUILT_IN:
        return ResultHttpErrorNotBuiltIn();
    case CURLE_COULDNT_RESOLVE_PROXY:
        return ResultHttpErrorCouldntResolveProxy();
    case CURLE_COULDNT_RESOLVE_HOST:
        return ResultHttpErrorCouldntResolveHost();
    case CURLE_COULDNT_CONNECT:
        return ResultHttpErrorCouldntConnect();
    case CURLE_REMOTE_ACCESS_DENIED:
        return ResultHttpErrorRemoteAccessDenied();
    case CURLE_HTTP2:
        return ResultHttpErrorHttp2();
    case CURLE_PARTIAL_FILE:
        return ResultHttpErrorPartialFile();
    case CURLE_QUOTE_ERROR:
        return ResultHttpErrorQuoteError();
    case CURLE_HTTP_RETURNED_ERROR:
        return ResultHttpErrorHttpReturnedError();
    case CURLE_WRITE_ERROR:
        return ResultHttpErrorWriteError();
    case CURLE_UPLOAD_FAILED:
        return ResultHttpErrorUploadFailed();
    case CURLE_READ_ERROR:
        return ResultHttpErrorReadError();
    case CURLE_OUT_OF_MEMORY:
        return ResultHttpErrorOutOfMemory();
    case CURLE_OPERATION_TIMEDOUT:
        return ResultHttpErrorOperationTimedout();
    case CURLE_RANGE_ERROR:
        return ResultHttpErrorRangeError();
    case CURLE_HTTP_POST_ERROR:
        return ResultHttpErrorHttpPostError();
    case CURLE_SSL_CONNECT_ERROR:
        return ResultHttpErrorSslConnectError();
    case CURLE_BAD_DOWNLOAD_RESUME:
        return ResultHttpErrorBadDownloadResume();
    case CURLE_FUNCTION_NOT_FOUND:
        return ResultHttpErrorFunctionNotFound();
    case CURLE_ABORTED_BY_CALLBACK:
        return ResultHttpErrorAbortedByCallback();
    case CURLE_BAD_FUNCTION_ARGUMENT:
        return ResultHttpErrorBadFunctionArgument();
    case CURLE_INTERFACE_FAILED:
        return ResultHttpErrorInterfaceFailed();
    case CURLE_TOO_MANY_REDIRECTS:
        return ResultHttpErrorTooManyRedirects();
    case CURLE_UNKNOWN_OPTION:
        return ResultHttpErrorUnknownOption();
    case CURLE_PEER_FAILED_VERIFICATION:
        return ResultHttpErrorPeerFailedVerification();
    case CURLE_GOT_NOTHING:
        return ResultHttpErrorGotNothing();
    case CURLE_SSL_ENGINE_NOTFOUND:
        return ResultHttpErrorSslEngineNotFound();
    case CURLE_SSL_ENGINE_SETFAILED:
        return ResultHttpErrorSslEngineSetFailed();
    case CURLE_SEND_ERROR:
        return ResultHttpErrorSendError();
    case CURLE_RECV_ERROR:
        return ResultHttpErrorRecvError();
    case CURLE_SSL_CERTPROBLEM:
        return ResultHttpErrorSslCertProblem();
    case CURLE_SSL_CIPHER:
        return ResultHttpErrorSslCipher();
    case CURLE_SSL_CACERT:
        return ResultHttpErrorSslCaCert();
    case CURLE_BAD_CONTENT_ENCODING:
        return ResultHttpErrorBadContentEncoding();
    case CURLE_FILESIZE_EXCEEDED:
        return ResultHttpErrorFileSizeExceeded();
    case CURLE_USE_SSL_FAILED:
        return ResultHttpErrorUseSslFailed();
    case CURLE_SEND_FAIL_REWIND:
        return ResultHttpErrorSendFailRewind();
    case CURLE_SSL_ENGINE_INITFAILED:
        return ResultHttpErrorSslEngineInitFailed();
    case CURLE_LOGIN_DENIED:
        return ResultHttpErrorLoginDenied();
    case CURLE_CONV_FAILED:
        return ResultHttpErrorConvFailed();
    case CURLE_CONV_REQD:
        return ResultHttpErrorConvReqd();
    case CURLE_SSL_CACERT_BADFILE:
        return ResultHttpErrorSslCaCertBadFile();
    case CURLE_SSL_SHUTDOWN_FAILED:
        return ResultHttpErrorSslShutdownFailed();
    case CURLE_AGAIN:
        return ResultHttpErrorAgain();
    case CURLE_SSL_CRL_BADFILE:
        return ResultHttpErrorSslCrlBadFile();
    case CURLE_SSL_ISSUER_ERROR:
        return ResultHttpErrorSslIssuerError();
    case CURLE_CHUNK_FAILED:
        return ResultHttpErrorChunkFailed();
    case CURLE_NO_CONNECTION_AVAILABLE:
        return ResultHttpErrorNoConnectionAvailable();
    case CURLE_SSL_PINNEDPUBKEYNOTMATCH:
        return ResultHttpErrorSslPinnedPubkeyNotMatch();
    case CURLE_SSL_INVALIDCERTSTATUS:
        return ResultHttpErrorSslInvalidCertStatus();
    case CURLE_SSL_CTX_FUNCTION_NOT_SET:
        return ResultHttpErrorSslCtxFunctionNotSet();
    case CURLE_SSL_CTX_INVALID:
        return ResultHttpErrorSslCtxInvalid();
    case CURLE_SSL_CTX_FATAL:
        return ResultHttpErrorSslCtxFatal();
    case CURLE_SSL_ALREADY_CONNECTED:
        return ResultHttpErrorSslAlreadyConnected();
    case CURLE_SSL_INVALID_REFERENCE:
        return ResultHttpErrorSslInvalidReference();
    default:
        return ResultUnexpectedHttpError();
    }
} // NOLINT(impl/function_size)

namespace
{
    inline nn::Result HandleHttpStatusCode3xx(int statusCode) NN_NOEXCEPT
    {
        switch (statusCode)
        {
        case 301:
            return ResultServerError301();
        case 302:
            return ResultServerError302();
        case 303:
            return ResultServerError303();
        case 305:
            return ResultServerError305();
        case 307:
            return ResultServerError307();
        case 308:
            return ResultServerError308();
        default:
            return ResultServerError3xx();
        }
    }

    inline nn::Result HandleHttpStatusCode4xx(int statusCode) NN_NOEXCEPT
    {
        switch (statusCode)
        {
        case 400:
            return ResultServerError400();
        case 401:
            return ResultServerError401();
        case 403:
            return ResultServerError403();
        case 404:
            return ResultServerError404();
        case 405:
            return ResultServerError405();
        case 407:
            return ResultServerError407();
        case 410:
            return ResultServerError410();
        case 416:
            return ResultServerError416();
        default:
            return ResultServerError4xx();
        }
    }

    inline nn::Result HandleHttpStatusCode5xx(int statusCode) NN_NOEXCEPT
    {
        switch (statusCode)
        {
        case 500:
            return ResultServerError500();
        case 503:
            return ResultServerError503();
        case 504:
            return ResultServerError504();
        default:
            return ResultServerError5xx();
        }
    }
}

nn::Result HandleHttpStatusCode(int statusCode) NN_NOEXCEPT
{
    const int category = statusCode / 100;

    switch (category)
    {
    case 2:
        return nn::ResultSuccess();
    case 3:
        return HandleHttpStatusCode3xx(statusCode);
    case 4:
        return HandleHttpStatusCode4xx(statusCode);
    case 5:
        return HandleHttpStatusCode5xx(statusCode);
    default:
        return ResultServerErrorXxx();
    }
}

}}}}}
