﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pl/srv/pl_SharedFontManagerServer.h>
#include <nn/pl/pl_SharedFontApi.h>
#include <nn/pl/detail/pl_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <mutex>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include "detail/pl_GetSharedFontData-os.win32.h"
#endif

namespace nn { namespace pl { namespace srv {

namespace {

    const size_t FontLoadThreadStackSize = 8192;
    NN_OS_ALIGNAS_GUARDED_STACK uint8_t g_FontLoadThreadStack[FontLoadThreadStackSize];
    nn::os::ThreadType g_FontLoadThread;

    const size_t FontFileReadTempBufferSize = 8192;
    char g_FontFileReadTempBuffer[FontFileReadTempBufferSize];

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::ncm::SystemDataId GetFontSystemDataId(int fontType) NN_NOEXCEPT
    {
        switch( fontType )
        {
        case pl::SharedFontTypeForSystem_Standard:
            return nn::ncm::SystemDataId{ NN_SYSTEM_DATA_FONT_STANDARD_ID };
        case pl::SharedFontTypeForSystem_ChineseSimple:
        case pl::SharedFontTypeForSystem_ChineseSimpleExtension:
            return nn::ncm::SystemDataId{ NN_SYSTEM_DATA_FONT_CHINESE_SIMPLE_ID };
        case pl::SharedFontTypeForSystem_ChineseTraditional:
            return nn::ncm::SystemDataId{ NN_SYSTEM_DATA_FONT_CHINESE_TRADITIONAL_ID };
        case pl::SharedFontTypeForSystem_Korean:
            return nn::ncm::SystemDataId{ NN_SYSTEM_DATA_FONT_KOREAN_ID };
        case pl::SharedFontTypeForSystem_NintendoExtension:
        case pl::SharedFontTypeForSystem_NintendoExtension2:
            return nn::ncm::SystemDataId{ NN_SYSTEM_DATA_FONT_NINTENDO_EXTENSION_ID };
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
#endif

    int GetFontTypes(SharedFontType outTypes[], size_t outTypesCount, const settings::LanguageCode& languageCode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outTypes);
        NN_SDK_ASSERT_GREATER_EQUAL(outTypesCount, static_cast<size_t>(SharedFontType_Max));
        NN_UNUSED(outTypesCount);
        int count = 0;
        if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Chinese).string, sizeof(languageCode.string)) == 0 ||
            util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_SimplifiedChinese).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontType_NintendoExtension;
            outTypes[count++] = SharedFontType_ChineseSimpleExtension;
            outTypes[count++] = SharedFontType_ChineseSimple;
            outTypes[count++] = SharedFontType_ChineseTraditional;
            outTypes[count++] = SharedFontType_Korean;
            outTypes[count++] = SharedFontType_Standard;
            return count;
        }
        else if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Korean).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontType_NintendoExtension;
            outTypes[count++] = SharedFontType_Korean;
            outTypes[count++] = SharedFontType_ChineseSimpleExtension;
            outTypes[count++] = SharedFontType_ChineseSimple;
            outTypes[count++] = SharedFontType_ChineseTraditional;
            outTypes[count++] = SharedFontType_Standard;
            return count;
        }
        else if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Taiwanese).string, sizeof(languageCode.string)) == 0 ||
            util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_TraditionalChinese).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontType_NintendoExtension;
            outTypes[count++] = SharedFontType_ChineseTraditional;
            outTypes[count++] = SharedFontType_ChineseSimpleExtension;
            outTypes[count++] = SharedFontType_ChineseSimple;
            outTypes[count++] = SharedFontType_Korean;
            outTypes[count++] = SharedFontType_Standard;
            return count;
        }
        else
        {
            outTypes[count++] = SharedFontType_NintendoExtension;
            outTypes[count++] = SharedFontType_Standard;
            outTypes[count++] = SharedFontType_ChineseSimpleExtension;
            outTypes[count++] = SharedFontType_ChineseSimple;
            outTypes[count++] = SharedFontType_ChineseTraditional;
            outTypes[count++] = SharedFontType_Korean;
            return count;
        }
    }

    int GetFontTypesForSystem(SharedFontTypeForSystem outTypes[], size_t outTypesSize, const settings::LanguageCode& languageCode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outTypes);
        NN_SDK_REQUIRES_EQUAL(outTypesSize, static_cast<size_t>(SharedFontTypeForSystem_Max));
        NN_UNUSED(outTypesSize);
        int count = 0;
        if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Chinese).string, sizeof(languageCode.string)) == 0 ||
            util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_SimplifiedChinese).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontTypeForSystem_NintendoExtension2;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimpleExtension;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimple;
            outTypes[count++] = SharedFontTypeForSystem_ChineseTraditional;
            outTypes[count++] = SharedFontTypeForSystem_Korean;
            outTypes[count++] = SharedFontTypeForSystem_Standard;
            return count;
        }
        else if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Korean).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontTypeForSystem_NintendoExtension2;
            outTypes[count++] = SharedFontTypeForSystem_Korean;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimpleExtension;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimple;
            outTypes[count++] = SharedFontTypeForSystem_ChineseTraditional;
            outTypes[count++] = SharedFontTypeForSystem_Standard;
            return count;
        }
        else if( util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_Taiwanese).string, sizeof(languageCode.string)) == 0 ||
            util::Strncmp(languageCode.string, settings::LanguageCode::Make(settings::Language_TraditionalChinese).string, sizeof(languageCode.string)) == 0 )
        {
            outTypes[count++] = SharedFontTypeForSystem_NintendoExtension2;
            outTypes[count++] = SharedFontTypeForSystem_ChineseTraditional;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimpleExtension;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimple;
            outTypes[count++] = SharedFontTypeForSystem_Korean;
            outTypes[count++] = SharedFontTypeForSystem_Standard;
            return count;
        }
        else
        {
            outTypes[count++] = SharedFontTypeForSystem_NintendoExtension2;
            outTypes[count++] = SharedFontTypeForSystem_Standard;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimpleExtension;
            outTypes[count++] = SharedFontTypeForSystem_ChineseSimple;
            outTypes[count++] = SharedFontTypeForSystem_ChineseTraditional;
            outTypes[count++] = SharedFontTypeForSystem_Korean;
            return count;
        }
    }

} // anonymous namespace

void SharedFontManagerServer::FontLoadThreadFunc(void* arg) NN_NOEXCEPT
{
    auto server = static_cast<SharedFontManagerServer*>(arg);

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const char* MountName = "SharedFont";

    // 定義順は pl_SharedFontApi.h の SharedFontType の順に合わせること
    const char* SharedFontName[] = {
        "nintendo_udsg-r_std_003.bfttf",        // Standard（日・米・欧）
        "nintendo_udsg-r_org_zh-cn_003.bfttf",  // ChineseSimple（中国語簡体字）
        "nintendo_udsg-r_ext_zh-cn_003.bfttf",  // ChineseSimpleExtension（中国語簡体字外字）
        "nintendo_udjxh-db_zh-tw_003.bfttf",    // ChineseTraditional（中国語繁体字）
        "nintendo_udsg-r_ko_003.bfttf",         // Korean（ハングル）
        "nintendo_ext_003.bfttf",               // NintendoExtension（任天堂外字）
        "nintendo_ext2_003.bfttf",              // NintendoExtension2（任天堂外字 4.0.0 NUP以降システム用）
    };
#endif

    int64_t readSizeSum = 0;
    for( int loadCount = 0; loadCount < pl::SharedFontTypeForSystem_Max; loadCount++ )
    {
        util::optional<int> loadIndex = server->GetNextLoadIndex();
        if( !loadIndex )
        {
            break;
        }
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::ncm::SystemDataId sharedFontId = GetFontSystemDataId(*loadIndex);
        size_t requiredCacheBufferSize;
        auto queryCacheSizeResult = nn::fs::QueryMountSystemDataCacheSize(&requiredCacheBufferSize, sharedFontId);
        if( queryCacheSizeResult.IsFailure() )
        {
            NN_DETAIL_PL_ERROR("SharedFontManager: Failed to query cache size to mount 0x%016llx (0x%08x). Shared fonts are unavailable.\n",
                sharedFontId, queryCacheSizeResult.GetInnerValueForDebug());
            return;
        }
        const size_t MountSystemDataCacheBufferSize = 256;
        char mountSystemDataCacheBuffer[MountSystemDataCacheBufferSize];
        NN_SDK_ASSERT_LESS_EQUAL(requiredCacheBufferSize, MountSystemDataCacheBufferSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSystemData(MountName, sharedFontId, mountSystemDataCacheBuffer, requiredCacheBufferSize));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        const int FileNameLengthMax = 64;
        char fileNameBuffer[FileNameLengthMax];
        nn::util::SNPrintf(fileNameBuffer, FileNameLengthMax, "%s:/%s", MountName, SharedFontName[*loadIndex]);

        nn::fs::FileHandle fileHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, fileNameBuffer, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

        int64_t fileSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));

        nn::Bit8* currentFontBuffer = server->m_SharedFontBuffer + static_cast<int32_t>(readSizeSum);

        // 共有メモリに直接は読み込めないので、一時バッファに読み込んでからコピー
        int64_t fileOffset = 0;
        while( fileOffset != fileSize )
        {
            size_t readSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, fileOffset, g_FontFileReadTempBuffer, FontFileReadTempBufferSize));
            memcpy(currentFontBuffer + fileOffset, g_FontFileReadTempBuffer, readSize);
            fileOffset += readSize;
        }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        nn::Bit8* currentFontBuffer = server->m_SharedFontBuffer + static_cast<int32_t>(readSizeSum);
        int64_t fileSize = detail::GetFontSize(static_cast<SharedFontTypeForSystem>(*loadIndex));
        auto fontAddr = detail::GetFontData(static_cast<SharedFontTypeForSystem>(*loadIndex));
        memcpy(currentFontBuffer, fontAddr, static_cast<size_t>(fileSize));
#endif

        // 共有フォント は bfttf。利用者には ttf として渡すため、デコードする。
        currentFontBuffer = static_cast<nn::Bit8*>(nn::fontll::ScalableFontEngineHelper::Decode(currentFontBuffer, static_cast<uint32_t>(fileSize)));

        // ttf にデコードしたフォントのサイズは bfttf のサイズ -8。
        server->m_DataSize[*loadIndex] = static_cast<size_t>(fileSize) - 8;
        server->m_AddressOffset[*loadIndex] = static_cast<int32_t>(reinterpret_cast<nn::Bit8*>(currentFontBuffer) - server->m_SharedFontBuffer);
        server->m_LoadState[*loadIndex] = SharedFontLoadState_Loaded;

        readSizeSum += fileSize;

        NN_DETAIL_PL_TRACE("SharedFontManager : SharedFont[%d] is loaded.\n", *loadIndex);
    }
}

SharedFontManagerServer::SharedFontManagerServer() NN_NOEXCEPT :
    m_NativeHandle(nn::os::InvalidNativeHandle),
    m_LoadMutex(false)
{
    for( int i = 0; i < pl::SharedFontType_Max; i++ )
    {
        m_LoadState[i] = SharedFontLoadState_Loading;
        m_DataSize[i] = 0;
        m_AddressOffset[i] = 0;
    }

    m_LoadRequestCount = 0;
}

SharedFontManagerServer::~SharedFontManagerServer() NN_NOEXCEPT
{
    if( m_SharedMemoryType._state == nn::os::SharedMemoryType::State_Mapped )
    {
        nn::os::WaitThread(&g_FontLoadThread);
        nn::os::DestroyThread(&g_FontLoadThread);

        nn::os::UnmapSharedMemory(&m_SharedMemoryType);
        nn::os::DestroySharedMemory(&m_SharedMemoryType);
    }
    if( m_ExtraSharedMemoryType._state == nn::os::SharedMemoryType::State_Mapped )
    {
        nn::os::UnmapSharedMemory(&m_ExtraSharedMemoryType);
        nn::os::DestroySharedMemory(&m_ExtraSharedMemoryType);
    }
}

nn::Result SharedFontManagerServer::Initialize() NN_NOEXCEPT
{
    if( m_SharedMemoryType._state == nn::os::SharedMemoryType::State_Mapped )
    {
        NN_DETAIL_PL_INFO("SharedFontManager: Already initialized.\n");
        NN_RESULT_SUCCESS;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSharedMemory(&m_SharedMemoryType, nn::pl::detail::SharedFontMemorySize, nn::os::MemoryPermission_ReadWrite, nn::os::MemoryPermission_ReadOnly));
    m_NativeHandle = nn::os::GetSharedMemoryHandle(&m_SharedMemoryType);
    m_SharedFontBuffer = reinterpret_cast<Bit8*>(nn::os::MapSharedMemory(&m_SharedMemoryType, nn::os::MemoryPermission_ReadWrite));
    memset(m_SharedFontBuffer, 0, nn::pl::detail::SharedFontMemorySize);

    NN_DETAIL_PL_INFO("SharedFontManager: %dKB of shared memory is allocated for shared fonts.\n", nn::pl::detail::SharedFontMemorySize / 1024);

    // SIGLO-76302
    //  将来的に使う可能性のある追加の 2MB 分の共有フォント用の共有メモリを
    //  確保しておく。ただし、旧 SDK でビルド済みのアプリでは、この追加分の
    //  共有メモリは使用できないため、現時点ではマップだけして放置しておく。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSharedMemory(&m_ExtraSharedMemoryType, nn::pl::detail::ExtraSharedFontMemorySize, nn::os::MemoryPermission_ReadWrite, nn::os::MemoryPermission_ReadOnly));
    m_ExtraSharedFontBuffer = reinterpret_cast<Bit8*>(nn::os::MapSharedMemory(&m_ExtraSharedMemoryType, nn::os::MemoryPermission_ReadWrite));
    memset(m_ExtraSharedFontBuffer, 0, nn::pl::detail::ExtraSharedFontMemorySize);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_FontLoadThread, FontLoadThreadFunc, this, g_FontLoadThreadStack, FontLoadThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(pl, LoadSharedFont)));
    nn::os::SetThreadNamePointer(&g_FontLoadThread, NN_SYSTEM_THREAD_NAME(pl, LoadSharedFont));
    nn::os::StartThread(&g_FontLoadThread);

    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::RequestLoad(std::int32_t sharedFontType) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LoadMutex);
    if( m_LoadRequestCount >= pl::SharedFontType_Max )
    {
        // 既に全フォント数分リクエストが出ている場合は何もしない。
        NN_RESULT_SUCCESS;
    }
    if( m_LoadState[sharedFontType] == SharedFontLoadState_Loaded )
    {
        // ロード済みフォントが指定された場合は何もしない
        NN_RESULT_SUCCESS;
    }
    for( int i = 0; i < m_LoadRequestCount; i++ )
    {
        if( m_LoadRequestQueue[i] == sharedFontType )
        {
            // 既に同じフォントへのリクエストがある場合には何もしない
            NN_RESULT_SUCCESS;
        }
    }
    m_LoadRequestQueue[m_LoadRequestCount] = static_cast<SharedFontType>(sharedFontType);
    m_LoadRequestCount++;
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetLoadState(std::int32_t sharedFontType, nn::sf::Out<std::int32_t> loadState) const NN_NOEXCEPT
{
    *loadState = static_cast<int32_t>(m_LoadState[sharedFontType]);
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetSize(std::int32_t sharedFontType, nn::sf::Out<std::int32_t> size) const NN_NOEXCEPT
{
    *size = static_cast<int32_t>(m_DataSize[sharedFontType]);
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetSharedMemoryAddressOffset(std::int32_t sharedFontType, nn::sf::Out<std::int32_t> offset) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_SharedFontBuffer != nullptr);
    *offset = static_cast<int32_t>(m_AddressOffset[sharedFontType]);
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetSharedMemoryNativeHandle(nn::sf::Out<nn::sf::NativeHandle> outHandle) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_SharedFontBuffer != nullptr);
    outHandle.Set(nn::sf::NativeHandle(m_NativeHandle, false));
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetSharedFontInOrderOfPriority(nn::sf::Out<bool> outAllFontLoaded, nn::sf::Out<std::int32_t> outFontCount,
    const nn::sf::OutArray<std::int32_t>& outTypes, const nn::sf::OutArray<std::int32_t>& outOffsets, const nn::sf::OutArray<std::int32_t>& outSizes, nn::settings::LanguageCode languageCode) NN_NOEXCEPT
{
    if( !IsAllFontLoaded() )
    {
        *outAllFontLoaded = false;
        NN_RESULT_SUCCESS;
    }
    *outAllFontLoaded = true;

    SharedFontType types[pl::SharedFontType_Max];
    *outFontCount = GetFontTypes(types, NN_ARRAY_SIZE(types), languageCode);

    auto outCount = std::min(static_cast<int>(outTypes.GetLength()), *outFontCount);
    for( int i = 0; i < outCount; i++ )
    {
        outTypes[i] = types[i];
    }

    GetFontInOrder(outOffsets.GetData(), outSizes.GetData(), types, outCount);
    NN_RESULT_SUCCESS;
}

nn::Result SharedFontManagerServer::GetSharedFontInOrderOfPriorityForSystem(nn::sf::Out<bool> outAllFontLoaded, nn::sf::Out<std::int32_t> outFontCount,
    const nn::sf::OutArray<std::int32_t>& outTypes, const nn::sf::OutArray<std::int32_t>& outOffsets, const nn::sf::OutArray<std::int32_t>& outSizes, nn::settings::LanguageCode languageCode) NN_NOEXCEPT
{
    if( !IsAllFontLoaded() )
    {
        *outAllFontLoaded = false;
        NN_RESULT_SUCCESS;
    }
    *outAllFontLoaded = true;

    SharedFontTypeForSystem types[SharedFontTypeForSystem_Max];
    *outFontCount = GetFontTypesForSystem(types, NN_ARRAY_SIZE(types), languageCode);

    auto outCount = std::min(static_cast<int>(outTypes.GetLength()), *outFontCount);
    for( int i = 0; i < outCount; i++ )
    {
        outTypes[i] = types[i];
    }

    GetFontInOrderForSystem(outOffsets.GetData(), outSizes.GetData(), types, outCount);
    NN_RESULT_SUCCESS;
}

bool SharedFontManagerServer::IsAllFontLoaded() const NN_NOEXCEPT
{
    for( int i = 0; i < NN_ARRAY_SIZE(m_LoadState); i++ )
    {
        if( m_LoadState[i] != SharedFontLoadState_Loaded )
        {
            return false;
        }
    }
    return true;
}

util::optional<int> SharedFontManagerServer::GetNextLoadIndex() const NN_NOEXCEPT
{
    // RequestQueue の中に未読み込みのフォントが存在すればそれを選択。なければ enum SharedFontType の定義順で読み込んでいないフォントを選択。
    {
        std::lock_guard<nn::os::Mutex> lock(m_LoadMutex);
        for( int i = 0; i < m_LoadRequestCount; i++ )
        {
            if( m_LoadState[m_LoadRequestQueue[i]] == SharedFontLoadState_Loading )
            {
                return m_LoadRequestQueue[i];
            }
        }
    }

    for( int i = 0; i < pl::SharedFontTypeForSystem_Max; i++ )
    {
        if( m_LoadState[i] == SharedFontLoadState_Loading )
        {
            return i;
        }
    }

    return util::optional<int>();
}

void SharedFontManagerServer::GetFontInOrder(int32_t* outOffsets, int32_t* outSizes, SharedFontType* types, int typeCount) const NN_NOEXCEPT
{
    for( int i = 0; i < typeCount; i++ )
    {
        outOffsets[i] = m_AddressOffset[types[i]];
        outSizes[i] = static_cast<int32_t>(m_DataSize[types[i]]);
    }
}

void SharedFontManagerServer::GetFontInOrderForSystem(int32_t* outOffsets, int32_t* outSizes, SharedFontTypeForSystem* types, int typeCount) const NN_NOEXCEPT
{
    for( int i = 0; i < typeCount; i++ )
    {
        outOffsets[i] = m_AddressOffset[types[i]];
        outSizes[i] = static_cast<int32_t>(m_DataSize[types[i]]);
    }
}

}}} // nn::pl::srv
