﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/pl/pl_SharedFontApi.h>
#include <nn/pl/pl_SharedFontApiForSystem.h>
#include <nn/pl/detail/pl_ISharedFontManager.sfdl.h>
#include <nn/pl/srv/pl_ServiceName.h>
#include <nn/pl/srv/pl_SharedFontManagerServer.h>
#include <nn/pl/srv/pl_SharedFontManagerServerFactory.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_SharedMemory.h>

namespace nn { namespace pl {

namespace {

    struct AllocatorTag;
    typedef nn::sf::ExpHeapStaticAllocator<1024, AllocatorTag> MyAllocator;
    // MyAllocator を静的コンストラクタで初期化するためのヘルパー
    class MyAllocatorInitializer
    {
    public:
        MyAllocatorInitializer() NN_NOEXCEPT
        {
            MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
        }
    };

#if defined( NN_BUILD_CONFIG_OS_WIN )
    nn::sf::SharedPointer<nn::pl::detail::ISharedFontManager>& GetSharedFontManagerImpl() NN_NOEXCEPT
    {
        static nn::sf::SharedPointer<nn::pl::detail::ISharedFontManager> sharedFontManager
            = nn::sf::ObjectFactory<MyAllocator::Policy>::CreateSharedEmplaced<detail::ISharedFontManager, srv::SharedFontManagerServer>();
        sf::ObjectFactory<MyAllocator::Policy>::GetEmplacedImplPointer<srv::SharedFontManagerServer, detail::ISharedFontManager>(sharedFontManager)->Initialize();
        return sharedFontManager;
    }
#endif

    nn::sf::SharedPointer<nn::pl::detail::ISharedFontManager> GetSharedFontManager() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(MyAllocatorInitializer, s_Instance);
        NN_UNUSED(s_Instance);
#if defined( NN_BUILD_CONFIG_OS_WIN )
        NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<nn::pl::detail::ISharedFontManager>, sharedFontManager, = GetSharedFontManagerImpl());
        return sharedFontManager;
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::sf::SharedPointer<nn::pl::detail::ISharedFontManager> sharedFontManager;
        NN_ABORT_UNLESS_RESULT_SUCCESS((nn::sf::CreateHipcProxyByName<nn::pl::detail::ISharedFontManager, MyAllocator::Policy>(&sharedFontManager, ServiceNameForApplication)));
        return sharedFontManager;
#else
        #error "unsupported os"
#endif
    }

    const Bit8* GetSharedMemoryAddressImpl() NN_NOEXCEPT
    {
        static nn::os::SharedMemoryType s_SharedMemoryType;

        auto sharedFontManager = GetSharedFontManager();
        nn::sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSharedMemoryNativeHandle(&nativeHandle));

        // サービスから受け取った共有メモリのハンドルで共有メモリを初期化
        nn::os::AttachSharedMemory(&s_SharedMemoryType, nn::pl::detail::SharedFontMemorySize, nativeHandle.GetOsHandle(), nativeHandle.IsManaged());
        nativeHandle.Detach();
        // 共有メモリをマップ
        return reinterpret_cast<Bit8*>(nn::os::MapSharedMemory(&s_SharedMemoryType, nn::os::MemoryPermission_ReadOnly));
    }

    const Bit8* GetSharedMemoryAddress() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(const Bit8*, s_SharedMemoryBuffer, = GetSharedMemoryAddressImpl());
        return s_SharedMemoryBuffer;
    }

} // anonymous namespace

void RequestSharedFontLoad(SharedFontType sharedFontType) NN_NOEXCEPT
{
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->RequestLoad(static_cast<int32_t>(sharedFontType)));
}

SharedFontLoadState GetSharedFontLoadState(SharedFontType sharedFontType) NN_NOEXCEPT
{
    int32_t loadState;
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetLoadState(static_cast<int32_t>(sharedFontType), &loadState));
    return static_cast<SharedFontLoadState>(loadState);
}

const void* GetSharedFontAddress(SharedFontType sharedFontType) NN_NOEXCEPT
{
    const Bit8* baseAddr = GetSharedMemoryAddress();

    int32_t offset;
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSharedMemoryAddressOffset(static_cast<int32_t>(sharedFontType), &offset));
    return baseAddr + offset;
}

size_t GetSharedFontSize(SharedFontType sharedFontType) NN_NOEXCEPT
{
    int32_t size;
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSize(static_cast<int32_t>(sharedFontType), &size));
    return static_cast<size_t>(size);
}

bool GetSharedFontInOrderOfPriority(int* pOutFontCount, SharedFontType outTypes[], const void* outAddresses[], size_t outSizes[], int outCountMax, settings::LanguageCode languageCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFontCount);
    NN_SDK_REQUIRES_NOT_NULL(outTypes);
    NN_SDK_REQUIRES_NOT_NULL(outAddresses);
    NN_SDK_REQUIRES_NOT_NULL(outSizes);

    bool allFontLoaded;
    int outFontCount;
    int32_t types[pl::SharedFontType_Max];
    int32_t offsets[pl::SharedFontType_Max];
    int32_t sizes[pl::SharedFontType_Max];
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSharedFontInOrderOfPriority(&allFontLoaded, &outFontCount,
        sf::OutArray<int32_t>(types, pl::SharedFontType_Max), sf::OutArray<int32_t>(offsets, pl::SharedFontType_Max), sf::OutArray<int32_t>(sizes, pl::SharedFontType_Max), languageCode));

    if( !allFontLoaded )
    {
        return false;
    }
    *pOutFontCount = outFontCount;

    const Bit8* baseAddr = GetSharedMemoryAddress();

    for( int i = 0; i < std::min(outCountMax, *pOutFontCount); i++ )
    {
        outTypes[i] = static_cast<SharedFontType>(types[i]);
        outSizes[i] = sizes[i];
        outAddresses[i] = baseAddr + offsets[i];
    }

    return true;
}

bool GetSharedFontInOrderOfPriorityForSystem(int* pOutFontCount, SharedFontTypeForSystem outTypes[], const void* outAddresses[], size_t outSizes[], int outCountMax, settings::LanguageCode languageCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFontCount);
    NN_SDK_REQUIRES_NOT_NULL(outTypes);
    NN_SDK_REQUIRES_NOT_NULL(outAddresses);
    NN_SDK_REQUIRES_NOT_NULL(outSizes);

    bool allFontLoaded;
    int outFontCount;
    int32_t types[pl::SharedFontTypeForSystem_Max];
    int32_t offsets[pl::SharedFontTypeForSystem_Max];
    int32_t sizes[pl::SharedFontTypeForSystem_Max];
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSharedFontInOrderOfPriorityForSystem(&allFontLoaded, &outFontCount,
        sf::OutArray<int32_t>(types, pl::SharedFontTypeForSystem_Max), sf::OutArray<int32_t>(offsets, pl::SharedFontTypeForSystem_Max), sf::OutArray<int32_t>(sizes, pl::SharedFontTypeForSystem_Max), languageCode));

    if( !allFontLoaded )
    {
        return false;
    }
    *pOutFontCount = outFontCount;

    const Bit8* baseAddr = GetSharedMemoryAddress();

    for( int i = 0; i < std::min(outCountMax, *pOutFontCount); i++ )
    {
        outTypes[i] = static_cast<pl::SharedFontTypeForSystem>(types[i]);
        outSizes[i] = sizes[i];
        outAddresses[i] = baseAddr + offsets[i];
    }

    return true;
}

bool GetSharedFont(int* pOutFontCount, SharedFontData outData[], int outCountMax, settings::LanguageCode languageCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFontCount);
    NN_SDK_REQUIRES_NOT_NULL(outData);

    bool allFontLoaded;
    int outFontCount;
    int32_t types[pl::SharedFontType_Max];
    int32_t offsets[pl::SharedFontType_Max];
    int32_t sizes[pl::SharedFontType_Max];
    auto sharedFontManager = GetSharedFontManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(sharedFontManager->GetSharedFontInOrderOfPriority(&allFontLoaded, &outFontCount,
        sf::OutArray<int32_t>(types, pl::SharedFontType_Max), sf::OutArray<int32_t>(offsets, pl::SharedFontType_Max), sf::OutArray<int32_t>(sizes, pl::SharedFontType_Max), languageCode));

    if( !allFontLoaded )
    {
        return false;
    }
    *pOutFontCount = outFontCount;

    const Bit8* baseAddr = GetSharedMemoryAddress();
    int outCountActual = std::min(outCountMax, *pOutFontCount);
    for( int i = 0; i < outCountActual; i++ )
    {
        outData[i].type = static_cast<pl::SharedFontType>(types[i]);
        outData[i].size = sizes[i];
        outData[i].address = baseAddr + offsets[i];
    }

    return true;
}

}} // nn::pl
