﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/perflog/OSDependencies.h>
#include <nn/nn_SdkAssert.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>




#if defined(PERFLOG_PLATFORM_HORIZON)
#ifdef WIN32
#include <windows.h> // // horizon for windows needs DuplicateHandle
#else
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h> // for register/unregister OS handle
#endif
#endif


namespace nn { namespace perflog {



#if defined(PERFLOG_PLATFORM_WINDOWS)

bool OSCreateThread(OSThread* pNewThreadId, uint32_t(*entryPoint)(void*), void *param)
{
    OSThread t;
    t = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)entryPoint, param, 0, NULL);
    (*pNewThreadId) = t;
    return (t != NULL);
}

bool OSCreateEvent(OSEvent* pNewEventId)
{
    OSEvent e = CreateEvent(NULL, FALSE, FALSE, NULL);
    (*pNewEventId) = e;
    return (e != NULL);
}

bool OSJoinThread(OSThread* pThr, uint32_t timeoutMs, bool *pbTimedOut)
{
    return OSWaitForEvent((OSEvent *)pThr, timeoutMs, pbTimedOut);
}

bool OSCloseThread(OSThread* pThr)
{
    return (CloseHandle(*pThr) != 0);
}

bool OSWaitForEvent(OSEvent* pEvent, uint32_t timeoutMs, bool *pbTimedOut)
{
    DWORD wr;
    wr = WaitForSingleObject(*pEvent, timeoutMs);
    (*pbTimedOut) = (wr == WAIT_TIMEOUT);
    return (wr == WAIT_OBJECT_0);
}

bool OSSignalEvent(OSEvent *pEvent)
{
    return (SetEvent(*pEvent) != 0);
}

bool OSCloseEvent(OSEvent* pEvent)
{
    return (CloseHandle(*pEvent) != 0);
}



#elif defined(PERFLOG_PLATFORM_HORIZON)


const size_t STACK_SIZE  = ( 128 * 1024 );

struct OSThreadInfo
{
    void *param;
    uint32_t(*entryPoint)(void*);
    char stack[ STACK_SIZE + ( nn::os::StackRegionAlignment * 2 ) ];
};

void HorizonThreadStub( void *pvThrInfo )
{
    OSThreadInfo *pThreadInfo = static_cast<OSThreadInfo *>(pvThrInfo);

    // run what we meant to run
    pThreadInfo->entryPoint( pThreadInfo->param );

    // can't delete the stack buffer yet, as we need the address to return to, which is in the stack :)
}


bool OSCreateThread(OSThread* pNewThread, uint32_t(*entryPoint)(void*), void *param)
{
    void *pAlignedStack;
    nn::Result r;
    OSThreadInfo *pThrInfo = new OSThreadInfo;
    pThrInfo->param = param;
    pThrInfo->entryPoint = entryPoint;
    pAlignedStack = reinterpret_cast<void*>(((reinterpret_cast<uintptr_t>(pThrInfo->stack) + (nn::os::StackRegionAlignment - 1)) &
                                      ~(nn::os::StackRegionAlignment - 1)));
    r = CreateThread(pNewThread, HorizonThreadStub, pThrInfo, pAlignedStack, STACK_SIZE, nn::os::DefaultThreadPriority);
    if(  !r.IsSuccess() )
    {
        delete pThrInfo;
        return false;
    }
    else
    {
        nn::os::StartThread( pNewThread );
        return true;
    }
}


bool OSCreateEvent(OSEvent* pNewEvent)
{
    nn::Result r;
    r = nn::os::CreateSystemEvent(pNewEvent, nn::os::EventClearMode_AutoClear , true );
    return r.IsSuccess();
}


bool OSJoinThread(OSThread* pThr, uint32_t timeoutMs, bool *pbTimedOut)
{
    NN_UNUSED(timeoutMs);
    *pbTimedOut = false; // no way to tell in Horizon
    // no WaitThread with timeout
    nn::os::WaitThread( pThr ); // no way to tell success
    return true;
}


bool OSCloseThread(OSThread* pThr)
{
    nn::Result r;
    nn::os::DestroyThread( pThr ); // no way to tell success
    return true;
}


bool OSWaitForEvent(OSEvent* pEvent, uint32_t timeoutMs, bool *pbTimedOut)
{
    nn::Result r;
    *pbTimedOut = false; // no way to tell in Horizon
    return nn::os::TimedWaitSystemEvent( pEvent,  nn::TimeSpan::FromMilliSeconds(timeoutMs) );
}


bool OSSignalEvent(OSEvent *pEvent)
{
    nn::os::SignalSystemEvent(pEvent);
    return true; // no way to tell success
}

bool OSCloseEvent(OSEvent* pEvent)
{
    nn::os::DestroySystemEvent(pEvent);
    return true; // no way to tell success
}






/// UNIX todo



#endif


}} // namespace nn { namespace perflog {
