﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_Utility.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/time.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/pdm/pdm_Result.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_QueryServiceHelper.h>
#include <nn/pdm/detail/pdm_Time.h>

namespace nn { namespace pdm { namespace detail {

namespace {

// 宣言有効時はデバッグ用のログを出力します。
//#define DEBUG_LOG_ENABLE
#if defined(DEBUG_LOG_ENABLE) && (defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP))
#define DEBUG_LOG(...) NN_DETAIL_PDM_TRACE(__VA_ARGS__)

// 宣言有効時は Query() 中のイベントループ詳細を出力します。レコード数に依存したログ出力のため負荷が高いです。
//#define TRACE_QUERY_DETAIL
#if defined(TRACE_QUERY_DETAIL)
#define DEBUG_LOG_QUERY_DETAIL(...) NN_DETAIL_PDM_TRACE(__VA_ARGS__)
#else
#define DEBUG_LOG_QUERY_DETAIL(...) static_cast<void>(0)
#endif
#define CONDITION_WARN_LOG(_condition_, ...) if (_condition_) NN_DETAIL_PDM_TRACE(__VA_ARGS__)

const char* GetTypeString(const UserAccountEventType& type) NN_NOEXCEPT
{
    static const char* s_Types[] = {"Open", "Close", "NsaOpen", "NsaClose"};
    const auto typeIndex = static_cast<int>(type);
    return s_Types[typeIndex];
}
const char* GetTypeString(const AppletEventType& type) NN_NOEXCEPT
{
    static const char* s_Types[] = {"Launch", "Exit", "InFocus", "OutOfFocus", "Background", "Terminate", "AbnormalExit"};
    const auto typeIndex = static_cast<int>(type);
    return s_Types[typeIndex];
}
const char* GetTypeString(const PowerStateChangeEventType& type) NN_NOEXCEPT
{
    static const char* s_Types[] = {"On", "Off", "Sleep", "Awake", "BackgroundServicesAwake", "Terminate"};
    const auto typeIndex = static_cast<int>(type);
    return s_Types[typeIndex];
}
void TraceEventUserAccount(const char* prefix, const PlayEvent& source) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] %s -> UserAccount[ %s ].\n", prefix, GetTypeString(source.userAccountEventData.eventType));
}
void TraceEventApplet(const char* prefix, const PlayEvent& source) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] %s -> Applet[ %s, 0x%016llx(%d) ].\n", prefix, GetTypeString(source.appletEventData.eventType), GetProgramId(source).value, source.appletEventData.appletId);
}
void TraceEventPowerChange(const char* prefix, const PlayEvent& source) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] %s -> PowerChange[ %s ].\n", prefix, GetTypeString(source.powerStateChangeEventData.eventType));
}
#else
#define CONDITION_WARN_LOG(_condition_, ...) static_cast<void>(0)
#define DEBUG_LOG(...) static_cast<void>(0)
#define DEBUG_LOG_QUERY_DETAIL(...) static_cast<void>(0)
#define GetTypeString(...) static_cast<void>(0)
#define TraceEventUserAccount(...) static_cast<void>(0)
#define TraceEventApplet(...) static_cast<void>(0)
#define TraceEventPowerChange(...) static_cast<void>(0)
#endif

struct MountConfiguration
{
    constexpr static const char* const MountVolumeBase = "pdmUA";           //!< マウントボリューム名ベース。
    static const uint64_t SaveDataOwnerId = 0x010000000000001F;             //!< セーブデータの所有者は ns のままにする。
    static const uint16_t EventHeaderSize = 1024;                           //!< PlayEvent.dat のヘッダ。
    static const uint32_t SaveDataSize = 4 * 1024 * 1024;                   //!< 割り当てセーブデータ総容量( byte )。
    static const uint32_t SaveDataJournalSize = SystemSaveDataJournalSize;  //!< セーブデータジャーナルサイズ( byte )
    static const fs::SystemSaveDataId SaveDataId = detail::SystemSaveDataId;//!< セーブデータID
    static const fs::SaveDataFlags SaveDataFlags = fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData; //!< セーブデータフラグ
    //! PlayEvent.dat ファイルイベント格納上限要素数
    static const uint32_t EventEntryCountMax = (SaveDataSize - EventHeaderSize - (32 * 1024)) / sizeof(AccountPlayEvent);
};

bool IncludesForApplicationPlayStatistics(const AccountPlayEvent::Applet& applet) NN_NOEXCEPT
{
    return (applet.appletId == applet::AppletId::AppletId_Application) ||
        (IsLibraryApplet(static_cast<applet::AppletId>(applet.appletId)) && applet.laInfo.version >= 1 && applet.laInfo.libraryAppletMode == applet::LibraryAppletMode::LibraryAppletMode_AllForeground);
}

} // ~nn::pdm::detail::<anonymous>

//!---------------------------------------------------------------------------
Result AvailableCondition::Suspend(bool ignoreNestCondition) NN_NOEXCEPT
{
    if (ignoreNestCondition)
    {
        // 要求ネスト状態を無視して強制サスペンド。
        m_EnsureCount = InitialReferenceCount;
        NN_RESULT_SUCCESS;
    }
    // 参照カウンタチェック。
    const auto ensureCount = m_EnsureCount + 1;
    if (1 == ensureCount)
    {
        // このスコープの時点でサスペンド状態への移行は確定。
        m_EnsureCount = 1;
    }
    else if (1 < ensureCount)
    {
        m_EnsureCount = ensureCount;
        NN_RESULT_THROW(ResultNotExecutedByWhileNesting()); // 入れ子状態による成功。
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AvailableCondition::Resume() NN_NOEXCEPT
{
    // 参照カウンタチェック。
    const auto ensureCount = m_EnsureCount - 1;
    if (0 == ensureCount)
    {
        m_EnsureCount = 0;
    }
    else if (0 < ensureCount)
    {
        m_EnsureCount = ensureCount;
        NN_RESULT_THROW(ResultNotExecutedByWhileNesting()); // 入れ子状態による成功。
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
AccountSaveMountHandle::AccountSaveMountHandle() NN_NOEXCEPT
    : m_Uid(account::InvalidUid)
    , m_IsMounted(false)
{
    m_VolumeName[0] = '\0';
}

//!---------------------------------------------------------------------------
void AccountSaveMountHandle::Initialize(uint32_t mountIndex) NN_NOEXCEPT
{
    m_IsMounted = false;
    m_Uid = account::InvalidUid;

    const auto nameCapcity = NN_ARRAY_SIZE(m_VolumeName);
    char indexAscii[2] = {static_cast<char>('0' + mountIndex), '\0'};
    auto nameTail = util::Strlcpy(m_VolumeName, MountConfiguration::MountVolumeBase, nameCapcity);
    util::Strlcpy(&m_VolumeName[nameTail], indexAscii, nameCapcity - nameTail);
}

//!---------------------------------------------------------------------------
Result AccountSaveMountHandle::Activate(const account::Uid& userId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(userId, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(!m_Uid, ResultAlreadyDatabaseOpened());

    m_Uid = userId;
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void AccountSaveMountHandle::Deactivate(bool withDeleteSaveData) NN_NOEXCEPT
{
    const auto uid = m_Uid;
    m_Uid = account::InvalidUid;
    m_IsMounted = false;

    if (false == uid)
    {
        // 未使用中だった。
        return;
    }
    if (false == withDeleteSaveData)
    {
        // セーブデータ削除を伴わない管理情報の破棄指定。
        return;
    }

    // セーブデータ削除試行。
    const fs::UserId userId = fs::ConvertAccountUidToFsUserId(uid);
    const auto result = fs::MountSystemSaveData(m_VolumeName, MountConfiguration::SaveDataId, userId);
    if (false == fs::ResultTargetNotFound::Includes(result))
    {
        fs::Unmount(m_VolumeName);

        //! セーブデータ削除実施。
        if (fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, MountConfiguration::SaveDataId, userId).IsSuccess())
        {
            NN_DETAIL_PDM_TRACE("[pdm] Success the system save data was deleted(%s).\n", m_VolumeName);
        }
        else
        {
            NN_DETAIL_PDM_ERROR("[pdm] Failure the system save data deletion(%s).\n", m_VolumeName);
        }
    }
    else
    {
        NN_DETAIL_PDM_TRACE("[pdm] Could not find the system save data(%s) for must be deleting by pdm process, That is expected safety actions.\n", m_VolumeName);
    }
}

//!---------------------------------------------------------------------------
Result AccountSaveMountHandle::VerifyUserExistence(const account::Uid& uid) const NN_NOEXCEPT
{
    auto isUserExist = false;
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());         // InvalidUid は Idle中と同意.
    nn::account::GetUserExistence(&isUserExist, uid);           // InvalidUid はABORT, 以外はSUCCESS.
    NN_RESULT_THROW_UNLESS(isUserExist, ResultUserNotExist());
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountSaveMountHandle::Mount(bool isAlreadySaveDataMountOnly) NN_NOEXCEPT
{
    // マウント状態チェック。
    if (false == m_IsMounted)
    {
        // 指定UIDのユーザーが存在するかチェック。
        const auto& uid = m_Uid;
        NN_RESULT_DO(VerifyUserExistence(uid));

        // マウント実施。
        const fs::UserId userId = fs::ConvertAccountUidToFsUserId(uid);
        NN_RESULT_TRY(fs::MountSystemSaveData(m_VolumeName, MountConfiguration::SaveDataId, userId))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                if (isAlreadySaveDataMountOnly)
                {
                    NN_RESULT_THROW(ResultDatabaseStorageFailure());
                }
                NN_RESULT_DO(fs::CreateSystemSaveData(
                    MountConfiguration::SaveDataId,
                    userId,
                    MountConfiguration::SaveDataOwnerId,
                    MountConfiguration::SaveDataSize,
                    MountConfiguration::SaveDataJournalSize,
                    MountConfiguration::SaveDataFlags
                ));
                NN_RESULT_DO(fs::MountSystemSaveData(
                    m_VolumeName,
                    MountConfiguration::SaveDataId,
                    userId
                ));
            }
            NN_RESULT_CATCH(fs::ResultTargetLocked)
            {
                // 対象がロックされてるのは異常。
                NN_RESULT_THROW(ResultDatabaseStorageLocked());
            }
            NN_RESULT_CATCH(fs::ResultMountNameAlreadyExists)
            {
                // 既にマウント済ならスルー
            }
        NN_RESULT_END_TRY;  // result failure & 未キャッチなら、ここでreturn.
        m_IsMounted = true;
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountSaveMountHandle::Unmount() NN_NOEXCEPT
{
    bool isMounted = m_IsMounted;

    // 指定UIDのユーザーが存在するかチェック。
    const auto& uid = m_Uid;
    NN_RESULT_TRY(VerifyUserExistence(uid))
        NN_RESULT_CATCH(ResultInvalidUserId)
        {
            m_IsMounted = false;
            NN_RESULT_SUCCESS;  // 無効なUIDならマウントされてないので成功。
        }
        NN_RESULT_CATCH(ResultUserNotExist)
        {
            // m_Uid は有効だが、アカウントシステムにユーザーが見つからない場合は強制アンマウント。
            isMounted = true;
        }
    NN_RESULT_END_TRY;

    // マウント状態チェック。
    if (isMounted)
    {
        // このスコープの時点でアンマウント状態への移行は確定。
        m_IsMounted = false;

        // セーブデータ有無・マウント中確認。
        const fs::UserId userId = fs::ConvertAccountUidToFsUserId(uid);
        NN_RESULT_TRY(fs::MountSystemSaveData(m_VolumeName, MountConfiguration::SaveDataId, userId))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                // なければマウントしてないので成功。
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                // その他エラーはマウントされている想定でアンマウント実施。
            }
        NN_RESULT_END_TRY;
        fs::Unmount(m_VolumeName);
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountSaveMountHandle::Commit() NN_NOEXCEPT
{
    return fs::CommitSaveData(m_VolumeName);
}






//!---------------------------------------------------------------------------
AccountPlayEventBuffer::AccountPlayEventBuffer() NN_NOEXCEPT
    : EventEntryFileStream(MountConfiguration::EventHeaderSize, sizeof(AccountPlayEvent))
    , m_ScopedReadLock(this)
    , m_EventCountMax(MountConfiguration::EventEntryCountMax)
    , m_DetectedPowerChange(PowerChange::None)
{
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::OnHeaderRead(fs::FileHandle fileHandle, int64_t position, size_t* pOutReadSize) NN_NOEXCEPT
{
    NN_RESULT_DO(fs::ReadFile(fileHandle, position, &m_CustomHeader, sizeof(CustomHeader)));
    if (nullptr != pOutReadSize)
    {
        *pOutReadSize = sizeof(CustomHeader);
    }
    size_t size = 0;
    position += sizeof(CustomHeader);
    NN_RESULT_DO(EventEntryFileStream::OnHeaderRead(fileHandle, position, &size));
    if (nullptr != pOutReadSize)
    {
        *pOutReadSize = sizeof(CustomHeader) + size;
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::OnHeaderWrite(fs::FileHandle fileHandle, int64_t position, const fs::WriteOption inheritedOption, size_t* pOutWriteSize) NN_NOEXCEPT
{
    NN_RESULT_DO(fs::WriteFile(fileHandle, position, &m_CustomHeader, sizeof(CustomHeader), inheritedOption));
    if (nullptr != pOutWriteSize)
    {
        *pOutWriteSize = sizeof(CustomHeader);
    }
    size_t size = 0;
    position += sizeof(CustomHeader);
    NN_RESULT_DO(EventEntryFileStream::OnHeaderWrite(fileHandle, position, inheritedOption, &size));
    if (nullptr != pOutWriteSize)
    {
        *pOutWriteSize = sizeof(CustomHeader) + size;
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::OnHeaderReset() NN_NOEXCEPT
{
    EventEntryFileStream::OnHeaderReset();
    m_CustomHeader.Reset();
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::Initialize(uint32_t mountIndex) NN_NOEXCEPT
{
    m_ServiceCondition.Initialize();
    m_MountHandle.Initialize(mountIndex);
    PrepareDatabaseFilePath(m_MountHandle.GetVolumeName());
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Activate(const account::Uid& uid, const MigrationState migrationState, bool withOpenStream) NN_NOEXCEPT
{
    m_ClientIdleSync.Wait();    // 全処理同期待ち
    auto fullLock = AcquireLockFull();
    m_CustomHeader.migration = migrationState;  // マイグレーション管理初期値

    NN_RESULT_TRY(m_MountHandle.Activate(uid))
        NN_RESULT_CATCH(ResultAlreadyDatabaseOpened)
        {
            DeactivateUnsafe();
            m_MountHandle.Activate(uid);    // 以前のUIDをDeactivate後、改めて指定のuidを設定します。
        }
    NN_RESULT_END_TRY;  // ResultInvalidUserId の場合は、ここで返却。

    // キャッシュバッファクリーン。
    m_EntryCache.Clear();
    m_DetectedPowerChange = PowerChange::None;

    // サービス開始。
    m_ServiceCondition.Resume();

    // ストリームオープンを伴った活性化要求か？
    if (!withOpenStream)
    {
        NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Activate(%s): Successful without mount the system save data.\n", m_MountHandle.GetVolumeName());
        NN_RESULT_SUCCESS;
    }

    // 新規作成付きストリームオープン.
    const auto result = PrepareStreamUnsafe(false);
    if (result.IsFailure())
    {
        m_MountHandle.Deactivate(false); // 失敗の場合はセーブデータを消さずに管理情報を破棄する。
        NN_DETAIL_PDM_ERROR("[pdm] AccountPlayEventBuffer::Activate(%s): Unexpected error... [0x%lx].\n", m_MountHandle.GetVolumeName(), result.GetInnerValueForDebug());
    }
    else
    {
        NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Activate(%s): Successful. Header[ start = %d, count = %d, migration = 0x%x ].\n",
            m_MountHandle.GetVolumeName(), GetHeaderUnsafe().start, GetHeaderUnsafe().count, m_CustomHeader.migration);
    }
    return result;
}

//!---------------------------------------------------------------------------
//! isAlreadyExistDataOnly == false :   セーブデータ＆データベースファイルが見つからない場合、生成します。
//! isAlreadyExistDataOnly == true  :   セーブデータが見つからない場合はマウントされずに返します。
//!                                     データベースファイルが見つからない場合はストリームのオープンに失敗します。
Result AccountPlayEventBuffer::PrepareStreamUnsafe(bool isAlreadyExistDataOnly) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsUnavailableUnsafe(), ResultServiceNotAvailable());    // Suspend or Idle なら拒否。
    NN_RESULT_DO(m_MountHandle.Mount(isAlreadyExistDataOnly));                      // マウント状態チェック＆要求。

    // データベースファイルオープン
    const auto openResult = OpenUnsafe(m_EventCountMax, isAlreadyExistDataOnly);
    if (OpenResult::Error == openResult)
    {
        // 既存ファイルが見つからない。( isAlreadyExistDataOnly( true )時のみ )
        NN_RESULT_THROW(ResultDatabaseStorageFailure());
    }
    else if (OpenResult::AlreadyOpened == openResult)
    {
        // 既にオープン済
        DEBUG_LOG("[pdm] AccountPlayEventBuffer::PrepareStreamUnsafe(%s, %s): Detect re-open request, this buffer was already opened.\n",
            m_MountHandle.GetVolumeName(), (isAlreadyExistDataOnly) ? "true" : "false"
        );
    }
    else if (OpenResult::NewlyCreated == openResult)
    {
        // 作成したファイルへの初回コミット( m_CustomHeader::migration 保証用 )
        WriteHeaderOnlyUnsafe();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
        DEBUG_LOG("[pdm] AccountPlayEventBuffer::PrepareStreamUnsafe(%s): Detected that a database file was newly created.\n", m_MountHandle.GetVolumeName());
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::Deactivate() NN_NOEXCEPT
{
    m_ClientIdleSync.Wait();    // 全処理同期待ち
    auto fullLock = AcquireLockFull();
    DeactivateUnsafe();
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::DeactivateUnsafe() NN_NOEXCEPT
{
    const auto result = SuspendImpl(true);
    if (result.IsSuccess())
    {
        Close();    // EventEntryFileStream
        m_EntryCache.Clear();
        m_MountHandle.Deactivate(true);
        NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::DeactivateUnsafe(%s): successful.\n", m_MountHandle.GetVolumeName());
    }
    else
    {
        // SuspendImpl(true) の場合は来ない想定だが、ロジック漏れなどの検出用にログだけ出力。
        NN_DETAIL_PDM_ERROR("[pdm] AccountPlayEventBuffer::DeactivateUnsafe(%s): Unexpected error... [0x%lx].\n", m_MountHandle.GetVolumeName(), result.GetInnerValueForDebug());
    }
    return result;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Discard() NN_NOEXCEPT
{
    auto fullLock = AcquireLockFull();
    if (IsUnavailableUnsafe()) // Suspend or Idle
    {
        NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventBuffer::Discard(%s): Failed discard because now condition with suspend or idle.\n", m_MountHandle.GetVolumeName());
        NN_RESULT_THROW(ResultServiceNotAvailable());
    }
    const char* message = "[pdm] AccountPlayEventBuffer::Discard(%s): Successful, without commit!.\n";
    if (DiscardUnsafe())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
        message = "[pdm] AccountPlayEventBuffer::Discard(%s): Successful, with commit.\n";
    }
    NN_DETAIL_PDM_TRACE(message, m_MountHandle.GetVolumeName());
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::DiscardUnsafe() NN_NOEXCEPT
{
    const auto result = PrepareStreamUnsafe(true);
    if (result.IsSuccess())
    {
        m_EntryCache.Clear();
        EventEntryFileStream::Discard();
        return true;
    }
    if (ResultServiceNotAvailable::Includes(result))
    {
        // サービス停止中は何もしない。
        return false;
    }
    // セーブデータ / データベースファイルが見つからないので、ファイル削除 / Commit なし。
    m_EntryCache.Clear();
    OnHeaderReset();
    return false;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Resume() NN_NOEXCEPT
{
    auto fullLock = AcquireLockFull();
    NN_RESULT_TRY(m_ServiceCondition.Resume())
        NN_RESULT_CATCH(ResultNotExecutedByWhileNesting)
        {
            NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Resume(%s): Successful with nesting.\n", m_MountHandle.GetVolumeName());
            NN_RESULT_SUCCESS;  // SUCCESS扱い
        }
    NN_RESULT_END_TRY;
    NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Resume(%s): Successful.\n", m_MountHandle.GetVolumeName());
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Suspend() NN_NOEXCEPT
{
    m_ClientIdleSync.Wait();    // 全処理同期待ち
    auto fullLock = AcquireLockFull();
    NN_RESULT_TRY(SuspendImpl(false))
        NN_RESULT_CATCH(ResultNotExecutedByWhileNesting)
        {
            NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Suspend(%s): Successful with nesting.\n", m_MountHandle.GetVolumeName());
            NN_RESULT_SUCCESS;  // SUCCESS扱い
        }
    NN_RESULT_END_TRY;
    NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Suspend(%s): Successful.\n", m_MountHandle.GetVolumeName());
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::SuspendImpl(bool ignoreNestCondition) NN_NOEXCEPT
{
    // ignoreNestCondition == true 時は、Deactivate 前提。
    if (!ignoreNestCondition)
    {
        FlushImpl();    // 累積コミット( Mount / セーブデータ作成が発生する可能性がある )
    }
    NN_RESULT_DO(m_ServiceCondition.Suspend(ignoreNestCondition));  // 入れ子なら返却.
    NN_RESULT_DO(m_MountHandle.Unmount());                          // 入れ子じゃなければマウント解除要求
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::Add(AccountPlayEvent& eventEntry, const account::Uid& checkAvailability) NN_NOEXCEPT
{
    // クライアントBUSY獲得
    auto managedUnlock = m_ClientIdleSync.Acquire();

    // Uid の変更は Activate / Deactivate 時想定で同スコープで最も長くロックされるのが added mutex.
    auto addedLock = m_EntryCache.AcquireLockForAdded();
    if (checkAvailability && checkAvailability != m_MountHandle.GetUid())
    {
        // Idle(deactivate) 時の拒否兼ねる.
        NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventBuffer::Add(%s): Unexpected request failure, there was different with the requested user account.\n", m_MountHandle.GetVolumeName());
        return false;
    }
    eventEntry.powerChange = GetDetectedPowerChange();
    m_EntryCache.Add(eventEntry, [&]() -> void
    {
        FlushImpl();
    });
    return true;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::Flush(const uint32_t excessRate) NN_NOEXCEPT
{
    // クライアントBUSY獲得
    auto managedUnlock = m_ClientIdleSync.Acquire();
    FlushImpl(excessRate);
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::FlushImpl(const uint32_t excessRate) NN_NOEXCEPT
{
    const auto excessCount = GetBufferSpaceCountMax() * ((excessRate > 100) ? 100 : excessRate) / 100;
    auto flushLock = m_EntryCache.SwapBuffer(excessCount);
    if (false == flushLock.owns_lock())
    {
        if (0 == excessRate)
        {
            // フルコミット指定時のイベントなしは報告.
            DEBUG_LOG("[pdm] AccountPlayEventBuffer::Flush(%s): There is no event in buffer to flush.\n", m_MountHandle.GetVolumeName());
        }
        // Flush条件を満たさない(イベントがない or 要求超過率を超過していない)
        return;
    }
    auto pFlushBuffer = m_EntryCache.GetFlushBuffer();
    {
        auto fileLock = AcquireLockForFile();                       // file stream lock.
        const auto resultActivation = PrepareStreamUnsafe(false);   // 新規作成付きストリームオープン.
        if (resultActivation.IsFailure())
        {
            // この時点で管理ユーザーがアカウントシステムに存在しない場合があり、ResultUserNotExist が返される事がある。
            // これは、account::DeleteUser() → Deactivate() までの間に以下処理が発生した場合。
            //  ※NXシステム運用では基本的に発生しない認識。( 発生するケースは機能検証テストなどで負荷をかけた場合など。 )
            //
            //  - 他プロセスからの Query 呼び出し。
            //  - SaveDataCommitThreadからの Flush 呼び出し。
            //
            // 尚、この経路通過自体は、意図しない書き込み/生成を排除できた事になるので問題ではない。
            NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventBuffer::Flush(%s): Stream activation failed => result[ 0x%lx ].\n", m_MountHandle.GetVolumeName(), resultActivation.GetInnerValueForDebug());
            return;
        }
        WriteUnsafe(pFlushBuffer->playEvent, pFlushBuffer->filledCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
        NN_DETAIL_PDM_TRACE("[pdm] AccountPlayEventBuffer::Flush(%s): Completed. [ start = %d, count = %d ].\n", m_MountHandle.GetVolumeName(), GetHeaderUnsafe().start, GetHeaderUnsafe().count);
    }
}

//!---------------------------------------------------------------------------
uint32_t AccountPlayEventBuffer::GetFilledBufferSpaceCount() const NN_NOEXCEPT
{
    return m_EntryCache.GetCachedEntryCount();
}

//!---------------------------------------------------------------------------
int AccountPlayEventBuffer::Find(const account::Uid* pUsers, int count) const NN_NOEXCEPT
{
    // Uid の変更は Activate / Deactivate 時想定で同スコープで最も長くロックされるのが added mutex.
    auto addedLock = m_EntryCache.AcquireLockForAdded();
    const auto& uid = m_MountHandle.GetUid();
    if (!uid)
    {
        return -1;
    }
    for (int i = 0; i < count; ++i)
    {
        if (uid == pUsers[i])
        {
            return i;
        }
    }
    return count;
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::IsIdle() const NN_NOEXCEPT
{
    // Uid の変更は Activate / Deactivate 時想定で同スコープで最も長くロックされるのが added mutex.
    auto addedLock = m_EntryCache.AcquireLockForAdded();
    return IsIdleUnsafe();
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::IsSuspended() const NN_NOEXCEPT
{
    // Uid の変更は Activate / Deactivate 時想定で同スコープで最も長くロックされるのが added mutex.
    auto addedLock = m_EntryCache.AcquireLockForAdded();
    return IsSuspendedUnsafe();
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::IsUnavailable() const NN_NOEXCEPT
{
    // Uid の変更は Activate / Deactivate 時想定で同スコープで最も長くロックされるのが added mutex.
    auto addedLock = m_EntryCache.AcquireLockForAdded();
    return IsUnavailableUnsafe();
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::GetReportData(ReportData* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    auto fullLock = AcquireLockFull();
    NN_RESULT_THROW_UNLESS(!IsUnavailableUnsafe(), ResultServiceNotAvailable());
    pOut->uid = m_MountHandle.GetUid();
    pOut->count = GetCountUnsafe();
    pOut->startIndex = GetStartIndexUnsafe();
    pOut->lastIndex = GetLastIndexUnsafe();
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::AcquireRead(ScopedReadLock* pOutScopedLock) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutScopedLock);
    NN_RESULT_THROW_UNLESS(!IsUnavailable(), ResultServiceNotAvailable());
    *pOutScopedLock = ScopedReadLock(m_ScopedReadLock);
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::BeginRead() NN_NOEXCEPT
{
    Flush();
    m_ClientIdleSync.Lock();    // クライアントBUSY獲得

    auto fileLock = AcquireLockForFile();
    if (PrepareStreamUnsafe(true).IsFailure())
    {
        // セーブデータ、もしくはデータベースファイルがない場合。
        DEBUG_LOG("[pdm] AccountPlayEventBuffer::BeginRead(%s): Could not preapre the read action, condition[ uid={0x%llx-0x%llx}, suspend=%s, mounted=%s ].\n",
            m_MountHandle.GetVolumeName(), m_MountHandle.GetUid()._data[0], m_MountHandle.GetUid()._data[1],
            (IsSuspendedUnsafe()) ? "true" : "false", (m_MountHandle.IsMounted()) ? "true" : "false"
        );
        EventEntryFileStream::BeginReadWithForceInvalidation();
        return;
    }
    EventEntryFileStream::BeginRead();
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::EndRead() NN_NOEXCEPT
{
    EventEntryFileStream::EndRead();
    m_ClientIdleSync.Unlock();    // クライアントBUSY解除
}



//!---------------------------------------------------------------------------
namespace {

// 登録された有限 PlayStatistics 配列からの検索、及び見つからなかった場合の登録を行うクラス。
class PlayStatisticsEntryFinder
{
public:
    explicit PlayStatisticsEntryFinder(PlayStatistics* pEntryBuffer, const uint32_t entryCapacity) NN_NOEXCEPT
        : m_pEntryBuffer(pEntryBuffer), m_EntryCapacity(entryCapacity), m_FoundCount(0) {}

    PlayStatistics* FindEntry(const AccountPlayEvent::PackedId64& target) NN_NOEXCEPT
    {
        if (AccountPlayEvent::PackedId64::InvalidValue == target)
        {
            return nullptr;
        }

        uint32_t index = 0;
        if (FindBy(target.ToProgramId(), &index))
        {
            // 発見済エントリバッファに存在するエントリが要求された
            return &m_pEntryBuffer[index];
        }

        // 新しいエントリが要求された
        uint32_t nowIndex = m_FoundCount;
        const uint32_t nextIndex = (nowIndex + 1) % m_EntryCapacity;
        if (0 == nextIndex)
        {
            // overflow.
            NN_DETAIL_PDM_WARN("[pdm] PlayStatisticsEntryFinder: The new application event with detected was overflow from given buffer capacity( %lu ).\n", m_EntryCapacity);
            nowIndex = detail::QueryServiceHelper::SelectPlayStatisticsIndexToReplace(m_pEntryBuffer, m_FoundCount);
        }
        else
        {
            m_FoundCount = nextIndex;
        }
        // PlayStatistics 初期化
        auto pEntry = &m_pEntryBuffer[nowIndex];
        detail::QueryServiceHelper::Initialize(pEntry, ncm::ApplicationId{target.ToValue64()});
        return pEntry;
    }

    const uint32_t GetFoundCount() const NN_NOEXCEPT
    {
        return m_FoundCount;
    }

private:
    bool FindBy(const ncm::ProgramId& target, uint32_t* pOutIndex) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutIndex);
        auto pEntries = m_pEntryBuffer;
        const uint32_t capacity = m_FoundCount;
        for (uint32_t i = 0; i < capacity; ++i)
        {
            auto& entry = pEntries[i];
            if (entry.applicationId == target)
            {
                *pOutIndex = i;
                return true;
            }
        }
        return false;
    }

    PlayStatistics* m_pEntryBuffer;
    uint32_t        m_EntryCapacity;
    uint32_t        m_FoundCount;
};

// InFocus イベント起因での PlayStatistics 更新
void UpdatePlayStatistics(PlayStatistics* pOutValue, const AccountPlayEvent& nowEvent, const uint32_t eventIndex, const Bit32 duration) NN_NOEXCEPT
{
    const auto userTime = nowEvent.time.user;
    const auto networkTime = nowEvent.time.network;

    // 対象プログラムID の 最後の InFocus イベントの InFocus 終了時刻 == 最後にプレイした時刻
    pOutValue->latestEventTime.eventIndex = eventIndex;
    pOutValue->latestEventTime.userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(userTime);
    pOutValue->latestEventTime.networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(networkTime);

    // 対象プログラムID の 最初の InFocus イベントの InFocus 開始時刻 == 最初にプレイした時刻
    if (0 == pOutValue->firstEventTime.userClockTime)
    {
        pOutValue->firstEventTime.eventIndex = eventIndex;
        pOutValue->firstEventTime.userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin({userTime.value - duration});
        pOutValue->firstEventTime.networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin({networkTime.value - duration});
    }
    // 対象プログラムID の InFocus 期間の総時間 == プレイ時間
    pOutValue->totalPlayTime += duration;
}

// InOpen イベント起因での AccountEvent 更新
inline void UpdateAccountEvent(AccountEvent* pOutValue, const account::Uid& uid,
    const UserAccountEventType type, uint32_t eventIndex, const AccountPlayEvent::Time& time, uint32_t duration = 0) NN_NOEXCEPT
{
    pOutValue->uid = uid;
    pOutValue->eventType = type;
    pOutValue->eventTime.eventIndex = eventIndex;
    pOutValue->eventTime.userClockTime = {time.user.value - duration};
    pOutValue->eventTime.networkClockTime = {time.network.value - duration};
    pOutValue->eventTime.steadyClockTimeValueSinceBase = 0;  // 無いので 0.
}

//! @brief  配列中の無効と判断された要素を詰めます。
//! @return 無効な要素を詰めた後の要素数を返します。全ての要素が無効であった場合は 0 を返します。
template<typename TValueType, typename TCountType, typename Predicate>
const uint32_t Compaction(TValueType* pOutValues, const TCountType availableCount, Predicate isPackingCondition)
{
    TCountType compactCount = availableCount;
    for (TCountType i = 0; i < compactCount;)
    {
        auto& entry = pOutValues[i];
        if (isPackingCondition(entry))
        {
            --compactCount;
            for (TCountType c = i; c < compactCount; ++c)
            {
                std::copy_n(&pOutValues[c + 1], 1, &pOutValues[c]);
            }
        }
        else
        {
            ++i;
        }
    }
    return compactCount;
}

}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Query(PlayStatistics* pOutValue, const ncm::ApplicationId& targetApplicationId,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pReadTemporary);
    NN_SDK_REQUIRES(byteOfTemporary >= sizeof(AccountPlayEvent));

    ScopedReadLock lock;
    NN_RESULT_DO(AcquireRead(&lock));

    DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( 0x%016llx ) -> Begin\n", targetApplicationId.value);
    const auto header = GetHeaderUnsafe();
    auto pEventBuffer = reinterpret_cast<AccountPlayEvent*>(pReadTemporary);
    const uint32_t readableCount = byteOfTemporary / sizeof(AccountPlayEvent);
    const uint32_t startIndex = header.start;
    uint32_t totalReadCount = 0;

    const AccountPlayEvent::PackedId64 targetId(targetApplicationId.value);
    QueryServiceHelper::Initialize(pOutValue, targetApplicationId);

    while (NN_STATIC_CONDITION(true))
    {
        const auto readCount = ReadImpl(pEventBuffer, startIndex + totalReadCount, readableCount);
        if (0 == readCount)
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            break;
        }
        for (uint32_t i = 0; i < readCount; ++i)
        {
            const auto& nowEvent = pEventBuffer[i];
            const uint32_t nowIndex = startIndex + totalReadCount + i;

            switch (nowEvent.category)
            {
            case AccountPlayEvent::Category::InFocus:
                {
                    const auto& inFocusContext = nowEvent.context.inFocus;
                    if (!IncludesForApplicationPlayStatistics(inFocusContext.applet))
                    {
                        continue;
                    }
                    if (targetId == inFocusContext.applet.programId)
                    {
                        UpdatePlayStatistics(pOutValue, nowEvent, nowIndex, inFocusContext.duration);

                        DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( 0x%016llx ) -> InFocus[ nowIndex=%lu, time<user>=%lld, duration=%lu ]\n",
                            targetApplicationId.value, nowIndex, nowEvent.time.user.value, pInFocusContext->duration);
                    }
                }
                break;
            case AccountPlayEvent::Category::InOpen:
                {
                    const auto& inOpenContext = nowEvent.context.inOpen;
                    if (targetId == inOpenContext.applet.programId)
                    {
                        // 対象プログラムID の InOpen 個数 == プレイ回数 (一時的なBG状態はカウント外とするため)
                        pOutValue->totalPlayCount++;

                        DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( 0x%016llx ) -> InOpen[ nowIndex=%lu, totalPlayCount=%lu, duration=%lu ]\n",
                            targetApplicationId.value, nowIndex, pOutValue->totalPlayCount, pInOpenContext->duration);
                    }
                }
                break;
            default:
                // Ignore because do not have an operation in this case.
                DEBUG_LOG("[pdm] QueryPlayStatistics( 0x%016llx ) -> %u[ nowIndex=%lu ]\n",
                    targetApplicationId.value, nowEvent.category, nowIndex);
                break;
            }
        }
        totalReadCount += readCount;
    }
    if ((0 >= pOutValue->firstEventTime.userClockTime) && (0 >= pOutValue->totalPlayTime) && (0 < pOutValue->totalPlayCount))
    {
        // イベント溢れなどで InFocus イベントが削除され InOpen のみが残った場合、
        //「最初に遊んだ時刻」、「最後に遊んだ時刻」、「プレイ時間」が取得できず、妥当な値での補填も不可のため、遊んだ回数も 0 にして PlayStatistics を無効化する。
        DEBUG_LOG("[pdm] QueryPlayStatistics( 0x%016llx ) -> Invalidate PlayStatistics. totalPlayCount = %u, but totalPlayTime and firstEventTime.userClockTime are 0 (InFocus event is not found)\n",
            targetApplicationId.value, pOutValue->totalPlayCount);
        pOutValue->totalPlayCount = 0;
    }
    QueryServiceHelper::ConvertTotalTimeFromSecondsToMinutes(pOutValue);
    DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( 0x%016llx ) -> End\n", targetApplicationId.value);
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Query(uint32_t* pOutCount, PlayStatistics* pOutValues, uint32_t receiveCapacity,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_NOT_NULL(pReadTemporary);
    NN_SDK_REQUIRES(byteOfTemporary >= sizeof(AccountPlayEvent));
    NN_SDK_REQUIRES(receiveCapacity > 0);

    ScopedReadLock lock;
    NN_RESULT_DO(AcquireRead(&lock));

    DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( List ) -> Begin\n");
    const auto header = GetHeaderUnsafe();
    auto pEventBuffer = reinterpret_cast<AccountPlayEvent*>(pReadTemporary);
    const uint32_t readableCount = byteOfTemporary / sizeof(AccountPlayEvent);
    const uint32_t startIndex = header.start;
    uint32_t totalReadCount = 0;

    *pOutCount = 0;
    PlayStatisticsEntryFinder finder(pOutValues, receiveCapacity);

    while (NN_STATIC_CONDITION(true))
    {
        const auto readCount = ReadImpl(pEventBuffer, startIndex + totalReadCount, readableCount);
        if (0 == readCount)
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            break;
        }
        for (uint32_t i = 0; i < readCount; ++i)
        {
            const auto& nowEvent = pEventBuffer[i];
            const uint32_t nowIndex = startIndex + totalReadCount + i;

            switch (nowEvent.category)
            {
            case AccountPlayEvent::Category::InFocus:
                {
                    const auto& inFocusContext = nowEvent.context.inFocus;
                    if (!IncludesForApplicationPlayStatistics(inFocusContext.applet))
                    {
                        continue;
                    }
                    auto pOutValue = finder.FindEntry(inFocusContext.applet.programId);
                    if (nullptr != pOutValue)
                    {
                        UpdatePlayStatistics(pOutValue, nowEvent, nowIndex, inFocusContext.duration);

                        DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( List ) -> InFocus[ nowIndex=%lu, duration=%lu, targetId=0x%016llx ]\n",
                            nowIndex, inFocusContext->duration, pOutValue->applicationId.value);
                    }
                }
                break;
            case AccountPlayEvent::Category::InOpen:
                {
                    const auto& inOpenContext = nowEvent.context.inOpen;
                    auto pOutValue = finder.FindEntry(inOpenContext.applet.programId);
                    if (nullptr != pOutValue)
                    {
                        pOutValue->totalPlayCount++;    // 対象プログラムID の InOpen 個数 == プレイ回数 (一時的なBG状態はカウント外とするため)

                        DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( List ) -> InOpen[ nowIndex=%lu, totalPlayCount=%lu, targetId=0x%016llx ]\n",
                            nowIndex, pOutValue->totalPlayCount, pOutValue->applicationId.value);
                    }
                }
                break;
            default:
                // Ignore because do not have an operation in this case.
                DEBUG_LOG("[pdm] QueryPlayStatistics( List ) -> [ Case=%u, nowIndex=%lu ]\n", nowEvent.category, nowIndex);
                break;
            }
        }
        totalReadCount += readCount;
    }
    // post compaction.
    const auto foundCount = Compaction(pOutValues, finder.GetFoundCount(), [](const PlayStatistics& entry) -> bool
    {
        // InFocus記録のみしかないケース / InOpen記録のみしかないケースは無効な記録とする。
        return ((0 >= entry.totalPlayCount) || ((0 >= entry.firstEventTime.userClockTime) && (0 >= entry.totalPlayTime)));
    });
    QueryServiceHelper::ConvertTotalTimeFromSecondsToMinutes(pOutValues, foundCount);
    *pOutCount = foundCount;
    DEBUG_LOG_QUERY_DETAIL("[pdm] QueryPlayStatistics( List ) -> End[ Count=%lu ]\n", foundCount);
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::Query(uint32_t* pOutCount, AccountEvent* pOutValues, uint32_t receiveCapacity,
    uint32_t eventIndexOffset, Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_NOT_NULL(pReadTemporary);
    NN_SDK_REQUIRES(byteOfTemporary >= sizeof(AccountPlayEvent));
    NN_SDK_REQUIRES(receiveCapacity > 0);

    ScopedReadLock lock;
    NN_RESULT_DO(AcquireRead(&lock));

    const auto header = GetHeaderUnsafe();
    auto pEventBuffer = reinterpret_cast<AccountPlayEvent*>(pReadTemporary);
    const uint32_t readableCount = byteOfTemporary / sizeof(AccountPlayEvent);
    const uint32_t startIndex = std::max(eventIndexOffset, header.start);
    uint32_t totalReadCount = 0;

    *pOutCount = 0;
    uint32_t foundCount = 0;
    const auto uid = m_MountHandle.GetUid();

    while (NN_STATIC_CONDITION(true))
    {
        const auto readCount = ReadImpl(pEventBuffer, startIndex + totalReadCount, readableCount);
        if (0 == readCount)
        {
            // 最後まで読み込み、これ以上読み込むものがなくなった。
            break;
        }
        for (uint32_t i = 0; i < readCount; ++i)
        {
            const auto& nowEvent = pEventBuffer[i];
            const uint32_t nowIndex = startIndex + totalReadCount + i;

            switch (nowEvent.category)
            {
            case AccountPlayEvent::Category::InOpen:
                {
                    const Bit32 duration = nowEvent.context.inOpen.duration;
                    const auto& time = nowEvent.time;

                    // open
                    UpdateAccountEvent(&pOutValues[foundCount++], uid, UserAccountEventType::Open, nowIndex, time, duration);
                    if (foundCount >= receiveCapacity)
                    {
                        *pOutCount = foundCount;
                        NN_RESULT_SUCCESS;
                    }

                    // close
                    UpdateAccountEvent(&pOutValues[foundCount++], uid, UserAccountEventType::Close, nowIndex, time);
                    if (foundCount >= receiveCapacity)
                    {
                        *pOutCount = foundCount;
                        NN_RESULT_SUCCESS;
                    }
                }
                break;
            default:
                // Ignore because do not have an operation in this case.
                break;
            }
        }
        totalReadCount += readCount;
    }
    *pOutCount = foundCount;
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::QueryRecentlyPlayedApplication(uint32_t* pOutCount, ncm::ApplicationId pOutValues[], uint32_t receiveCapacity, uint32_t presetCount,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_NOT_NULL(pReadTemporary);
    NN_SDK_REQUIRES(byteOfTemporary >= sizeof(AccountPlayEvent));
    NN_SDK_REQUIRES(receiveCapacity > 0);

    ScopedReadLock lock;
    NN_RESULT_DO(AcquireRead(&lock));

    if( GetCountUnsafe() == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    auto pEventBuffer = reinterpret_cast<AccountPlayEvent*>(pReadTemporary);
    const uint32_t readableCount = byteOfTemporary / sizeof(AccountPlayEvent);

    *pOutCount = presetCount;

    auto bufferStartIndex = GetStartIndexUnsafe();
    auto readEndIndex = GetLastIndexUnsafe();
    auto readStartIndex = readEndIndex - std::min(readEndIndex - bufferStartIndex + 1u, readableCount) + 1u;

    while( NN_STATIC_CONDITION(true) )
    {
        const auto readCount = ReadImpl(pEventBuffer, readStartIndex, (readEndIndex + 1u - readStartIndex));
        NN_SDK_ASSERT(readCount > 0);
        for( uint32_t i = readCount - 1;/* i >= 0 */; --i )
        {
            const auto& nowEvent = pEventBuffer[i];
            ncm::ApplicationId applicationId = ncm::ApplicationId::GetInvalidId();

            switch( nowEvent.category )
            {
            case AccountPlayEvent::Category::InOpen:
                {
                    const auto& inOpenContext = nowEvent.context.inOpen;
                    if( inOpenContext.applet.appletId == applet::AppletId_Application )
                    {
                        applicationId.value = inOpenContext.applet.programId.ToValue64();
                    }
                }
                break;
            case AccountPlayEvent::Category::InFocus:
                {
                    // 強制電源断等によって InFocus の記録のみが残っている場合がありえるので InFocus もケアする。
                    const auto& inFocusContext = nowEvent.context.inFocus;
                    if( inFocusContext.applet.appletId == applet::AppletId_Application )
                    {
                        applicationId.value = inFocusContext.applet.programId.ToValue64();
                    }
                }
                break;
            case AccountPlayEvent::Category::NsaAvailable:
            case AccountPlayEvent::Category::NsaUnavailable:
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            if( applicationId != ncm::ApplicationId::GetInvalidId() )
            {
                bool alreadyFound = false;
                for( uint32_t j = 0; j < *pOutCount; j++ )
                {
                    if( pOutValues[j] == applicationId )
                    {
                        alreadyFound = true;
                        break;
                    }
                }
                if( !alreadyFound )
                {
                    pOutValues[*pOutCount] = applicationId;
                    *pOutCount += 1;
                    if( *pOutCount == receiveCapacity )
                    {
                        NN_RESULT_SUCCESS;
                    }
                }
            }

            if( i == 0u )
            {
                break;
            }
        }

        if( readStartIndex == bufferStartIndex )
        {
            // 始点まで読み込みが終わったので終了。
            NN_RESULT_SUCCESS;
        }

        readEndIndex = readStartIndex - 1;
        if( readStartIndex - bufferStartIndex > readableCount )
        {
            readStartIndex -= readableCount;
        }
        else
        {
            readStartIndex = bufferStartIndex;
        }
    }
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::QueryAvailableRange(uint32_t* pOutCount, uint32_t* pOutStartIndex, uint32_t* pOutLastIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutLastIndex);
    NN_SDK_REQUIRES_NOT_NULL(pOutStartIndex);

    auto fullLock = AcquireLockFull();
    NN_RESULT_THROW_UNLESS(!IsUnavailableUnsafe(), ResultServiceNotAvailable());
    FlushImpl();
    *pOutCount = GetCountUnsafe();
    *pOutLastIndex = GetLastIndexUnsafe();
    *pOutStartIndex = GetStartIndexUnsafe();
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventBuffer::QueryRaw(uint32_t* pOutCount, AccountPlayEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(receiveCapacity, 0u);

    ScopedReadLock lock;
    NN_RESULT_DO(AcquireRead(&lock));

    const uint32_t startIndex = GetStartIndexUnsafe() + eventIndexOffset;
    *pOutCount = ReadImpl(pOutValues, startIndex, receiveCapacity);
    NN_RESULT_SUCCESS;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::PrepareMigration(MigrationRecorder* pRecorder) NN_NOEXCEPT
{
    NN_UNUSED(pRecorder);

    // バッファ有効状態チェック。
    if (false == m_MountHandle.GetUid())
    {
        return;
    }
    // マイグレーションが完了していない場合は、既存状態を更地に。
    if (MigrationState::None == m_CustomHeader.migration && true == DiscardUnsafe())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
    }
}

//!---------------------------------------------------------------------------
bool AccountPlayEventBuffer::ImportMigration(MigrationRecorder* pRecorder, AccountPlayEvent& entry, const account::Uid& checkAvailability) NN_NOEXCEPT
{
    // @ref Add() と違い、このメソッドは InvalidUid を許可しません。
    if (!checkAvailability || checkAvailability != m_MountHandle.GetUid())
    {
        return false;
    }
    // 既に移行済状態のバッファの場合はインポートを無視します。
    if (MigrationState::Done == m_CustomHeader.migration)
    {
        // これは従来手法「変換処理実施時に無条件に全アカウントDBを Discard()」の仕組みだと、
        // 以下のような経路を行うと、ユーザーA が`未変換` のために「移行処理の中断」扱いとなってしまい、
        // 「意図しない移行処理」の呼び出しが発生してしまうため、「意図しない移行処理」対策として、
        // `移行済` のアカウントDBにはインポートしないようにするためです。
        // 1. ユーザーA作成後、セーブデータコミットが行われるまでの間に、システム遮断( 電源遮断など )発生。
        // 2. メンテナンスモード起動して「ユーザーを残して初期化」してから通常起動。
        return true;
    }
    entry.powerChange = GetDetectedPowerChange();
    auto& migration = *pRecorder;
    if (migration.HasFullEntry())
    {
        WriteUnsafe(migration.GetEntryBuffer(), migration.GetFilledCount());
        // ジャーナル領域のファイル生成などファイルエントリ操作用の4ブロック(4 * 16KiB)を引いた領域が猶予領域。
        const size_t firstCommitLimit = MountConfiguration::SaveDataJournalSize - (64 * 1024);
        // １回コミット後はファイル生成などの分がない。
        const size_t secondaryCommitLimit = MountConfiguration::SaveDataJournalSize - (32 * 1024);
        const size_t commitLimit = (migration.GetCommitCount() > 0) ? secondaryCommitLimit : firstCommitLimit;
        const size_t flushedByte = migration.Flush() + migration.GetFullByteSize();
        if (flushedByte > commitLimit)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
            migration.Commit();
        }
    }
    migration.Add(entry);
    return true;
}

//!---------------------------------------------------------------------------
void AccountPlayEventBuffer::FinishMigration(MigrationRecorder* pRecorder) NN_NOEXCEPT
{
    // 既に移行済状態のバッファの場合はインポートを無視します。
    if (MigrationState::Done == m_CustomHeader.migration)
    {
        // これは従来手法「変換処理実施時に無条件に全アカウントDBを Discard()」の仕組みだと、
        // 以下のような経路を行うと、ユーザーA が`未変換` のために「移行処理の中断」扱いとなってしまい、
        // 「意図しない移行処理」の呼び出しが発生してしまうため、「意図しない移行処理」対策として、
        // `移行済` のアカウントDBにはインポートしないようにするためです。
        // 1. ユーザーA作成後、セーブデータコミットが行われるまでの間に、システム遮断( 電源遮断など )発生。
        // 2. メンテナンスモード起動して「ユーザーを残して初期化」してから通常起動。
        return;
    }

    // 移行完了状態付与。( ファイルへのフラッシュ/コミットが必要 )
    m_CustomHeader.migration = MigrationState::Done;

    // バッファ有効状態チェック。
    if (false == m_MountHandle.GetUid())
    {
        return;
    }

    // キャッシュエントリの最終フラッシュ。
    auto& migration = *pRecorder;
    if (migration.NeedsFlush())
    {
        WriteUnsafe(migration.GetEntryBuffer(), migration.GetFilledCount());
        migration.Flush();
    }
    else
    {
        // 移行完了状態を付与したヘッダをファイルへ反映。
        WriteHeaderOnlyUnsafe();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_MountHandle.Commit());
    migration.Commit();
    migration.Reset();
    NotifyPowerChange(PowerChange::None);
}











//!---------------------------------------------------------------------------
namespace {

    template<typename TEntryData, typename Predicate, int EntryCapacity>
    TEntryData* FindEntryFrom(TEntryData (&entries)[EntryCapacity], Predicate onMatch) NN_NOEXCEPT
    {
        for (int i = 0; i < EntryCapacity; ++i)
        {
            if (onMatch(entries[i]))
            {
                return &entries[i];
            }
        }
        return nullptr;
    }
}

//!---------------------------------------------------------------------------
//! 監視状態のリセット。
//! PowerOn 時に呼び出される想定。
//!---------------------------------------------------------------------------
void AccountPlayEventFactory::Condition::Invalidate() NN_NOEXCEPT
{
    dock.Invalidate();
    FindEntryFrom(applet, [](RecordEntry<Applet>& entry) -> bool
    {
        entry.Invalidate();
        return false;
    });
    FindEntryFrom(open, [](RecordEntry<UserAccount>& entry) -> bool
    {
        entry.Invalidate();
        return false;
    });
    application.Invalidate();
}

//!---------------------------------------------------------------------------
//! 現時点の起動中アプリケーションエントリを取得。
//!---------------------------------------------------------------------------
bool AccountPlayEventFactory::Condition::ApplyActiveApplication(AccountPlayEvent::Applet* pOutValue, const AccountPlayEvent::RecordCause& cause) NN_NOEXCEPT
{
    if( !application.IsValid() )
    {
        pOutValue->Invalidate();
        return false;
    }
    pOutValue->Apply(application.data, cause, dock.data.value);
    return true;
}

//!---------------------------------------------------------------------------
//! アカウントオープンイベント経由でのアカウントオープン状態監視の更新。
//!---------------------------------------------------------------------------
bool AccountPlayEventFactory::Condition::ReceiveOpenAccount(const PlayEvent& source) NN_NOEXCEPT
{
    const auto uid = GetUid(source);
    auto pEntry = FindEntryFrom(open, [&uid](const RecordEntry<UserAccount>& entry) -> bool { return entry.data.userId == uid; });
    if (!pEntry)
    {
        // 空きを探す。
        pEntry = FindEntryFrom(open, [](const RecordEntry<UserAccount>& entry) -> bool { return !entry.IsValid(); });
    }
    if (!pEntry)
    {
        NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventFactory: Could not find the empty stock buffer in the account open / close condition.\n");
        return false;   // Conditionストックの空きがない。
    }
    pEntry->data.userId = uid;
    pEntry->ApplyTime(source);

    // アプリケーション記録捜索。
    //   - 付属のアプレットデータはアプリケーションのものを利用する。
    //   - 基本的にはアプレット起動なしでのアカウントオープンはない想定だが、なければ空情報にする。
    ApplyActiveApplication(&pEntry->data, AccountPlayEvent::RecordCause::None);
    return true;
}

//!---------------------------------------------------------------------------
//! アカウントクローズイベント経由でのアカウントオープン状態監視の更新。
//!---------------------------------------------------------------------------
bool AccountPlayEventFactory::Condition::ReceiveCloseAccount(const PlayEvent& source, RecordEntry<UserAccount>* pOutRecordPreOpenedEntry) NN_NOEXCEPT
{
    const auto uid = GetUid(source);
    auto pEntry = FindEntryFrom(open, [&uid](const RecordEntry<UserAccount>& entry) -> bool { return entry.data.userId == uid; });
    if (!pEntry || !(pEntry->IsValid()))
    {
        NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventFactory: Could not find the opened record in the account close event received.\n");
        return false;   // Open レコードが見つからない。
    }
    *pOutRecordPreOpenedEntry = *pEntry;
    pEntry->Invalidate();
    return true;
}

//!---------------------------------------------------------------------------
//! アプレットイベント経由でのアプレット状態監視の更新。
//!---------------------------------------------------------------------------
bool AccountPlayEventFactory::Condition::ReceiveApplet(const PlayEvent& source, RecordEntry<Applet>* pOutRecordPreFocusedEntry) NN_NOEXCEPT
{
    // 起動中のアプリケーションに関する情報の更新。
    if( source.appletEventData.appletId == applet::AppletId_Application )
    {
        switch( source.appletEventData.eventType )
        {
        case AppletEventType::Launch:
        case AppletEventType::InFocus:
            {
                NN_SDK_ASSERT(!application.IsValid() || application.data.Equals(source.appletEventData));
                application.ApplyTime(source);
                application.data.Apply(source.appletEventData);
            }
            break;
        case AppletEventType::Exit:
        case AppletEventType::Terminate:
        case AppletEventType::AbnormalExit:
            {
                NN_SDK_ASSERT(application.data.Equals(source.appletEventData));
                application.Invalidate();
            }
            break;
        default:
            break;
        }
    }

    // 既存検索
    auto pEntry = FindEntryFrom(applet, [&source](const RecordEntry<Applet>& entry) -> bool { return entry.data.Equals(source.appletEventData); });

    // Application は Launch も Condition に含む。記録生成契機は InFocus のみ。
    const auto eventType = source.appletEventData.eventType;
    if (AppletEventType::InFocus == eventType || (AppletEventType::Launch == eventType && source.appletEventData.appletId == applet::AppletId_Application) )
    {
        if (!pEntry)
        {
            // 空きを探す。
            pEntry = FindEntryFrom(applet, [](const RecordEntry<Applet>& entry) -> bool { return !entry.IsValid(); });
        }
        if (pEntry)
        {
            pEntry->ApplyTime(source);
            pEntry->data.Apply(source.appletEventData);
        }
        else
        {
            NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventFactory: RecordEntry<Applet> is full. Unable to store applet(%u)'s record.\n", source.appletEventData.appletId);
        }
        return false;   // InFocusなら、コンディションストックのみ。
    }
    if (!pEntry || !(pEntry->IsValid()))
    {
        // 準不正: OutFocusなのに前回の記録がない、或いは無視可能なケース ( Background( OutFocus ) → Exit など )
        return false;
    }
    if (!(pEntry->data.IsInFocused()))
    {
        // 不正: 前回の記録が InFocus 以外になってる.
        DEBUG_LOG("[pdm] AccountPlayEventFactory: Could not find the previous `InFocus` record in the applet event(%s) received.\n", GetTypeString(eventType));
        pEntry->Invalidate();
        return false;
    }
    // 新イベントが Launch / InFocus 以外、かつ、前回の Condition が InFocus / Launch の時のみイベント生成。
    *pOutRecordPreFocusedEntry = *pEntry;
    pEntry->Invalidate();
    return true;
}



//!---------------------------------------------------------------------------
//! 「アカウントオープン期間」イベントの生成
//!---------------------------------------------------------------------------
bool AccountPlayEventFactory::MakeEventAccountOpenPeriod(AccountPlayEvent* pOutInOpenEvent,
    const RecordEntry<Condition::UserAccount>& openedEntry,
    const RecordCause& recordCause,
    int64_t closedSteadyClockValue) NN_NOEXCEPT
{
    auto& context = pOutInOpenEvent->context.inOpen;
    context.duration = closedSteadyClockValue > openedEntry.steadyClockValue ? static_cast<Bit32>(closedSteadyClockValue - openedEntry.steadyClockValue) : 0;

    // アプリケーション記録割り当て。
    auto& condition = m_Condition;
    if (openedEntry.data.IsValid())
    {
        // アカウントオープン時の記録があれば、そちらを利用する。
        context.applet.Apply(openedEntry.data, recordCause, openedEntry.data.operationMode);
        return true;
    }
    else if (condition.ApplyActiveApplication(&context.applet, recordCause))
    {
        // なければ、現時点のアクティブアプリケーション捜索。
        return true;
    }
    // アカウントに紐づくようなアプリケーションが見つからない。
    return false;
}

//!---------------------------------------------------------------------------
//! 指定アカウントの「アカウントオープン期間」及び、InFocus中アプレットの「InFocus期間」イベントの生成、及び通知。
//!---------------------------------------------------------------------------
void AccountPlayEventFactory::PublishEventAtAccountClose(const PlayEvent& occurEvent,
    const RecordEntry<Condition::UserAccount>& openedEntry, const RecordCause& recordCause,
    const account::Uid& uid) NN_NOEXCEPT
{
    // - 対象のアカウントに対して「アカウントクローズ」を記録追加。
    AccountPlayEvent inOpen(AccountPlayEvent::Category::InOpen, occurEvent);
    if (MakeEventAccountOpenPeriod(&inOpen, openedEntry, recordCause, occurEvent.steadyTime))
    {
        // 紐づくアプリケーションが見つかった場合は InOpen記録要求。
        OnReceiveEventCreated(uid, inOpen);
    }

    // - InFocus中アプレット全てに対して「InFocus終了」を記録追加。
    auto openedSteadyClockValue = openedEntry.steadyClockValue;
    AccountPlayEvent inFocus(AccountPlayEvent::Category::InFocus, occurEvent);
    auto& context = inFocus.context.inFocus;

    auto& condition = m_Condition;
    FindEntryFrom(condition.applet, [&](RecordEntry<Condition::Applet>& entry) -> bool
    {
        if (entry.IsValid() && entry.data.IsInFocused())
        {
            // アカウントオープンとエントリの InFocus時刻を比較して新しい時刻を開始時刻とします。
            //  - applet 起動中に account close → open がありえるため。
            const auto beginUserTime = std::max(entry.steadyClockValue, openedSteadyClockValue);
            context.duration = occurEvent.steadyTime > beginUserTime ? static_cast<Bit32>(occurEvent.steadyTime - beginUserTime) : 0;
            context.applet.Apply(entry.data, recordCause, condition.dock.data.value);   // 対動作モード集計対策(竹)
            OnReceiveEventCreated(uid, inFocus);
        }
        return false;
    });
}

//!---------------------------------------------------------------------------
void AccountPlayEventFactory::ImportEventForApplet(const PlayEvent& source) NN_NOEXCEPT
{
    auto& condition = m_Condition;
    RecordEntry<Condition::Applet> recordPreFocused;
    if (condition.ReceiveApplet(source, &recordPreFocused))
    {
        // イベント生成＆通知
        // - オープン中のアカウント全てに対して「InFocus終了」を記録追加。
        AccountPlayEvent inFocus(AccountPlayEvent::Category::InFocus, source);
        auto& context = inFocus.context.inFocus;
        context.applet.Apply(source.appletEventData, RecordCause::FocusOut, condition.dock.data.value);  // 対動作モード集計対策(竹)

        FindEntryFrom(condition.open, [&](const RecordEntry<Condition::UserAccount>& entry) -> bool
        {
            if (entry.IsValid())
            {
                // アカウントオープンとエントリの InFocus時刻を比較して新しい時刻を開始時刻とします。
                //  - applet 起動中に account close → open がありえるため。
                const auto beginUserTime = std::max(entry.steadyClockValue, recordPreFocused.steadyClockValue);
                context.duration = source.steadyTime > beginUserTime ? static_cast<Bit32>(source.steadyTime - beginUserTime) : 0;
                OnReceiveEventCreated(entry.data.userId, inFocus);
            }
            return false;
        });
    }
}

//!---------------------------------------------------------------------------
void AccountPlayEventFactory::ImportEventForUserAccount(const PlayEvent& source) NN_NOEXCEPT
{
    auto& condition = m_Condition;
    switch (source.userAccountEventData.eventType)
    {
    case UserAccountEventType::Open:
        // アカウントオープン時はコンディションストックのみ。
        if (!condition.ReceiveOpenAccount(source))
        {
            NN_DETAIL_PDM_ERROR("[pdm] AccountPlayEventFactory::ImportEventForUserAccount: Could not register the open account condition.\n");
        }
        break;
    case UserAccountEventType::Close:
        {
            RecordEntry<Condition::UserAccount> recordPreOpened;
            if (condition.ReceiveCloseAccount(source, &recordPreOpened))
            {
                // イベント生成＆通知
                PublishEventAtAccountClose(source, recordPreOpened, RecordCause::AccountClose, GetUid(source));
            }
        }
        break;
    case UserAccountEventType::NetworkServiceAccountAvailable:
        {
            // イベント生成＆通知
            // 対象アカウントに「NSA利用可能」の記録追加。
            const auto uid = GetUid(source);
            AccountPlayEvent eventValue(AccountPlayEvent::Category::NsaAvailable, source);
            eventValue.context.nsaAvailable.nsaId = AccountPlayEvent::PackedId64(
                source.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdHi,
                source.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdLow
            );
            OnReceiveEventCreated(uid, eventValue);
        }
        break;
    case UserAccountEventType::NetworkServiceAccountUnavailable:
        {
            // イベント生成＆通知
            // 対象アカウントに「NSA利用不可」の記録追加。
            const auto uid = GetUid(source);
            AccountPlayEvent eventValue(AccountPlayEvent::Category::NsaUnavailable, source);
            OnReceiveEventCreated(uid, eventValue);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

//!---------------------------------------------------------------------------
void AccountPlayEventFactory::ImportEventForPowerStateChange(const PlayEvent& source) NN_NOEXCEPT
{
    switch (source.powerStateChangeEventData.eventType)
    {
    case PowerStateChangeEventType::On:
        {
            // 電源ONイベントの場合は、監視状態をリセットします。(異常電源切断などへの担保)
            m_Condition.Invalidate();
            OnReceivePowerOn();
        }
        break;
    case PowerStateChangeEventType::Off:
        {
            // イベント生成＆通知 (アカウント / アプレットのストックがある場合)
            FindEntryFrom(m_Condition.open, [&](const RecordEntry<Condition::UserAccount>& entry) -> bool
            {
                if (entry.IsValid())
                {
                    PublishEventAtAccountClose(source, entry, RecordCause::PowerOff, entry.data.userId);
                }
                return false;
            });

            // 全稼働中 Buffer の Flush / Commit.
            OnReceivePowerOff();
        }
        break;
    default:
        // Ignore because do not have an operation in this case.
        break;
    }
}

//!---------------------------------------------------------------------------
void AccountPlayEventFactory::ImportEventForOperationModeChange(const PlayEvent& source) NN_NOEXCEPT
{
    auto& condition = m_Condition;
    condition.dock.ApplyTime(source);
    condition.dock.data.value = source.operationModeEventData.operationMode;
}

//!---------------------------------------------------------------------------
void AccountPlayEventFactory::ImportEventUnsafe(const PlayEvent& source) NN_NOEXCEPT
{
    switch (source.eventCategory)
    {
    case PlayEventCategory::Applet:
        TraceEventApplet("Import", source);
        ImportEventForApplet(source);
        break;
    case PlayEventCategory::UserAccount:
        TraceEventUserAccount("Import", source);
        ImportEventForUserAccount(source);
        break;
    case PlayEventCategory::PowerStateChange:
        TraceEventPowerChange("Import", source);
        ImportEventForPowerStateChange(source);
        break;
    case PlayEventCategory::OperationModeChange:
        ImportEventForOperationModeChange(source);
        break;
    case PlayEventCategory::SteadyClockReset:
        // Ignore because do not have an operation in this case.
        break;
    default:
        // Ignore because do not have an operation in this case.
        break;
    }
}

AccountPlayEventFactory::Condition& AccountPlayEventFactory::GetCondition() NN_NOEXCEPT
{
    return m_Condition;
}








//!---------------------------------------------------------------------------
bool AccountPlayEventProvider::CheckRequestForForcedMigration() NN_NOEXCEPT
{
    bool forceMigration = false;
    if (nn::settings::fwdbg::GetSettingsItemValue(&forceMigration, sizeof(forceMigration), "pdm", "force_migrate_account_database") != sizeof(forceMigration))
    {
        forceMigration = false;
    }
    m_MigrationHandle.ApplyForceExecution(forceMigration);
    NN_DETAIL_PDM_TRACE("[pdm] force_migrate_account_database: %s\n", (forceMigration ? "true" : "false"));
    return forceMigration;
}

//!---------------------------------------------------------------------------
bool AccountPlayEventProvider::Initialize(const account::Uid* pBootUsers, int userCount, const MigrationState initialMigrationState) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_BufferUpdateLock);

    // 本関数呼び出し前に CheckRequestForForcedMigration() で設定取得している想定。
    // 引数受け取りでも良いけど設定情報の事前読み込み/保管を保障した方が良い気がしたので。
    const auto forceMigration = m_MigrationHandle.IsForceExecution();
    bool isDoneAllBufferMigration = !forceMigration;    // ユーザーゼロ時用.
    for (int i = 0; i < BufferCapacity; ++i)
    {
        auto& buffer = m_Buffers[i];
        buffer.Initialize(i);
        if (i < userCount)
        {
            // 登録済ユーザーだけ起動する。( 起動時はセーブデータ生成/マウントを行います )
            NN_ABORT_UNLESS_RESULT_SUCCESS(buffer.Activate(pBootUsers[i], initialMigrationState, true));
            if (false == buffer.IsCompleteMigration())
            {
                isDoneAllBufferMigration = false;
            }
            else if (forceMigration)
            {
                buffer.DiscardCompleteMigration();
            }
        }
    }
    return isDoneAllBufferMigration;
}


//!---------------------------------------------------------------------------
// 新しいユーザーリストと、既存のオープン済バッファを比較。
void AccountPlayEventProvider::UpdateUserRegistry(const account::Uid* pNewUsers, const int userCount) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_BufferUpdateLock);

    // m_Buffers にしかない。 -> 削除されたユーザー

    for (int i = 0; i < BufferCapacity; ++i)
    {
        auto& buffer = m_Buffers[i];
        if (userCount == buffer.Find(pNewUsers, userCount))
        {
            // 削除されてるんで寝かす。
            buffer.Deactivate();
        }
    }

    // pNewUsers にしかない。 -> 新しく登録されたユーザー

    int newlyCount = 0;
    uint16_t newlyIndex[BufferCapacity] = {};
    for (int j = 0; j < userCount; ++j)
    {
        bool hit = false;
        for (int i = 0; i < BufferCapacity; ++i)
        {
            auto& buffer = m_Buffers[i];
            if (0 == buffer.Find(&pNewUsers[j], 1))
            {
                hit = true;
                break;
            }
        }
        if (!hit)
        {
            newlyIndex[newlyCount++] = static_cast<uint16_t>(j);
        }
    }
    // 新しく初期化。
    for (int i = 0; i < newlyCount; ++i)
    {
        auto pIdleBuffer = FindIdleBufferUnsafe();
        if (pIdleBuffer)
        {
            pIdleBuffer->Activate(pNewUsers[newlyIndex[i]]);
        }
    }
}

//!---------------------------------------------------------------------------
AccountPlayEventBuffer* AccountPlayEventProvider::FindIdleBufferUnsafe() NN_NOEXCEPT
{
    return FindEntryFrom(m_Buffers, [](const AccountPlayEventBuffer& buffer) -> bool { return buffer.IsIdle(); });
}

//!---------------------------------------------------------------------------
AccountPlayEventBuffer* AccountPlayEventProvider::FindBufferUnsafe(const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindEntryFrom(m_Buffers, [&uid](const AccountPlayEventBuffer& buffer) -> bool
    {
        return (0 == buffer.Find(&uid, 1));
    });
    if (!pBuffer)
    {
        NN_DETAIL_PDM_WARN("[pdm] AccountPlayEventProvider: Could not find the specified user account in activated services.\n");
    }
    return pBuffer;
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::Import(const PlayEvent& source) NN_NOEXCEPT
{
    NN_SDK_ASSERT(CanImportNormally());
    ImportEventUnsafe(source);
    if( source.eventCategory == PlayEventCategory::UserAccount &&
        source.userAccountEventData.eventType == UserAccountEventType::Open )
    {
        m_RecentlyPlayedApplicationUpdateEvent.Signal();
    }
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::OnReceivePowerOn() NN_NOEXCEPT
{
    m_ActiveFactoryReceiver->OnReceivePowerOn(this);
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::OnReceivePowerOff() NN_NOEXCEPT
{
    m_ActiveFactoryReceiver->OnReceivePowerOff(this);
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::OnReceiveEventCreated(const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] AccountPlayEventProvider: received new event [ %d ].\n", newEvent.category);
    m_ActiveFactoryReceiver->OnReceiveEventCreated(this, uid, newEvent);
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::HandlePowerOn() NN_NOEXCEPT
{
    FindEntryFrom(m_Buffers, [](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.NotifyPowerChange(AccountPlayEventBuffer::PowerChange::PowerOn);
        return false;
    });
}

void AccountPlayEventProvider::HandlePowerOff() NN_NOEXCEPT
{
    FindEntryFrom(m_Buffers, [](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.Flush();
        buffer.NotifyPowerChange(AccountPlayEventBuffer::PowerChange::None);
        return false;
    });
}

void AccountPlayEventProvider::HandleEventCreated(const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        pBuffer->Add(newEvent, uid);
    }
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::HandleMigrationPowerOn(MigrationHandle* pHandle) NN_NOEXCEPT
{
    NN_UNUSED(pHandle);
    FindEntryFrom(m_Buffers, [](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.NotifyPowerChange(AccountPlayEventBuffer::PowerChange::PowerOn);
        return false;
    });
}

void AccountPlayEventProvider::HandleMigrationPowerOff(MigrationHandle* pHandle) NN_NOEXCEPT
{
    NN_UNUSED(pHandle);
    FindEntryFrom(m_Buffers, [](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.NotifyPowerChange(AccountPlayEventBuffer::PowerChange::None);
        return false;
    });
}

void AccountPlayEventProvider::HandleMigrationEventCreated(MigrationHandle* pHandle, const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pHandle->GetRecorderCount() >= BufferIndexCountMax);
    int index = 0;
    auto pRecorders = pHandle->GetRecorders();
    FindEntryFrom(m_Buffers, [&](AccountPlayEventBuffer& buffer) -> bool
    {
        return buffer.ImportMigration(&pRecorders[index++], newEvent, uid);
    });
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::Flush(const uint32_t excessRate) NN_NOEXCEPT
{
    FindEntryFrom(m_Buffers, [excessRate](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.Flush(excessRate);
        return false;
    });
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Resume(const account::Uid& uid) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] AccountPlayEventProvider: resume requested.\n");

    // Buffer側で排他してるので不正動作しない想定だが、UpdateUserRegistry() との競合は好ましくないので排他しておく。
    NN_UTIL_LOCK_GUARD(m_BufferUpdateLock);
    auto pBuffer = FindBufferUnsafe(uid);
    return (pBuffer) ? pBuffer->Resume() : ResultServiceNotAvailable();
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Suspend(const account::Uid& uid) NN_NOEXCEPT
{
    DEBUG_LOG("[pdm] AccountPlayEventProvider: suspend requested.\n");

    // Buffer側で排他してるので不正動作しない想定だが、UpdateUserRegistry() との競合は好ましくないので排他しておく。
    NN_UTIL_LOCK_GUARD(m_BufferUpdateLock);
    auto pBuffer = FindBufferUnsafe(uid);
    return (pBuffer) ? pBuffer->Suspend() : ResultServiceNotAvailable();
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Query(PlayStatistics* pOutValue, const ncm::ApplicationId& id,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary,
    const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        NN_RESULT_DO(pBuffer->Query(pOutValue, id, pReadTemporary, byteOfTemporary));
        AdjustPlayStatisticsWithCurrentCondition(pOutValue, id, uid);
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Query(uint32_t* pOutCount, PlayStatistics* pOutValues, uint32_t receiveCapacity,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary,
    const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        NN_RESULT_DO(pBuffer->Query(pOutCount, pOutValues, receiveCapacity, pReadTemporary, byteOfTemporary));

        // クエリ対象のアカウントがオープン中であれば、オープンしたアプリの情報を調整する。
        auto pOpenEntry = FindEntryFrom(GetCondition().open, [&](RecordEntry<Condition::UserAccount>& e) -> bool {
            return e.IsValid() && e.data.userId == uid;
        });
        if( !pOpenEntry )
        {
            NN_RESULT_SUCCESS;
        }
        auto appId = ncm::ApplicationId{ pOpenEntry->data.programId.ToValue64() };

        for( uint32_t i = 0; i < *pOutCount; i++ )
        {
            if( appId == pOutValues[i].applicationId )
            {
                // リストに含まれる場合（一度遊んだことがあり、記録が残っているアプリケーションを再度遊んでいる場合）はそれを対象に調整。
                AdjustPlayStatisticsWithCurrentCondition(&pOutValues[i], appId, uid);
                NN_RESULT_SUCCESS;
            }
        }

        // リストにない場合（初めて遊んでいる最中）は、リストの容量に余裕があれば新規エントリーとして追加、埋まっていれば入れ替え。
        if( *pOutCount < receiveCapacity )
        {
            detail::QueryServiceHelper::Initialize(&pOutValues[*pOutCount], appId);
            AdjustPlayStatisticsWithCurrentCondition(&pOutValues[*pOutCount], appId, uid);
            *pOutCount += 1;
        }
        else
        {
            int r = detail::QueryServiceHelper::SelectPlayStatisticsIndexToReplace(pOutValues, *pOutCount);
            detail::QueryServiceHelper::Initialize(&pOutValues[r], appId);
            AdjustPlayStatisticsWithCurrentCondition(&pOutValues[r], appId, uid);
        }
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Query(uint32_t* pOutCount, AccountEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary,
    const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        return pBuffer->Query(pOutCount, pOutValues, receiveCapacity, eventIndexOffset, pReadTemporary, byteOfTemporary);
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::QueryRecentlyPlayedApplication(uint32_t* pOutCount, ncm::ApplicationId* pOutValues, uint32_t receiveCapacity,
    Bit8* const pReadTemporary, uint32_t byteOfTemporary,
    const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if( pBuffer )
    {
        auto pOpenEntry = FindEntryFrom(GetCondition().open, [&](RecordEntry<Condition::UserAccount>& e) -> bool {
            return e.IsValid() && e.data.appletId == applet::AppletId_Application && e.data.userId == uid;
        });
        uint32_t presetCount = 0;
        if( pOpenEntry )
        {
            presetCount = 1;
            pOutValues[0].value = pOpenEntry->data.programId.ToValue64();
        }
        return pBuffer->QueryRecentlyPlayedApplication(pOutCount, pOutValues, receiveCapacity, presetCount, pReadTemporary, byteOfTemporary);
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

os::SystemEvent* AccountPlayEventProvider::GetRecentlyPlayedApplicationUpdateEvent() NN_NOEXCEPT
{
    return &m_RecentlyPlayedApplicationUpdateEvent;
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::Discard(const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        return pBuffer->Discard();
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
//! 注) AccountPlayEventFactory::Condition はクリアしない。
//! Condition の状態はシステム全体と整合性を保つべきであるため。
void AccountPlayEventProvider::Discard() NN_NOEXCEPT
{
    FindEntryFrom(m_Buffers, [](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.Discard();
        return false;
    });
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::QueryRaw(uint32_t* pOutCount, AccountPlayEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset, const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        return pBuffer->QueryRaw(pOutCount, pOutValues, receiveCapacity, eventIndexOffset);
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
Result AccountPlayEventProvider::QueryAvailableRange(uint32_t* pOutCount, uint32_t* pOutStartIndex, uint32_t* pOutLastIndex, const account::Uid& uid) NN_NOEXCEPT
{
    auto pBuffer = FindBufferUnsafe(uid);
    if (pBuffer)
    {
        return pBuffer->QueryAvailableRange(pOutCount, pOutStartIndex, pOutLastIndex);
    }
    NN_RESULT_THROW(ResultUserNotExist());
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::AdjustPlayStatisticsWithCurrentCondition(PlayStatistics* pStatistics, const ncm::ApplicationId& appId, const account::Uid& uid) NN_NOEXCEPT
{
    auto pOpenEntry = FindEntryFrom(GetCondition().open, [&](RecordEntry<Condition::UserAccount>& e) -> bool {
        return e.IsValid() && e.data.programId.ToValue64() == appId.value && e.data.userId == uid;
    });
    if( !pOpenEntry )
    {
        return;
    }

    // 対象のアプリでアカウントが Open 中の場合、プレイ回数を +1 する。
    NN_DETAIL_PDM_TRACE("[pdm] AdjustPlayStatisticsWithCurrentCondition -> Uid %016llx-%016llx is open in 0x%016llx. PlayCount + 1.\n",
        uid._data[0], uid._data[1], appId);
    pStatistics->totalPlayCount += 1;

    auto pAppletEntry = FindEntryFrom(GetCondition().applet, [&](RecordEntry<Condition::Applet>& e) -> bool {
        return e.IsValid() && e.data.programId.ToValue64() == appId.value;
    });
    if( !pAppletEntry )
    {
        return;
    }

    // Open 中でかつアプリが InFocus 中の場合、プレイ時間をその期間分補正する。
    auto currentTimePoint = GetSteadyClockTimeForPlayEvent();
    auto inFocusStart = std::max(pOpenEntry->steadyClockValue, pAppletEntry->steadyClockValue);
    if( !(currentTimePoint >= inFocusStart) )
    {
        NN_DETAIL_PDM_WARN("[pdm] AdjustPlayStatisticsWithCurrentCondition -> InFocus/Open started at %lld. Is is later than current time point, which is %lld.\n",
            inFocusStart, currentTimePoint);
        return;
    }

    auto currentInFocusSessionTimeInSeconds = currentTimePoint - inFocusStart;
    NN_DETAIL_PDM_TRACE("[pdm] AdjustPlayStatisticsWithCurrentCondition -> 0x%016llx is InFocus state with Uid %016llx-%016llx opened since %lld seconds ago (current = %lld, start = %lld). PlayTime + %lld (minutes).\n",
        appId, uid._data[0], uid._data[1], currentInFocusSessionTimeInSeconds, currentTimePoint, inFocusStart, currentInFocusSessionTimeInSeconds / 60);
    pStatistics->totalPlayTime += static_cast<uint32_t>(currentInFocusSessionTimeInSeconds / 60);

    // 「最初に遊んだ時刻」が未設定の場合（初 Open 初 InFocus で記録が全くない場合）は起点イベントの時刻を設定。
    if( pStatistics->firstEventTime.userClockTime == 0 )
    {
        // Open 後に InFocus の場合、その前に OutOfFocus もしくは Background があるはず = InFocus が記録されているはず、
        // なので必ず Open が後になる想定ではあるものの、念のため判定。
        auto entryTime = pOpenEntry->steadyClockValue > pAppletEntry->steadyClockValue ? pOpenEntry->time : pAppletEntry->time;

        // 未記録の情報を使用するためインデックスは設定できない。使用されていないので問題はない。
        pStatistics->firstEventTime.eventIndex = 0;
        pStatistics->firstEventTime.userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(entryTime.user);
        pStatistics->firstEventTime.networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(entryTime.network);
    }
    // 「最後に遊んだ時刻」を現在時刻に時刻を設定。
    pStatistics->latestEventTime.eventIndex = 0;
    pStatistics->latestEventTime.userClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(detail::GetUserClockTimeForPlayEvent());
    pStatistics->latestEventTime.networkClockTime = detail::GetElapsedMinutesSinceInputPosixTimeMin(detail::GetNetworkClockTimeForPlayEvent());
}



//!---------------------------------------------------------------------------
void AccountPlayEventProvider::InitializeMigration(MigrationRecorder* pRecorders, const uint8_t recorderCount) NN_NOEXCEPT
{
    m_MigrationHandle.Reset(pRecorders, recorderCount);
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::PreapreMigration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_MigrationHandle.GetRecorderCount() >= BufferIndexCountMax);

    // 以降はマイグレーション( デバイスDB → アカウントDB )インポート扱いになります。
    m_ActiveFactoryReceiver = &m_MigrationHandle;

    int index = 0;
    auto pRecorders = m_MigrationHandle.GetRecorders();
    FindEntryFrom(m_Buffers, [&](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.PrepareMigration(&pRecorders[index++]);
        return false;
    });
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::ImportMigration(const PlayEvent& source) NN_NOEXCEPT
{
    ImportEventUnsafe(source);
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::FinishMigration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_MigrationHandle.GetRecorderCount() >= BufferIndexCountMax);
    int index = 0;
    auto pRecorders = m_MigrationHandle.GetRecorders();
    FindEntryFrom(m_Buffers, [&](AccountPlayEventBuffer& buffer) -> bool
    {
        buffer.FinishMigration(&pRecorders[index++]);
        return false;
    });

    // 以降は通常のインポート扱いになります。
    m_ActiveFactoryReceiver = &m_FactoryReceiver;
}

//!---------------------------------------------------------------------------
void AccountPlayEventProvider::FinalizeMigration() NN_NOEXCEPT
{
    m_MigrationHandle.Reset(nullptr, 0);

    // 以降は通常のインポート扱いになります。
    m_ActiveFactoryReceiver = &m_FactoryReceiver;

    // PowerOn を送信して、Provider を起動状態として初期化します。
    // システムから正しく PowerOn を受信しても Condition の初期化のみなので問題ありません。
    // PowerOn に起因したイベント記録が発生する仕様に変更された場合は、仕様にあわせて調整してください。
    PlayEvent powerOn;
    powerOn.eventCategory = PlayEventCategory::PowerStateChange;
    powerOn.powerStateChangeEventData.eventType = PowerStateChangeEventType::On;
    ImportEventUnsafe(powerOn);
}

}}}





//!---------------------------------------------------------------------------
//! @name デバイスDB → アカウントDB マイグレーション。
//@{
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>

//! @name       速度検証用コンパイルスイッチ
//! @details    製品採用構成は全て `無効` です。
//@{

//#define ENABLE_LOGOUT_IN_RELEASE              // リリースビルドでのマイグレーションレポートログ出力を実施したい場合に有効にします。
//#define ENABLE_MIGRATION_PERFORMANCE_MEASURE  // 速度計測。
//#define ENABLE_LUXURIOUS_WRITE_BUFFER         // 贅沢に静的書き込みバッファ(bss)を確保して検証する場合に有効にします。

//@}

//! @brief マイグレーション用ログエイリアス
#if defined(ENABLE_LOGOUT_IN_RELEASE)

#define MIGRATION_SETUP_SDK_LOG_META_DATA(                              \
    logMetaData, inModule, inSeverity, inVerbosity)                     \
    do {                                                                \
        logMetaData.sourceInfo.lineNumber = __LINE__;                   \
        logMetaData.sourceInfo.fileName = __FILE__;                     \
        logMetaData.sourceInfo.functionName = NN_CURRENT_FUNCTION_NAME; \
        logMetaData.moduleName = "$" NN_MACRO_STRINGIZE(inModule);      \
        logMetaData.severity = ::nn::diag::LogSeverity_##inSeverity;    \
        logMetaData.verbosity = inVerbosity;                            \
        logMetaData.useDefaultLocaleCharset = false;                    \
        logMetaData.pAdditionalData = static_cast<void*>(0);            \
        logMetaData.additionalDataBytes = 0;                            \
    } while (NN_STATIC_CONDITION(0))

#define MIGRATION_STRUCTURED_SDK_LOG(                                   \
    module, severity, verbosity, ...)                                   \
    do {                                                                \
        ::nn::diag::LogMetaData logMetaData;                            \
        MIGRATION_SETUP_SDK_LOG_META_DATA(                              \
            logMetaData, module, severity, verbosity);                  \
        ::nn::diag::detail::LogImpl(logMetaData, __VA_ARGS__);          \
    } while (NN_STATIC_CONDITION(0))

#define MIGRATION_TRACE(...)    MIGRATION_STRUCTURED_SDK_LOG(pdm, Trace, 0, ##__VA_ARGS__)

#else

#define MIGRATION_TRACE(...) NN_DETAIL_PDM_TRACE(__VA_ARGS__)

#endif

namespace nn { namespace pdm { namespace detail {

namespace {

//!---------------------------------------------------------------------------
#if defined(ENABLE_MIGRATION_PERFORMANCE_MEASURE)

class MigrationReporter
{
public:
    void Create(const AccountPlayEventProvider& provider, MigrationRecorder* const pRecorders, const size_t recorderCount) NN_NOEXCEPT
    {
        for (AccountPlayEventProvider::BufferIndex i = 0; i < AccountPlayEventProvider::BufferIndexCountMax; ++i)
        {
            auto& report = m_Report[i];
            report.commitCount = (static_cast<size_t>(i) < recorderCount)
                ? pRecorders[i].GetTotalCommitCount()
                : 0;
            report.eventCount = provider.GetBuffer(i).GetCount();
        }
    }

    void Output() NN_NOEXCEPT
    {
        for (int i = 0; i < AccountPlayEventProvider::BufferCapacity; ++i)
        {
            auto& report = m_Report[i];
            MIGRATION_TRACE("[pdm] Migration report pdmUA[%d]( commit: %lu, event: %lu )\n",
                i, report.commitCount, report.eventCount
            );
            NN_UNUSED(report);
        }
    }

private:
    struct Report
    {
        uint32_t    eventCount;
        uint32_t    commitCount;
    };

    Report  m_Report[AccountPlayEventProvider::BufferIndexCountMax];
};
MigrationReporter g_MigrationReporter;

#endif

//!---------------------------------------------------------------------------
void MarkCompleteMigration() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(MigrationCompleteFilePath, 0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(detail::MountName));
}

//!---------------------------------------------------------------------------
void RunMigrationImpl(void* pReadBuffer, uint32_t readCapacity) NN_NOEXCEPT
{
    auto& dbAccount = AccountPlayEventProvider::GetInstance();

    dbAccount.PreapreMigration();
    {
        auto& dbDevice = PlayEventBuffer::GetInstance();
        dbDevice.BeginRead();
        NN_UTIL_SCOPE_EXIT{dbDevice.EndRead(); };
        uint32_t readedCount, startIndex = dbDevice.GetStartIndex();
        auto pReadableBuffer = reinterpret_cast<PlayEvent*>(pReadBuffer);
        const auto readableCount = static_cast<int>(readCapacity / sizeof(PlayEvent));
        while ((readedCount = dbDevice.Read(pReadableBuffer, startIndex, readableCount)) > 0)
        {
            for (uint32_t i = 0; i < readedCount; ++i)
            {
                dbAccount.ImportMigration(pReadableBuffer[i]);
            }
            startIndex += readedCount;
        }
    }
    dbAccount.FinishMigration();
}

//!---------------------------------------------------------------------------
void RunMigration(AccountPlayEventProvider* const pProvider, LockableMemoryBuffer* const pLockableBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pLockableBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pProvider);

    auto pReadBuffer = pLockableBuffer->GetPointer();

#if defined(ENABLE_LUXURIOUS_WRITE_BUFFER)

    // 確保済ジャーナル領域 - 64 KiB を上限とします。
    const size_t BufferByteCapacity = 32 * 1024;
    NN_FUNCTION_LOCAL_STATIC(Bit8, s_WriteBuffers, [AccountPlayEventProvider::BufferCapacity * BufferByteCapacity]);

    const auto readCapacity = pLockableBuffer->GetCapacity();
    Bit8* pWriteBuffers = s_WriteBuffers;

#else

    const auto readCapacity = pLockableBuffer->GetCapacity() / 2;
    const size_t BufferByteCapacity = readCapacity / AccountPlayEventProvider::BufferCapacity;
    Bit8* pWriteBuffers = &pReadBuffer[readCapacity];

#endif

    MigrationRecorder sharedRecorders[AccountPlayEventProvider::BufferCapacity];
    for (int i = 0; i < AccountPlayEventProvider::BufferCapacity; ++i)
    {
        sharedRecorders[i].Initialize(pWriteBuffers, BufferByteCapacity);
        pWriteBuffers = &pWriteBuffers[BufferByteCapacity];
    }
    pProvider->InitializeMigration(sharedRecorders, NN_ARRAY_SIZE(sharedRecorders));
    RunMigrationImpl(pReadBuffer, readCapacity);
    pProvider->FinalizeMigration();

#if defined(ENABLE_MIGRATION_PERFORMANCE_MEASURE)
    g_MigrationReporter.Create(*pProvider, sharedRecorders, NN_ARRAY_SIZE(sharedRecorders));
#endif
}

}

//!---------------------------------------------------------------------------
// Generic テスト向けに単独関数化。
// そのため、無名空間での宣言ではなく、pdm::detail 内に宣言します。
void MigrateAccountPlayEventDatabase(AccountPlayEventProvider* pProvider, LockableMemoryBuffer* const pLockableBuffer) NN_NOEXCEPT
{
#if defined(ENABLE_MIGRATION_PERFORMANCE_MEASURE)
    const auto tickBegin = nn::os::GetSystemTick();
#endif

    NN_UTIL_LOCK_GUARD(*pLockableBuffer);
    RunMigration(pProvider, pLockableBuffer);

#if defined(ENABLE_MIGRATION_PERFORMANCE_MEASURE)
    MIGRATION_TRACE("[pdm] MigrateAccountPlayEventDatabase: Migration tarnsfer time => [ %lld ] ms.\n",
        (nn::os::GetSystemTick() - tickBegin).ToTimeSpan().GetMilliSeconds());
    g_MigrationReporter.Output();
    NN_UNUSED(tickBegin);
#endif
}

//!---------------------------------------------------------------------------
void InitializeAccountPlayEventDatabase(LockableMemoryBuffer* const pLockableBuffer) NN_NOEXCEPT
{
    // アカウントデータベースインスタンス取得。
    auto& dbAccount = AccountPlayEventProvider::GetInstance();
    if (dbAccount.CheckRequestForForcedMigration())
    {
        MIGRATION_TRACE("[pdm] InitializeAccountPlayEventDatabase: Detected the forced migration request !!\n");
        // 強制マイグレーション実施要求があった場合、"pdm:/migration.ok" を削除します。
        fs::DeleteFile(MigrationCompleteFilePath);
    }

    // "pdm:/migration.ok" 確認。
    fs::DirectoryEntryType entryType;
    const auto hasMigrationCompleted = fs::GetEntryType(&entryType, MigrationCompleteFilePath);
    const MigrationState initialState = (hasMigrationCompleted.IsSuccess()) ? MigrationState::Done : MigrationState::None;

    int count;
    account::Uid uids[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, uids, account::UserCountMax));
    if (dbAccount.Initialize(uids, count, initialState))
    {
        if (MigrationState::None == initialState)
        {
            // "pdm:/migration.ok" ないが、以下の何れかの条件に合致するため移行済と判断。
            //  - 既存バッファの全てが migration 完了済。
            //      ケース: デバイスDB → アカウントDB変換完了後、「ユーザーを残して初期化」してから通常起動。
            //  - 起動時登録済ユーザーがいない。
            //      ケース: account::ListAllUsers() のカウントがゼロ。
            MarkCompleteMigration();
            MIGRATION_TRACE("[pdm] InitializeAccountPlayEventDatabase: Could not find the \"migration.ok\", but account database was migration already completed => [initial users: %d].\n", count);
        }
        else
        {
            // "pdm:/migration.ok" が存在して、各アカウントDBファイルのヘッダマークも全て「移行済」の場合。
            MIGRATION_TRACE("[pdm] InitializeAccountPlayEventDatabase: Migration was already completed => [initial users: %d].\n", count);
        }
        return;  // 移行済
    }
    NN_SDK_ASSERT_EQUAL(initialState, MigrationState::None);

    //-----------------------------------------------------------------
    // 以降はマイグレーション実施。( デバイスDB→アカウントDB変換 )
    // "pdm:/migration.ok" が存在せず、起動時ユーザーのアカウントDBファイルの何れかのヘッダマークが「移行済」以外の場合。
    //-----------------------------------------------------------------
    MIGRATION_TRACE("[pdm] InitializeAccountPlayEventDatabase: Migration required.\n");

    // 変換処理。
    MigrateAccountPlayEventDatabase(&dbAccount, pLockableBuffer);

    // 変換完了。
    MarkCompleteMigration();

    MIGRATION_TRACE("[pdm] InitializeAccountPlayEventDatabase: Migration was successfully completed.\n");
}

}}}
//@}
