﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/pcv.h>
#include <nn/pcv/pcv_IArbitrationManager.sfdl.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include "../detail/pcv_ServiceName.h"
#include "pcv_ArbitrationServer.h"
#include "pcv_ModuleListForArbitration.h"
#include "pcv_PcvServiceImpl.h"

namespace nn { namespace pcv { namespace server {

namespace {

const int SessionCountMax = 1;

const int NumberOfManagers = 1;

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, NumberOfManagers>
{};

std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

nn::sf::UnmanagedServiceObject<IArbitrationManager, ArbitrationManagerImpl> g_ArbitrationManager;

struct ModuleEntry
{
    Module id;
    bool released;
};

ModuleEntry g_ModuleIdList[NumberOfModulesForArbitration];

} // namespace

void InitializeForArbitration() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    // 調停対象モジュールのリストの初期化。
    for ( int index = 0; index < NumberOfModulesForArbitration; index++ )
    {
        g_ModuleIdList[index].id = ModuleListForArbitration[index];
        g_ModuleIdList[index].released = false;
    }

    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    g_pMyServerManager->RegisterObjectForPort(g_ArbitrationManager.GetShared(), SessionCountMax, nn::pcv::detail::ArbitrationServiceName);

    g_pMyServerManager->Start();
}

void WaitForArbitration() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // 調停対象モジュールが無い場合はすぐに終了する。
    if ( NumberOfModulesForArbitration == 0 )
    {
        return;
    }

    g_pMyServerManager->LoopAuto();
}

void FinalizeForArbitration() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

void ReleaseControl(Module moduleId) NN_NOEXCEPT
{
    bool releasedAll = true;

    for ( auto& entry : g_ModuleIdList )
    {
        if ( entry.id == moduleId )
        {
            NN_DETAIL_PCV_TRACE("Release clock module %d.\n", moduleId);
            entry.released = true;
        }

        if ( !entry.released )
        {
            releasedAll = false;
        }
    }

    // LoopAuto 内で RequestStop は呼んで良い？
    if ( releasedAll )
    {
        NN_DETAIL_PCV_TRACE("Complete clock arbitration.\n");
        g_pMyServerManager->RequestStop();
    }
}

}}} // namespace nn::pcv::server
