﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_NotificationDataReceiver.h>

#include <nn/npns/npns_Api.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/json/pctl_JsonApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonInputStream.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    struct ParsedData
    {
        ServerDeviceId deviceId;
        NotificationEventId eventId;
        AlarmSettingState alarmSettingState;
        bool isDeviceIdReceived;
    };

    static bool HandleDeviceId(void* param, int, const char* value, size_t length) NN_NOEXCEPT
    {
        auto p = static_cast<ParsedData*>(param);
        if (!HandleIdString(&p->deviceId, value, length))
        {
            return false;
        }
        p->isDeviceIdReceived = true;
        return true;
    }

    static bool HandleEventId(void* param, int, const char* value, size_t length) NN_NOEXCEPT
    {
        auto p = static_cast<ParsedData*>(param);
        auto i = GetIndexFromString(value, length, NotificationEventIdName);
        if (i < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid 'eventId' value: length = %d, value = %*s\n",
                static_cast<int>(length), static_cast<int>(length), value);
            return false;
        }
        p->eventId = static_cast<NotificationEventId>(i);
        return true;
    }

    static bool HandleAlarmSettingsStateData(void* param, int, const char* value, size_t length) NN_NOEXCEPT
    {
        auto p = static_cast<ParsedData*>(param);
        auto i = GetIndexFromString(value, length, AlarmSettingStateName);
        if (i < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid 'alarmSettingState.status' value: length = %d, value = %*s\n",
                static_cast<int>(length), static_cast<int>(length), value);
            return false;
        }
        p->alarmSettingState = static_cast<AlarmSettingState>(i);
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectNotificationDataJsonFormat)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("deviceId") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleDeviceId)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("eventId")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleEventId)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("alarmSettingState") NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("status")   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleAlarmSettingsStateData)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()
}

void NotificationDataReceiver::ProcessReceiveNotificationData(common::NetworkBuffer& bufferInfo, common::Cancelable& cancelable) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    while (NN_STATIC_CONDITION(true))
    {
        {
            auto result = nn::npns::Receive(&m_NotificationData);

            if (nn::npns::ResultNotReceived::Includes(result))
            {
                break;
            }
            if (result.IsFailure())
            {
                NN_DETAIL_PCTL_INFO("[pctl] nn::npns::Receive() failed. code = 2%03d-%04d (result = 0x%08lX)\n",
                    result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                break;
            }
        }

        {
            // dataはJSON形式なのでそれを ParseNotificationData に渡して解析してもらう
            ParseNotificationData(bufferInfo, cancelable,
                m_NotificationData.GetPayload(), m_NotificationData.GetPayloadSize());
        }
    }
#else
    //
    NN_UNUSED(bufferInfo);
    NN_UNUSED(cancelable);
#endif
}

void NotificationDataReceiver::ParseNotificationData(common::NetworkBuffer& bufferInfo, common::Cancelable& cancelable, const char* data, size_t dataSize) NN_NOEXCEPT
{
    if (dataSize == 0)
    {
        return;
    }

    ParsedData parsedData;
    json::JsonMemoryInputStream inputStream;
    json::JsonDataHandler handler(&parsedData, ExpectNotificationDataJsonFormat);
    parsedData.alarmSettingState = AlarmSettingState::Undefined;

    // データ内に deviceId が見つからなかった場合はそのメッセージを無視する
    parsedData.isDeviceIdReceived = false;

    nn::Result result;
    result = inputStream.Open(data, dataSize);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_INFO("[pctl] Failed to prepare for parsing notification data (result = 0x%08lX, json = '%s')\n",
            result.GetInnerValueForDebug(), data);
        return;
    }
    inputStream.SetBuffer(
        reinterpret_cast<json::JsonInputStream::Ch*>(bufferInfo.GetBufferForJsonValue()),
        common::NetworkBuffer::JsonValueBufferMemorySize,
        reinterpret_cast<json::JsonInputStream::Ch*>(bufferInfo.GetBufferForStream()),
        common::NetworkBuffer::StreamBufferMemorySize
    );

    result = json::Parse(&handler, &inputStream, &cancelable);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_INFO("[pctl] Failed to parse notification data (result = 0x%08lX, json = '%s')\n",
            result.GetInnerValueForDebug(), data);
        return;
    }

    if (!parsedData.isDeviceIdReceived)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Received event from NPNS but device id is not found: type = %s\n",
            NotificationEventIdName[static_cast<int>(parsedData.eventId)].value);
        return;
    }
    // 自分向けのメッセージではない場合は無視する
    if (parsedData.deviceId != g_pWatcher->GetNetworkManager().GetSavedDeviceId())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Received event from NPNS but not for me: type = %s, my id = %016llX, data's id = %016llX\n",
            NotificationEventIdName[static_cast<int>(parsedData.eventId)].value,
            g_pWatcher->GetNetworkManager().GetSavedDeviceId(),
            parsedData.deviceId);
        return;
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Received event from NPNS: %s\n",
        NotificationEventIdName[static_cast<int>(parsedData.eventId)].value);
    switch (parsedData.eventId)
    {
        case NotificationEventId::ChangedParentalControlSetting:
            OnEventSettingChanged();
            break;
        case NotificationEventId::Unlinked:
            OnEventUnlinked();
            break;
        case NotificationEventId::OnlineCheck:
            OnEventOnlineCheck();
            break;
        case NotificationEventId::UpdateAlarmSettingState:
            OnEventUpdateAlarmSettingState(parsedData.alarmSettingState);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void NotificationDataReceiver::OnEventSettingChanged() NN_NOEXCEPT
{
    // 通知を受け取って開始する場合は常に1からの試行回数とする(リセットする)
    g_pWatcher->GetWatcherEventManager().RequestRetrieveSettingsBackground(1, true);
}

void NotificationDataReceiver::OnEventUnlinked() NN_NOEXCEPT
{
    // 設定を削除してイベントを発生させる
    g_pWatcher->GetNetworkManager().ClearPairingInfo(true);
    g_pWatcher->GetWatcherEventManager().GetSynchronizationEvent()->Signal();

    // オーバーレイ通知を発行
    auto r = overlay::NotifyUnlink(nn::ovln::format::PctlUnlinkReasonFlag_FromServer);
    if (r.IsFailure())
    {
        NN_DETAIL_PCTL_INFO("[pctl] Failed to send overlay notification for unlink (0x%08lX)\n", r.GetInnerValueForDebug());
    }
}

void NotificationDataReceiver::OnEventOnlineCheck() NN_NOEXCEPT
{
    auto& manager = g_pWatcher->GetNetworkManager();
    manager.RequestOnlineCheckResponseBackground(manager.GetSavedDeviceId());
}

void NotificationDataReceiver::OnEventUpdateAlarmSettingState(AlarmSettingState alarmSettingState) NN_NOEXCEPT
{
    if (!(alarmSettingState == AlarmSettingState::ToVisible || alarmSettingState == AlarmSettingState::ToInvisible))
    {
        NN_DETAIL_PCTL_WARN("[pctl] Unknown alarmSettingState value: enum value = %d\n",
            static_cast<int>(alarmSettingState));
        return;
    }
    auto& manager = g_pWatcher->GetNetworkManager();
    manager.RequestApplyAlarmSettingBackground(alarmSettingState == AlarmSettingState::ToInvisible ? true : false);
}

}}}}}
