﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_DeviceEvent.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_PlayState.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>
#include <nn/pctl/detail/service/watcher/pctl_DeviceUser.h>
#include <nn/pctl/detail/service/watcher/pctl_Notifications.h>
#include <nn/pctl/detail/service/watcher/pctl_UpdateDevice.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_DeviceEventDispatcher.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <algorithm>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{

    ////////////////////////////////////////////////////////////////////////////
    // イベント送信処理

    // 送信可能なイベント数の上限
    static const int MaxEventCountForPost = 200;
    EventData g_TempEventData[MaxEventCountForPost];
}

////////////////////////////////////////////////////////////////////////////////

nn::Result UploadDeviceEventsExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    return ExecuteUploadDeviceEvents(m_pEventCount, m_pEventData, m_EventDataSize,
        UploadMode_Normal, bufferInfo, m_DeviceId, m_pCancelable,
        &m_EtagBuffer, &m_NotificationTokenBuffer);
}

void UploadDeviceEventsExecutor::Cancel() NN_NOEXCEPT
{
    if (m_pCancelable != nullptr)
    {
        m_pCancelable->Cancel();
    }
}

nn::Result UploadDeviceEventsExecutor::ExecuteUploadDeviceEvents(UploadMode mode, common::NetworkBuffer& bufferInfo,
    ServerDeviceId deviceId, common::Cancelable* pCancelable, EtagInfo* etagBuffer,
    nn::npns::NotificationToken* pNotificationTokenBuffer) NN_NOEXCEPT
{
    return ExecuteUploadDeviceEvents(nullptr, nullptr, 0,
        mode, bufferInfo, deviceId, pCancelable, etagBuffer, pNotificationTokenBuffer);
}

nn::Result UploadDeviceEventsExecutor::ExecuteUploadDeviceEvents(size_t* outEventCount, EventData* pOutEventData, size_t eventDataSize,
    UploadMode mode, common::NetworkBuffer& bufferInfo,
    ServerDeviceId deviceId, common::Cancelable* pCancelable, EtagInfo* etagBuffer,
    nn::npns::NotificationToken* pNotificationTokenBuffer) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::os::SdkMutexType, s_MutexForEventData, =NN_OS_SDK_MUTEX_INITIALIZER());
    NN_UTIL_LOCK_GUARD(s_MutexForEventData);

    size_t totalWrittenCount = 0;
    NN_UTIL_SCOPE_EXIT
    {
        if (outEventCount != nullptr)
        {
            *outEventCount = totalWrittenCount;
        }
    };

    if (!g_pWatcher->GetWatcherEventStorage().IsAnyDataRecorded() && g_pWatcher->GetNetworkManager().IsLastUpdateDeviceSuccess())
    {
        // 送信するイベントが無く、デバイス更新も不要の場合はそのまま終了する
        // (通常の定期処理では IDLE イベントが追加される)
        NN_RESULT_SUCCESS;
    }

    // イベント送信、デバイス更新いずれかでも必要であればトークン取得を実行
    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, pCancelable));

    // ユーザー情報を更新
    NN_RESULT_DO(
        UpdateDeviceUserExecutor::UpdateDeviceUsers(bufferInfo, pCancelable, tokenHolder.GetToken(), deviceId)
    );

    while (NN_STATIC_CONDITION(true))
    {
        auto dataCount = g_pWatcher->GetWatcherEventStorage().GetPartialDataForPost(g_TempEventData,
            static_cast<uint16_t>(std::extent<decltype(g_TempEventData)>::value));
        if (dataCount == 0)
        {
            // NOTE: dataCount == 0 でも isUpdateDeviceNecessary == true のケースがある
            break;
        }

#if !defined(NN_SDK_BUILD_RELEASE)
        NN_DETAIL_PCTL_TRACE("[pctl] Post events: Event dump:\n");
        for (uint32_t u = 0; u < static_cast<uint32_t>(dataCount); ++u)
        {
            auto& e = g_TempEventData[u];
            if (e.eventType == EventType::DidUserOpen || e.eventType == EventType::DidUserClose)
            {
                auto& uid = e.payload.openedUser.uid;
                NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld, uid = %08x_%08x_%08x_%08x\n",
                    u, static_cast<int>(e.eventType), e.timestamp.value,
                    static_cast<uint32_t>(uid._data[0] >> 32),
                    static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                    static_cast<uint32_t>(uid._data[1] >> 32),
                    static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
            }
            else if (e.eventType == EventType::DidApplicationLaunch || e.eventType == EventType::DidApplicationSuspend ||
                e.eventType == EventType::DidAddNewManagedApplication || e.eventType == EventType::DidRemoveManagedApplication ||
                e.eventType == EventType::DidApplicationPlay ||
                e.eventType == EventType::DidApplicationResume || e.eventType == EventType::DidApplicationTerminate ||
                e.eventType == EventType::DidApplicationDownloadStart || e.eventType == EventType::DidLimitedApplicationLaunch)
            {
                NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld, appId = 0x%016llX\n",
                    u, static_cast<int>(e.eventType), e.timestamp.value,
                    e.payload.applicationLaunch.applicationId.value);
            }
            else
            {
                NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld\n",
                    u, static_cast<int>(e.eventType), e.timestamp.value);
            }
        }
#endif

        uint16_t writtenCount = 0;
        auto result = dispatcher::UploadDeviceEvents::Execute(&writtenCount, bufferInfo, pCancelable,
            tokenHolder.GetToken(), deviceId, g_TempEventData, dataCount);

        if (pOutEventData != nullptr && eventDataSize > 0)
        {
            if (totalWrittenCount < eventDataSize)
            {
                auto copyCount = eventDataSize > totalWrittenCount + writtenCount ? writtenCount : eventDataSize - totalWrittenCount;
                std::memcpy(&pOutEventData[totalWrittenCount], g_TempEventData, sizeof(EventData) * copyCount);
            }
        }
        totalWrittenCount += writtenCount;

        g_pWatcher->GetWatcherEventStorage().FinishPostPartialData(writtenCount);

        NN_RESULT_DO(result);

        // 送信を行った状態(writtenCount > 0)でまだイベントが残っている場合は
        // 通常の定期処理よりも早いタイミングで次を送るようにする
        bool needsRepeat = false;
        if (writtenCount > 0)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Posted events: count = %d\n", static_cast<int>(writtenCount));
            if (g_pWatcher->GetWatcherEventStorage().IsAnyDataRecorded())
            {
                switch (mode)
                {
                    case UploadMode_PostAllImmediately:
                        // 全イベントをPostする場合はループを行う
                        // (token の取り直しは不要のはず)
                        NN_DETAIL_PCTL_TRACE("[pctl] Loop for next post...\n");
                        needsRepeat = true;
                        break;
                    case UploadMode_Normal:
                    default:
                        // 通常処理の場合は一旦処理を抜けて改めて(一定時間後に)実行できるようにトリガーする
                        NN_DETAIL_PCTL_TRACE("[pctl] call RequestPostRemainingEvents\n");
                        g_pWatcher->GetWatcherEventManager().RequestPostRemainingEvents();
                        break;
                }
            }
            else
            {
                NN_DETAIL_PCTL_TRACE("[pctl] All events have been posted (by pctl).\n");
            }
        }
        if (!needsRepeat)
        {
            break;
        }
    }

    // 更新が必要であれば実行
    //
    if (!g_pWatcher->GetNetworkManager().IsLastUpdateDeviceSuccess())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Start post update device (after post events).\n");
        g_pWatcher->GetNetworkManager().GetEtagForSettings(etagBuffer);
        NN_RESULT_DO(
            UpdateDeviceExecutor::ExecuteUpdateDevice(bufferInfo, pCancelable, tokenHolder, deviceId, etagBuffer, pNotificationTokenBuffer)
        );
    }

    NN_RESULT_SUCCESS;
}// NOLINT(impl/function_size)

}}}}}
