﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/pctl_PlayState.h>

#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/watcher/pctl_EventLog.h>

#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace pctl { namespace detail { namespace service {

namespace
{
    // プレイ状態を保持するデータ
    PlayState g_PlayState = {};
    nn::os::SdkMutexType g_PlayStateMutex = NN_OS_SDK_MUTEX_INITIALIZER();

    // 計算用の一時的なプレイ状態を保持するデータ
    PlayState g_TempPlayState;
}

PlayStateHolder PlayStateHolder::Acquire() NN_NOEXCEPT
{
    return PlayStateHolder(&g_PlayState, &g_PlayStateMutex);
}

PlayState* GetTempPlayStateData() NN_NOEXCEPT
{
    return &g_TempPlayState;
}


void PlayState::ResetPlayedStatus(const nn::time::PosixTime& timeNow) NN_NOEXCEPT
{
    // 一旦開始時刻を現在時刻にする(アプリケーション実行中の場合は効果無し)
    this->miscTimeStart = timeNow;
    this->miscElapsedTime = nn::TimeSpan(0);
    // 無視される期間も同様に設定(スリープや一時解除が起きていない場合は効果無し)
    this->ignoreTimeStart = timeNow;
    this->ignoreElapsedTime = nn::TimeSpan(0);
    this->temporaryUnlockedTimeStart = timeNow;
    this->temporaryUnlockedElapsedTime = nn::TimeSpan(0);
    this->playTimerStoppedTime = timeNow;
    this->lastElapsedTimeSeconds = 0;

    for (auto& d : data)
    {
        // 動作していないアプリケーションをクリア
        {
            auto p = PlayedApplicationState::FromDataBlock(&d);
            if (p != nullptr && p->IsValidData())
            {
                if (p->IsRunning())
                {
                    p->ResetStartTime(timeNow);
                }
                else
                {
                    // ユーザー別の状態情報も初期化する
                    for (auto& d2 : data)
                    {
                        auto pUserData = PlayedApplicationUserState::FromDataBlock(&d2);
                        if (pUserData != nullptr && pUserData->IsEqualApplicationId(p->applicationId))
                        {
                            pUserData->InvalidateData();
                        }
                    }
                    p->InvalidateData();
                }
            }
        }
        // Open されていないユーザー情報をクリア
        {
            auto p = PlayedApplicationUserState::FromDataBlock(&d);
            if (p != nullptr && p->IsValidData())
            {
                if (p->IsOpened())
                {
                    p->ResetStartTime(timeNow);
                }
                else
                {
                    p->InvalidateData();
                }
            }
        }
    }
}

void PlayState::InitializeData() NN_NOEXCEPT
{
    NN_STATIC_ASSERT(std::is_pod<PlayState>::value);
    std::memset(this, 0, sizeof(PlayState));
}

const PlayedApplicationState* PlayState::FindApplicationStateData(const nn::ncm::ApplicationId& applicationId) const NN_NOEXCEPT
{
    for (auto& d : data)
    {
        auto p = PlayedApplicationState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualApplicationId(applicationId))
            {
                return p;
            }
        }
    }
    return nullptr;
}

PlayedApplicationState* PlayState::FindApplicationStateDataForRegister(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        auto p = PlayedApplicationState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualApplicationId(applicationId))
            {
                return p;
            }
        }
    }
    for (auto& d : data)
    {
        if (!d.IsAnyDataStored())
        {
            // 未記録のブロックは初期化して返す
            return PlayedApplicationState::InitializeData(&d);
        }
    }
    // 記録する上限に達しているので nullptr を返す
    return nullptr;
}

const PlayedApplicationUserState* PlayState::FindApplicationUserGlobalStateData(const nn::account::Uid& uid) const NN_NOEXCEPT
{
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualUid(uid) && p->IsGlobalStatusData())
            {
                return p;
            }
        }
    }
    return nullptr;
}

PlayedApplicationUserState* PlayState::FindApplicationUserGlobalStateDataForRegister(const nn::account::Uid& uid) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualUid(uid) && p->IsGlobalStatusData())
            {
                return p;
            }
        }
    }
    for (auto& d : data)
    {
        if (!d.IsAnyDataStored())
        {
            // 未記録のブロックは初期化して返す
            auto p = PlayedApplicationUserState::InitializeData(&d);
            NN_SDK_ASSERT_EQUAL(p->IsGlobalStatusData(), true);
            return p;
        }
    }
    // 記録する上限に達しているので nullptr を返す
    return nullptr;
}

PlayedApplicationUserState* PlayState::FindApplicationUserStateDataForRegister(const nn::account::Uid& uid, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualUid(uid) && p->IsEqualApplicationId(applicationId))
            {
                return p;
            }
        }
    }
    for (auto& d : data)
    {
        if (!d.IsAnyDataStored())
        {
            // 未記録のブロックは初期化して返す
            auto p = PlayedApplicationUserState::InitializeData(&d);
            p->applicationId = applicationId;
            return p;
        }
    }
    // 記録する上限に達しているので nullptr を返す
    return nullptr;
}

void PlayState::SetApplicationStarted(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeStarted) NN_NOEXCEPT
{
    // 何らかの理由で起動していない状態である場合を考慮しセットする
    // (イベント情報が途中からである場合に発生)
    SetDeviceRunning(true);

    auto pState = FindApplicationStateDataForRegister(applicationId);
    if (pState == nullptr)
    {
        return;
    }

    nn::time::PosixTime timeActual = timeStarted;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)アプリケーション開始時刻として扱う
        timeActual = ignoreTimeStart;
    }

    // 何かしらアプリケーションが開始するので、現時点で
    // アプリケーションが実行中でなければ「その他時間」を終了させる
    if (!IsAnyApplicationRunning())
    {
        SetMiscTimeFinished(timeActual);
    }

    // 実行中に設定
    pState->SetStarted(applicationId, timeActual);

    // グローバルなユーザーの状態情報を探し、Open 状態であればアプリケーションごとの状態情報を増やす
    bool isAnyUserOpened = false;
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr && p->IsGlobalStatusData() && p->IsOpened())
        {
            auto pAppUser = FindApplicationUserStateDataForRegister(p->uid, applicationId);
            if (pAppUser != nullptr)
            {
                NN_SDK_ASSERT_EQUAL(pAppUser->applicationId, applicationId);
                // Open 時刻は開始時刻とする
                pAppUser->SetOpened(p->uid, timeActual);
                isAnyUserOpened = true;
            }
        }
    }
    // ユーザーが選択されていればその旨を設定
    if (isAnyUserOpened)
    {
        pState->SetNonAnonymousState(timeActual);
    }
}

void PlayState::SetApplicationStopped(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeStopped) NN_NOEXCEPT
{
    auto pState = FindApplicationStateData(applicationId);
    if (pState == nullptr)
    {
        return;
    }

    auto isAnyApplicationRunningPreviously = IsAnyApplicationRunning();

    nn::time::PosixTime timeActual = timeStopped;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)終了時刻として扱う
        timeActual = ignoreTimeStart;
    }
    pState->SetStopped(timeActual);

    // アプリケーションが1つも実行中でなければ「その他時間」を開始させる
    if (isAnyApplicationRunningPreviously && !IsAnyApplicationRunning())
    {
        SetMiscTimeStarted(timeActual);
    }

    // Open 状態である、applicationId に一致するアプリケーションごとの
    // ユーザー状態情報をすべて探し、それらを Close 状態に書き換える
    // (グローバルな状態はそのまま)
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr && p->IsOpened() && p->IsEqualApplicationId(applicationId))
        {
            p->SetClosed(timeActual);
        }
    }
}

void PlayState::SetApplicationSuspended(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeSuspended) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeSuspended;
    // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)中断時刻として扱う
    if (IsIgnorableTime())
    {
        timeActual = this->ignoreTimeStart;
    }
    bool isAnyApplicationRunning = false;
    bool isStateChanged = false;
    for (auto& d : data)
    {
        // アプリケーションとユーザー両方の状態を更新
        {
            auto pAppState = PlayedApplicationState::FromDataBlock(&d);
            if (pAppState != nullptr)
            {
                if (pAppState->IsEqualApplicationId(applicationId))
                {
                    auto changed = pAppState->SetSuspended(timeActual);
                    isStateChanged = changed || isStateChanged;
                }
                else
                {
                    isAnyApplicationRunning = isAnyApplicationRunning ||
                        (pAppState->IsRunning() && !pAppState->IsSuspended());
                }
            }
        }
        {
            auto pUserState = PlayedApplicationUserState::FromDataBlock(&d);
            if (pUserState != nullptr && pUserState->IsEqualApplicationId(applicationId))
            {
                pUserState->SetSuspended(timeActual);
            }
        }
    }

    // アプリケーション実行中が無くなった場合はその他時間を開始させる
    if (isStateChanged && !isAnyApplicationRunning)
    {
        SetMiscTimeStarted(timeActual);
    }
}

void PlayState::SetApplicationResumed(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeResumed) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeResumed;
    // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)再開時刻として扱う
    if (IsIgnorableTime())
    {
        timeActual = this->ignoreTimeStart;
    }
    bool isAnyApplicationRunningPreviously = false;
    bool isStateChanged = false;
    for (auto& d : data)
    {
        // アプリケーションとユーザー両方の状態を更新
        {
            auto pAppState = PlayedApplicationState::FromDataBlock(&d);
            if (pAppState != nullptr)
            {
                isAnyApplicationRunningPreviously = isAnyApplicationRunningPreviously ||
                    (pAppState->IsRunning() && !pAppState->IsSuspended());
                if (pAppState->IsEqualApplicationId(applicationId))
                {
                    auto changed = pAppState->SetResumed(timeActual);
                    isStateChanged = changed || isStateChanged;
                }
            }
        }
        {
            auto pUserState = PlayedApplicationUserState::FromDataBlock(&d);
            if (pUserState != nullptr && pUserState->IsEqualApplicationId(applicationId))
            {
                pUserState->SetResumed(timeActual);
            }
        }
    }

    // アプリケーション実行中がもともと無く実行が再開された場合はその他時間を終了させる
    if (isStateChanged && !isAnyApplicationRunningPreviously)
    {
        SetMiscTimeFinished(timeActual);
    }
}

void PlayState::SetAllApplicationStopped(const nn::time::PosixTime& timeStopped) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeStopped;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)アプリケーション停止時刻として扱う
        timeActual = ignoreTimeStart;
    }
    bool isAnyApplicationRunningPreviously = false;
    for (auto& d : data)
    {
        {
            auto p = PlayedApplicationState::FromDataBlock(&d);
            if (p != nullptr && p->IsRunning())
            {
                isAnyApplicationRunningPreviously = isAnyApplicationRunningPreviously ||
                    !p->IsSuspended();
                p->SetStopped(timeActual);
            }
        }
        // Open 状態であるアプリケーションごとのユーザー状態情報をすべて探し、
        // それらを Close 状態に書き換える
        // (グローバルな状態はそのまま)
        {
            auto p = PlayedApplicationUserState::FromDataBlock(&d);
            if (p != nullptr && p->IsOpened() && !p->IsGlobalStatusData())
            {
                p->SetClosed(timeActual);
            }
        }
    }

    // 実行中のアプリケーションがあった場合、すべて終了したので「その他時間」を開始させる
    if (isAnyApplicationRunningPreviously)
    {
        SetMiscTimeStarted(timeActual);
    }
}

void PlayState::SetPlayTimerEnabled(bool enabled, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    if (enabled == IsPlayTimerEnabled())
    {
        return;
    }

    SetFlagForIgnorableTime<PlayStateFlags::PlayTimerEnabled, false>(enabled, timeAt);

    if (enabled)
    {
        if (IsTemporaryUnlocked())
        {
            // 一時解除中の時間を調整
            auto elapsed = timeAt - playTimerStoppedTime;
            temporaryUnlockedTimeStart += elapsed;
        }
    }
    // 開始/停止状態にかかわらず停止時刻を timeAt に揃える
    // (以前の WatcherEventManager::OnTimerRunningStatusChanged の実装に揃える)
    playTimerStoppedTime = timeAt;
}

void PlayState::SetSleeping(bool sleeping, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    if (sleeping == IsSleeping())
    {
        return;
    }
    NN_SDK_REQUIRES(!IsTemporaryUnlocked());

    SetFlagForIgnorableTime<PlayStateFlags::Sleeping, true>(sleeping, timeAt);

    if (sleeping)
    {
        // スリープ時はプレイタイマー自動停止
        SetPlayTimerEnabled(false, timeAt);
        // スリープに入るので、アプリケーションが実行中であった場合は
        // 「アプリケーションが実行していない/中断している期間」にセットしておく
        if (IsAnyApplicationRunning())
        {
            SetMiscTimeStarted(timeAt);
        }
    }
    else
    {
        SetDeviceRunning(true);
        // スリープから抜けているので、アプリケーションが実行中である場合は
        // 「アプリケーションが実行していない/中断している期間」を終了させる
        if (IsAnyApplicationRunning())
        {
            SetMiscTimeFinished(timeAt);
        }
    }
}

void PlayState::SetTemporaryUnlocked(bool temporaryUnlocked, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    if (temporaryUnlocked == IsTemporaryUnlocked())
    {
        return;
    }
    NN_SDK_REQUIRES(!IsSleeping());

    SetFlagForIgnorableTime<PlayStateFlags::TemporaryUnlocked, true>(temporaryUnlocked, timeAt);

    // プレイタイマー停止中は便宜上停止した時刻を計算に用いる
    auto timeForCalculate = (IsPlayTimerEnabled() ? timeAt : playTimerStoppedTime);
    if (temporaryUnlocked)
    {
        temporaryUnlockedTimeStart = timeForCalculate;
    }
    else
    {
        // これまでの無効時間を計上
        temporaryUnlockedElapsedTime += (timeForCalculate - temporaryUnlockedTimeStart);
    }
}

void PlayState::ProcessDeviceRunningStatus(bool isRunning, const nn::time::PosixTime& timeAt, bool isDayChanged) NN_NOEXCEPT
{
    bool lastIsIgnorableTime = IsIgnorableTime();

    nn::time::PosixTime stopTime = timeAt;

    // 直前まで電源ON状態(true == IsDeviceRunning())であった＆時間計測を無視しない状態であった
    // → イベント送信間隔である最大10分(watcher::MaxElapsedTimeSpanOnUnexpectedShutdown)までは時間経過・その時刻で止まったこととする
    if (IsDeviceRunning() && isRunning && !lastIsIgnorableTime)
    {
        auto span = timeAt - this->lastTimestamp;
        const auto maxTimeSpan = nn::TimeSpan::FromSeconds(watcher::MaxElapsedTimeSpanOnUnexpectedShutdown);
        NN_DETAIL_PCTL_TRACE("[pctl] ProcessDeviceRunningStatus: Check time: %lld vs. %lld\n",
            span.GetSeconds(), maxTimeSpan.GetSeconds());
        if (span > maxTimeSpan)
        {
            stopTime = this->lastTimestamp + maxTimeSpan;
            span = maxTimeSpan;
        }
        // elapsedTime もこれに合わせる
        this->lastElapsedTimeSeconds += span.GetSeconds();
    }

    // デバイスの起動・停止にかかわらず全部停止状態にする
    // (起動時も強制電源断などで状態が残っている可能性があるため)
    SetAllApplicationStopped(stopTime);
    SetAllUserClosed(stopTime);
    SetSleeping(false, stopTime); // 電源断の期間≠スリープ中 なので timeAt ではなく stopTime を用いる
    SetTemporaryUnlocked(false, stopTime);

    // miscTime はデバイスが起動中であった場合のみ一旦停止する
    if (IsDeviceRunning())
    {
        SetMiscTimeFinished(stopTime);
    }
    // どちらの場合もプレイタイマーは停止状態とする
    // (起動時は停止状態から開始)
    SetPlayTimerEnabled(false, stopTime);
    playTimerStoppedTime = stopTime; // (状態変化が無い場合でも時刻は上書きする)

    if (isDayChanged)
    {
        // 日付が変わった場合はすべてリセットする。
        // 上記の SetAllApplicationStopped 呼び出しによりすべてのアプリケーションを停止状態にしているので
        // すべての状態がリセット対象になる
        // (前回のデバイス起動状態にかかわらずすべてリセットできる)
        ResetPlayedStatus(timeAt);
    }

    SetDeviceRunning(isRunning);

    if (isRunning)
    {
        // その他時間の開始時刻に設定
        SetMiscTimeStarted(timeAt);
    }
}

void PlayState::SetUserOpened(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeAt;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)Open 時刻として扱う
        timeActual = ignoreTimeStart;
    }
    // グローバルな状態の更新
    {
        auto p = FindApplicationUserGlobalStateDataForRegister(uid);
        if (p != nullptr)
        {
            p->SetOpened(uid, timeActual);
        }
        else
        {
            NN_DETAIL_PCTL_TRACE("[pctl] No more data block space for global user data (uid: %08x_%08x_%08x_%08x)\n",
                static_cast<uint32_t>(uid._data[0] >> 32),
                static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(uid._data[1] >> 32),
                static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
        }
    }
    // 実行中のアプリケーションがあればユーザー選択状態へ移行させ、それらに対応する記録も作る
    for (auto& d : data)
    {
        auto p = PlayedApplicationState::FromDataBlock(&d);
        if (p != nullptr && p->IsRunning())
        {
            // ユーザー選択状態へ移行
            if (p->IsAnonymous())
            {
                p->SetNonAnonymousState(timeActual);
            }
            auto pUser = FindApplicationUserStateDataForRegister(uid, p->applicationId);
            if (pUser != nullptr)
            {
                NN_SDK_ASSERT_EQUAL(pUser->IsGlobalStatusData(), false);
                pUser->SetOpened(uid, timeActual);
                // アプリケーションがSuspendである場合は合わせる
                if (p->IsSuspended())
                {
                    pUser->SetSuspended(timeActual);
                }
            }
            else
            {
                NN_DETAIL_PCTL_TRACE("[pctl] No more data block space for application user data (uid: %08x_%08x_%08x_%08x)\n",
                    static_cast<uint32_t>(uid._data[0] >> 32),
                    static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                    static_cast<uint32_t>(uid._data[1] >> 32),
                    static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
            }
        }
    }
}

void PlayState::SetUserClosed(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeAt;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)Close 時刻として扱う
        timeActual = ignoreTimeStart;
    }

    bool isAnyUserOpened = false;
    // 全 User の情報を更新
    for (auto& d : data)
    {
        auto p = PlayedApplicationUserState::FromDataBlock(&d);
        if (p != nullptr)
        {
            if (p->IsEqualUid(uid))
            {
                p->SetClosed(timeActual);
            }
            else if (p->IsGlobalStatusData()) // (グローバルな状態のみ見れば問題ないはず)
            {
                isAnyUserOpened = isAnyUserOpened || p->IsOpened();
            }
        }
    }
    // 全ユーザーが非 Open 状態になっているのでユーザー非選択状態に移行させる
    if (!isAnyUserOpened)
    {
        for (auto& d : data)
        {
            auto p = PlayedApplicationState::FromDataBlock(&d);
            if (p != nullptr && p->IsRunning() && !p->IsAnonymous())
            {
                p->SetAnonymousState(timeActual);
            }
        }
    }
}

void PlayState::SetAllUserClosed(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    nn::time::PosixTime timeActual = timeAt;
    if (IsIgnorableTime())
    {
        // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)Close 時刻として扱う
        timeActual = ignoreTimeStart;
    }

    for (auto& d : data)
    {
        // 全 User の情報を更新
        {
            auto p = PlayedApplicationUserState::FromDataBlock(&d);
            if (p != nullptr)
            {
                p->SetClosed(timeActual);
            }
        }
        // ユーザー非選択状態に設定
        {
            auto p = PlayedApplicationState::FromDataBlock(&d);
            if (p != nullptr && p->IsRunning() && !p->IsAnonymous())
            {
                p->SetAnonymousState(timeActual);
            }
        }
    }
}

void PlayState::SetMiscTimeStarted(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    // 無視が必要な時間帯の中にいる場合はその開始時刻を(仮想的な)開始時刻として扱う
    auto timeStart = (IsIgnorableTime()) ? ignoreTimeStart : timeAt;
    this->miscTimeStart = timeStart;
}

void PlayState::SetMiscTimeFinished(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    auto timeEnd = (IsIgnorableTime()) ? ignoreTimeStart : timeAt;
    auto elapsedTime = timeEnd - this->miscTimeStart;
    if (elapsedTime < nn::TimeSpan(0))
    {
        elapsedTime = nn::TimeSpan(0);
    }
    this->miscElapsedTime += elapsedTime;
}

bool PlayState::IsAnyApplicationRunning() const NN_NOEXCEPT
{
    for (auto& d : data)
    {
        // アプリケーションとユーザー両方の状態を更新
        auto pAppState = PlayedApplicationState::FromDataBlock(&d);
        if (pAppState != nullptr && pAppState->IsRunning() && !pAppState->IsSuspended())
        {
            return true;
        }
    }
    return false;
}

void PlayState::AdjustIgnoreTime(const nn::time::PosixTime& ignoreStartTime, const nn::time::PosixTime& ignoreEndTime) NN_NOEXCEPT
{
    if (ignoreStartTime < this->miscTimeStart)
    {
        // 開始前から除外が始まっていた場合は除外区間終了時刻を開始時刻に設定
        this->miscTimeStart = ignoreEndTime;
    }
    else
    {
        // 除外する時間分だけ開始時刻を調整
        this->miscTimeStart += ignoreEndTime - ignoreStartTime;
    }

    for (auto& d : data)
    {
        {
            auto p = PlayedApplicationState::FromDataBlock(&d);
            if (p != nullptr && p->IsValidData() && p->IsRunning())
            {
                p->AdjustIgnoreTime(ignoreStartTime, ignoreEndTime);
            }
        }
        {
            auto p = PlayedApplicationUserState::FromDataBlock(&d);
            if (p != nullptr && p->IsValidData() && p->IsOpened())
            {
                p->AdjustIgnoreTime(ignoreStartTime, ignoreEndTime);
            }
        }
    }
}

nn::TimeSpan PlayState::GetApplicationRunningTime(const PlayedApplicationState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT
{
    // 無効期間中はそれを開始した時刻を指定することで、それまでの実行時間が得られる
    if (IsIgnorableTime())
    {
        return state.GetRunningTime(ignoreTimeStart);
    }
    else
    {
        return state.GetRunningTime(timeAt);
    }
}

nn::TimeSpan PlayState::GetApplicationAnonymousPlayingTime(const PlayedApplicationState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT
{
    // 一時解除中は一時解除を開始した時刻を指定することで、それまでの実行時間が得られる
    if (IsIgnorableTime())
    {
        return state.GetAnonymousPlayingTime(ignoreTimeStart);
    }
    else
    {
        return state.GetAnonymousPlayingTime(timeAt);
    }
}

nn::TimeSpan PlayState::GetUserPlayingTime(const PlayedApplicationUserState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT
{
    // 一時解除中は一時解除を開始した時刻を指定することで、それまでの実行時間が得られる
    if (IsIgnorableTime())
    {
        return state.GetPlayingTime(ignoreTimeStart);
    }
    else
    {
        return state.GetPlayingTime(timeAt);
    }
}


// イベントデータをプレイ状態に反映
bool PlayState::UpdatePlayStateFromEvent(const watcher::EventData& eventData) NN_NOEXCEPT
{
    // プレイ状態計算が考慮されたバージョン未満の場合は取り扱わない
    if (eventData.dataVersion < watcher::EventDataVersionPlayStateSupported)
    {
        return false;
    }

    // Load の頻度が多くならないように前回読み込み時のデータをキープする
    // (LocationName は基本的にほとんど変わらないはずのため)
    NN_FUNCTION_LOCAL_STATIC(nn::time::TimeZoneRule, g_Rule);
    NN_FUNCTION_LOCAL_STATIC(nn::time::LocationName, g_LastTimeZoneName, ={});
    nn::Result result;
    if (g_LastTimeZoneName != eventData.timezoneName)
    {
        result = nn::time::LoadTimeZoneRule(&g_Rule, eventData.timezoneName);
        if (result.IsFailure())
        {
            // タイムゾーン情報が取得できなかった場合は何もしない
            NN_DETAIL_PCTL_WARN("Failed to load time-zone rule for event: 0x%08lX\n", result.GetInnerValueForDebug());
            return false;
        }
        g_LastTimeZoneName = eventData.timezoneName;
    }

    nn::time::CalendarTime calTime;
    result = nn::time::ToCalendarTime(&calTime, nullptr, eventData.timestamp, g_Rule);
    NN_SDK_ASSERT(result.IsSuccess(), "Failed to convert to calendar time: result = 0x%08lX", result.GetInnerValueForDebug());

    // DidDeviceActivate は連携完了直後の最初のイベントになるので特殊処理
    if (eventData.eventType == watcher::EventType::DidDeviceActivate)
    {
        // 一旦すべてを初期化する
        this->InitializeData();
        this->ProcessDeviceRunningStatus(true, eventData.timestamp, false);

        this->lastDate = calTime;
        this->lastTimestamp = eventData.timestamp;
        this->lastElapsedTimeSeconds = 0; // 初期状態は 0
        return true;
    }

    bool isDayChanged = false;

    // 地域名変更イベントのときはそのイベントの時刻を日付変更の時刻とみなしてリフレッシュする
    // NOTE: 地域名変更→DidLocationNameChange の間にイベント生成が行われる可能性はあるが、
    //       その期間のイベントは不定なデータとする
    //       (実際には日付が変わっている場合は 0 になり、変わっていない場合は
    //       eventData.elapsedTimeSeconds < (その他の時刻データ) になるので
    //       eventData.elapsedTimeSeconds と同じ値として pctl_DeviceEvent.cpp で扱われる)
    if (eventData.eventType == watcher::EventType::DidLocationNameChange)
    {
        isDayChanged = true;
        this->ResetPlayedStatus(eventData.timestamp);
    }
    // 上記以外で日付部分が変わっているときは実行時間等をリフレッシュする
    else if (this->lastDate.year != calTime.year ||
        this->lastDate.month != calTime.month ||
        this->lastDate.day != calTime.day)
    {
        // 0時ちょうどの時刻を計算
        nn::time::PosixTime timeDayChanged;
        // (この式は不適切: 夏時間等が考慮できていない)
        //nn::time::PosixTime timeDayChanged = {
        //    eventData.timestamp.value - (
        //        static_cast<int64_t>(calTime.hour) * 60 * 60 +
        //        static_cast<int64_t>(calTime.minute) * 60 +
        //        static_cast<int64_t>(calTime.second)
        //        )
        //};
        calTime.hour = 0;
        calTime.minute = 0;
        calTime.second = 0;
        int c = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToPosixTime(&c, &timeDayChanged, 1, calTime, g_Rule));
        if (c == 0)
        {
            // 取得できない場合は「lastTimestamp + (ToPosixTimeFromUtc(calTime) - ToPosixTimeFromUtc(lastDate))」で計算
            // (eventData.timestamp から差し引くのではなく lastTimestamp からの差分を取る)
            timeDayChanged = this->lastTimestamp +
                (nn::time::ToPosixTimeFromUtc(calTime) - nn::time::ToPosixTimeFromUtc(this->lastDate));
        }
        isDayChanged = true;
        this->ResetPlayedStatus(timeDayChanged);
        // 「最後のイベント時刻」を日付変更のタイミングにする
        this->lastTimestamp = timeDayChanged;
        this->lastDate = calTime;
    }

    // 状態が変わる前に(それまでの)プレイ時間を計算
    // ※ デバイス起動イベントの場合はここでは計算しない
    if (eventData.eventType != watcher::EventType::DidDeviceLaunch)
    {
        UpdateElapsedTime(eventData.timestamp);
    }

    // イベントの種類によってアプリケーションの稼働状態を更新する
    switch (eventData.eventType)
    {
        case watcher::EventType::DidApplicationLaunch:
            this->SetApplicationStarted(eventData.payload.applicationLaunch.applicationId, eventData.timestamp);
            break;
        case watcher::EventType::DidApplicationTerminate:
            this->SetApplicationStopped(eventData.payload.applicationTerminate.applicationId, eventData.timestamp);
            break;
        case watcher::EventType::DidApplicationSuspend:
            this->SetApplicationSuspended(eventData.payload.applicationLaunch.applicationId, eventData.timestamp);
            break;
        case watcher::EventType::DidApplicationResume:
            this->SetApplicationResumed(eventData.payload.applicationLaunch.applicationId, eventData.timestamp);
            break;
        case watcher::EventType::DidUserOpen:
            this->SetUserOpened(eventData.payload.openedUser.uid, eventData.timestamp);
            break;
        case watcher::EventType::DidUserClose:
            this->SetUserClosed(eventData.payload.closedUser.uid, eventData.timestamp);
            break;
        case watcher::EventType::DidUnlock:
            this->SetTemporaryUnlocked(true, eventData.timestamp);
            break;
        case watcher::EventType::DidLock:
            this->SetTemporaryUnlocked(false, eventData.timestamp);
            break;
        case watcher::EventType::DidSleep:
            // DidLock の処理も含める
            this->SetTemporaryUnlocked(false, eventData.timestamp);
            this->SetSleeping(true, eventData.timestamp);
            break;
        case watcher::EventType::DidWakeup:
            this->SetSleeping(false, eventData.timestamp);
            break;
        case watcher::EventType::DidDeviceLaunch:
            this->ProcessDeviceRunningStatus(true, eventData.timestamp, isDayChanged);
            break;
        case watcher::EventType::DidUnexpectedShutdownOccur:
            NN_FALL_THROUGH;
        case watcher::EventType::DidShutdown:
            this->ProcessDeviceRunningStatus(false, eventData.timestamp, isDayChanged);
            break;
        case watcher::EventType::DidPlayTimerStart:
            this->SetPlayTimerEnabled(true, eventData.timestamp);
            break;
        case watcher::EventType::DidPlayTimerStop:
            this->SetPlayTimerEnabled(false, eventData.timestamp);
            break;
        case watcher::EventType::DidDeviceActivate:
            NN_SDK_ASSERT(false, "Unreachable");
            break;
        default:
            // 他のイベントは「稼働状態」には影響しない
            break;
    }

    // 日付更新
    this->lastDate = calTime;
    this->lastTimestamp = eventData.timestamp;

    return true;
} // NOLINT(impl/function_size)

bool PlayState::UpdateElapsedTime(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    if (IsIgnorableTime())
    {
        return false;
    }
    // MEMO: Sys 4.x.x のセーブデータにおいても lastElapsedTimeSeconds は有効な値をもっている
    // (プレイタイマー処理で計算された値が常に入っている)
    this->lastElapsedTimeSeconds += (timeAt - this->lastTimestamp).GetSeconds();
    return true;
}

}}}}
