﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/json/pctl_Json.h>
#include <nn/pctl/detail/service/json/pctl_JsonOutputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <limits>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

namespace
{
    // @brief Putの数をカウントする出力ストリームです。
    class LengthCalculatorOutputStream
    {
    public:
        typedef char Ch;

        LengthCalculatorOutputStream() NN_NOEXCEPT :
            m_Length(0)
        {
        }

        void Put(Ch ) NN_NOEXCEPT
        {
            ++m_Length;
        }

        void Flush() NN_NOEXCEPT
        {
        }

        size_t GetLength() const NN_NOEXCEPT
        {
            return m_Length;
        }

    private:
        size_t m_Length;
    };

    static const size_t LengthForNullValue = 4;

    typedef RAPIDJSON_NAMESPACE::Writer<LengthCalculatorOutputStream, RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, json::FixedSizeAllocator>
        LengthCalculatorWriter;
    typedef RAPIDJSON_NAMESPACE::Writer<JsonMemoryOutputStream, RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, json::FixedSizeAllocator>
        MemoryOutputStreamWriter;

    template <typename T, typename Writer>
    class WriteValueCaller
    {
    public:
        static bool Write(Writer& writer, T value);
    };

    template <typename Writer> class WriteValueCaller<int64_t, Writer>
    {
    public:
        static inline bool Write(Writer& writer, int64_t value) { return writer.Int64(value); }
    };
    template <typename Writer> class WriteValueCaller<uint64_t, Writer>
    {
    public:
        static inline bool Write(Writer& writer, uint64_t value) { return writer.Uint64(value); }
    };
    template <typename Writer> class WriteValueCaller<double, Writer>
    {
    public:
        static inline bool Write(Writer& writer, double value) { return writer.Double(value); }
    };
    template <typename Writer> class WriteValueCaller<bool, Writer>
    {
    public:
        static inline bool Write(Writer& writer, bool value) { return writer.Bool(value); }
    };
    template <typename Writer> class WriteValueCaller<const char*, Writer>
    {
    public:
        static inline bool Write(Writer& writer, const char* value, size_t length) { return writer.String(value, static_cast<RAPIDJSON_NAMESPACE::SizeType>(length), false); }
    };

    template <typename T>
    bool GetValueLength(size_t* outLength, T value, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        LengthCalculatorOutputStream os;
        LengthCalculatorWriter writer(os, pAllocator, 16);
        if (!WriteValueCaller<T, LengthCalculatorWriter>::Write(writer, value))
        {
            return false;
        }
        *outLength = os.GetLength() + (isCommaNecessary ? 1 : 0);
        return true;
    }
    bool GetValueLength(size_t* outLength, const char* value, size_t length, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        LengthCalculatorOutputStream os;
        LengthCalculatorWriter writer(os, pAllocator, 16);
        if (!WriteValueCaller<const char*, LengthCalculatorWriter>::Write(writer, value, length))
        {
            return false;
        }
        *outLength = os.GetLength() + (isCommaNecessary ? 1 : 0);
        return true;
    }
    bool GetKeyLength(size_t* outLength, const char* key, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        LengthCalculatorOutputStream os;
        LengthCalculatorWriter writer(os, pAllocator, 16);
        if (!writer.Key(key))
        {
            return false;
        }
        *outLength = os.GetLength() + (isCommaNecessary ? 2 : 1); // including ':' character
        return true;
    }
    bool GetObjectBeginLength(size_t* outLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        // RapidJSON の StartObject を使う代わりに自前でデータを出力
        // (StartObject で RapidJSON がwriterの状態を更新しており、EndObject をしないので中途半端になるため)
        NN_UNUSED(pAllocator);
        *outLength = (isCommaNecessary ? 2 : 1); // "{" or ",{"
        return true;
    }
    bool GetObjectEndLength(size_t* outLength, json::FixedSizeAllocator* pAllocator)
    {
        // RapidJSON の EndObject を使う代わりに自前でデータを出力
        // (EndObject で RapidJSON がwriterの状態をチェックしており、StartObject をしていないため ASSERT になるので使用できない)
        NN_UNUSED(pAllocator);
        *outLength = 1; // "}"
        return true;
    }
    bool GetArrayBeginLength(size_t* outLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        // RapidJSON の StartArray を使う代わりに自前でデータを出力
        // (StartArray で RapidJSON がwriterの状態を更新しており、EndArray をしないので中途半端になるため)
        NN_UNUSED(pAllocator);
        *outLength = (isCommaNecessary ? 2 : 1); // "[" or ",["
        return true;
    }
    bool GetArrayEndLength(size_t* outLength, json::FixedSizeAllocator* pAllocator)
    {
        // RapidJSON の EndArray を使う代わりに自前でデータを出力
        // (EndArray で RapidJSON がwriterの状態をチェックしており、StartArray をしていないため ASSERT になるので使用できない)
        NN_UNUSED(pAllocator);
        *outLength = 1; // "]"
        return true;
    }

    template <typename T>
    bool PutValueToBuffer(size_t* outLength, T value, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        JsonMemoryOutputStream os;
        MemoryOutputStreamWriter writer(os, pAllocator, 16);
        os.Open(static_cast<JsonMemoryOutputStream::Ch*>(buffer), bufferLength);
        os.PutBegin();
        if (isCommaNecessary)
        {
            os.Put(',');
        }
        if (!WriteValueCaller<T, MemoryOutputStreamWriter>::Write(writer, value))
        {
            return false;
        }
        *outLength = os.PutEnd();
        return true;
    }
    bool PutValueToBuffer(size_t* outLength, const char* value, size_t length, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        JsonMemoryOutputStream os;
        MemoryOutputStreamWriter writer(os, pAllocator, 16);
        os.Open(static_cast<JsonMemoryOutputStream::Ch*>(buffer), bufferLength);
        os.PutBegin();
        if (isCommaNecessary)
        {
            os.Put(',');
        }
        if (!WriteValueCaller<const char*, MemoryOutputStreamWriter>::Write(writer, value, length))
        {
            return false;
        }
        *outLength = os.PutEnd();
        return true;
    }
    bool PutKeyToBuffer(size_t* outLength, const char* key, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        JsonMemoryOutputStream os;
        MemoryOutputStreamWriter writer(os, pAllocator, 16);
        os.Open(static_cast<JsonMemoryOutputStream::Ch*>(buffer), bufferLength);
        os.PutBegin();
        if (isCommaNecessary)
        {
            os.Put(',');
        }
        if (!writer.Key(key))
        {
            return false;
        }
        os.Put(':');
        *outLength = os.PutEnd();
        return true;
    }
    bool PutNullToBuffer(size_t* outLength, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        JsonMemoryOutputStream os;
        MemoryOutputStreamWriter writer(os, pAllocator, 16);
        os.Open(static_cast<JsonMemoryOutputStream::Ch*>(buffer), bufferLength);
        os.PutBegin();
        if (isCommaNecessary)
        {
            os.Put(',');
        }
        if (!writer.Null())
        {
            return false;
        }
        *outLength = os.PutEnd();
        return true;
    }
    bool PutObjectBeginToBuffer(size_t* outLength, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        // RapidJSON の StartObject を使う代わりに自前でデータを出力
        // (StartObject で RapidJSON がwriterの状態を更新しており、EndObject をしないので中途半端になるため)
        NN_UNUSED(pAllocator);
        char* p = static_cast<char*>(buffer);
        size_t written = 0;
        if (isCommaNecessary && bufferLength > 0)
        {
            *p++ = ',';
            --bufferLength;
            ++written;
        }
        if (bufferLength > 0)
        {
            *p++ = '{';
            --bufferLength;
            ++written;
        }
        *outLength = written;
        return true;
    }
    bool PutObjectEndToBuffer(size_t* outLength, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator)
    {
        // RapidJSON の EndObject を使う代わりに自前でデータを出力
        // (EndObject で RapidJSON がwriterの状態をチェックしており、StartObject をしていないため ASSERT になるので使用できない)
        NN_UNUSED(pAllocator);
        if (bufferLength > 0)
        {
            *static_cast<char*>(buffer) = '}';
            *outLength = 1;
        }
        else
        {
            *outLength = 0;
        }
        return true;
    }
    bool PutArrayBeginToBuffer(size_t* outLength, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator, bool isCommaNecessary)
    {
        // RapidJSON の StartArray を使う代わりに自前でデータを出力
        // (StartArray で RapidJSON がwriterの状態を更新しており、EndArray をしないので中途半端になるため)
        NN_UNUSED(pAllocator);
        char* p = static_cast<char*>(buffer);
        size_t written = 0;
        if (isCommaNecessary && bufferLength > 0)
        {
            *p++ = ',';
            --bufferLength;
            ++written;
        }
        if (bufferLength > 0)
        {
            *p++ = '[';
            --bufferLength;
            ++written;
        }
        *outLength = written;
        return true;
    }
    bool PutArrayEndToBuffer(size_t* outLength, void* buffer, size_t bufferLength, json::FixedSizeAllocator* pAllocator)
    {
        // RapidJSON の EndArray を使う代わりに自前でデータを出力
        // (EndArray で RapidJSON がwriterの状態をチェックしており、StartArray をしていないため ASSERT になるので使用できない)
        NN_UNUSED(pAllocator);
        if (bufferLength > 0)
        {
            *static_cast<char*>(buffer) = ']';
            *outLength = 1;
        }
        else
        {
            *outLength = 0;
        }
        return true;
    }
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::FillData(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT
{
    return m_pCacheBuffer != nullptr ? FillDataWithCache(outLengthWritten, buffer, length) :
        FillDataWithoutCache(outLengthWritten, buffer, length);
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::FillDataWithCache(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT
{
    // (FillDataWithoutCache は length == 0 のときに NeedMoreBuffer を返すが、
    // FillDataWithCache では length == 0 のとき NoMoreData になるので戻り値を合わせる)
    if (length == 0)
    {
        *outLengthWritten = 0;
        return WriteResult::NeedMoreBuffer;
    }

    NN_SDK_ASSERT_NOT_NULL(m_pCacheBuffer);
    NN_SDK_ASSERT_GREATER(m_CacheBufferSize, static_cast<size_t>(0));

    size_t written = 0;
    while (NN_STATIC_CONDITION(true))
    {
        size_t cacheRemainingSize = m_CacheCurrentSize - m_CacheCurrentOffset;
        // キャッシュ済みデータの出力
        if (cacheRemainingSize > 0)
        {
            // cacheRemainingSize と length のいずれか小さい方のサイズ分コピーする
            size_t writeLength = (cacheRemainingSize < length ? cacheRemainingSize : length);
            std::memcpy(buffer, static_cast<char*>(m_pCacheBuffer) + m_CacheCurrentOffset,
                writeLength);
            m_CacheCurrentOffset += writeLength;
            written += writeLength;
            if (writeLength == length) // buffer に length 分ちょうど書き込んだ場合は終了
            {
                break;
            }
            // まだ空きがあるのでポインターを進める
            buffer = static_cast<char*>(buffer) + writeLength;
            length -= writeLength;
        }
        // データのキャッシュ
        char* pCacheBufferCurrentPosition = static_cast<char*>(m_pCacheBuffer);
        size_t remainingBufferSize = m_CacheBufferSize;
        m_CacheCurrentSize = 0;
        m_CacheCurrentOffset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            size_t writtenSize = 0;
            auto result = FillDataWithoutCache(&writtenSize, pCacheBufferCurrentPosition, remainingBufferSize);
            switch (result)
            {
                case WriteResult::Succeeded:
                    m_CacheCurrentSize += writtenSize;
                    remainingBufferSize -= writtenSize;
                    pCacheBufferCurrentPosition += writtenSize;
                    break;
                case WriteResult::NeedMoreBuffer:
                    if (remainingBufferSize == m_CacheBufferSize)
                    {
                        // remainingBufferSize が変わっていない == 何もキャッシュに書き込まれていない
                        NN_DETAIL_PCTL_WARN("[pctl] Warning: cache buffer size is too small\n");
                    }
                    break;
                case WriteResult::NoMoreData:
                    break;
                case WriteResult::Error:
                    return WriteResult::Error;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
            if (result != WriteResult::Succeeded)
            {
                break;
            }
        }
        // remainingBufferSize が変わらない == これ以上データがない
        if (remainingBufferSize == m_CacheBufferSize)
        {
            break;
        }
    }

    *outLengthWritten = written;
    return written > 0 ? WriteResult::Succeeded : WriteResult::NoMoreData;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::FillDataWithoutCache(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT
{
    size_t written = 0;
    char* pointer = static_cast<char*>(buffer);
    while (NN_STATIC_CONDITION(true))
    {
        size_t writtenSingle = 0;
        WriteResult result = WriteCurrentData(&writtenSingle, pointer, length);
        switch (result)
        {
            case WriteResult::Succeeded:
                written += writtenSingle;
                pointer += writtenSingle;
                length -= writtenSingle;
                break;
            case WriteResult::NeedMoreBuffer:
            case WriteResult::NoMoreData:
                *outLengthWritten = written;
                return (written == 0) ? result : WriteResult::Succeeded;
            case WriteResult::Error:
                return WriteResult::Error;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
    // unreachable
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::CalculateTotalSize(size_t* outLength) NN_NOEXCEPT
{
    static const size_t MaxSizeTValue = std::numeric_limits<size_t>::max();
    size_t written = 0;
    while (NN_STATIC_CONDITION(true))
    {
        size_t writtenSingle = 0;
        WriteResult result = WriteCurrentData(&writtenSingle, nullptr, MaxSizeTValue);
        switch (result)
        {
            case WriteResult::Succeeded:
                written += writtenSingle;
                break;
            case WriteResult::NeedMoreBuffer:
            case WriteResult::NoMoreData:
                *outLength = written;
                return (written == 0) ? result : WriteResult::Succeeded;
            case WriteResult::Error:
                return WriteResult::Error;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
    // unreachable
}

void JsonStructuredWriter::SetCacheBuffer(void* pCacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT
{
    if (pCacheBuffer == nullptr || cacheBufferSize == 0)
    {
        m_pCacheBuffer = nullptr;
        m_CacheBufferSize = 0;
    }
    else
    {
        m_pCacheBuffer = pCacheBuffer;
        m_CacheBufferSize = cacheBufferSize;
    }
    m_CacheCurrentOffset = 0;
    m_CacheCurrentSize = 0;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteCurrentData(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    int nowDepth = m_CurrentDepth;
    auto& info = m_DepthInfo[nowDepth];
    uint8_t type;
    WriteResult result;
    bool hasItem = true;
    bool isRepeat = false;
    if (info.currentPosition == info.subArrayLength)
    {
        if (nowDepth == 0)
        {
            return WriteResult::NoMoreData;
        }
        if (info.isObject)
        {
            type = WriteType::WriteType_ObjectEnd;
        }
        else
        {
            type = WriteType::WriteType_ArrayEnd;
        }
        result = WriteData(outLength, &hasItem, type, nullptr, buffer, bufferLength);
    }
    else if (info.isObject && info.repeatObjectFunction != nullptr && !info.isNextValue)
    {
        type = WriteType::WriteType_ObjectRepeat;
        result = WriteObjectRepeatNext(outLength, info.repeatObjectFunction, buffer, bufferLength);
    }
    else
    {
        const auto& dataInfo = info.subArray[info.currentPosition];

        type = (dataInfo.type & WriteType::WriteType_TypeMask);
        isRepeat = ((dataInfo.type & WriteType::WriteType_ArrayRepeatValues) != 0);

        result = WriteData(outLength, &hasItem, dataInfo.type, dataInfo.data, buffer, bufferLength);
    }
    if (result == WriteResult::Succeeded)
    {
        // Object や Array で m_CurrentDepth が変化する(その場合 info != m_DepthInfo[m_CurrentDepth])ことがあるため取り直す
        auto& currentInfo = m_DepthInfo[m_CurrentDepth];
        if (type == WriteType::WriteType_ArrayBegin)
        {
            // null 値を使ったなど階層が変わっていない場合はリピートしない
            if (m_CurrentDepth == nowDepth)
            {
                isRepeat = false;
            }
        }
        // 次の読み取りにおける階層が配列の要素繰り返し状態かどうかを確認
        // (ObjectRepeat / ArrayBegin のときは既に確認済み)
        else if (m_CurrentDepth > 0 && currentInfo.repeatFunction != nullptr)
        {
            if (currentInfo.isObject)
            {
                // ObjectRepeat の場合は次が値でなければ繰り返し処理とする
                if (!currentInfo.isNextValue)
                {
                    hasItem = true;
                    isRepeat = true;
                }
            }
            else
            {
                nn::util::optional<bool> hasMoreItem = static_cast<bool>(false);
                auto& parentInfo = m_DepthInfo[m_CurrentDepth - 1];
                if (!currentInfo.repeatFunction(&hasMoreItem, m_Param, parentInfo.index, currentInfo.index))
                {
                    return WriteResult::Error;
                }
                if (hasMoreItem == nn::util::nullopt)
                {
                    return WriteResult::Error;
                }
                hasItem = *hasMoreItem;
                isRepeat = true;
            }
        }
        if (isRepeat)
        {
            // 要素がある場合は次の読み取り位置を階層の先頭に、要素がもう無い場合は ArrayEnd に設定する
            if (hasItem)
            {
                currentInfo.currentPosition = 0;
            }
            else
            {
                currentInfo.currentPosition = currentInfo.subArrayLength;
            }
        }
    }
    return result;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteData(size_t* outLength, bool* outHasItem, uint8_t type, void* data, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    bool isRepeat = ((type & WriteType::WriteType_ArrayRepeatValues) != 0);
    switch (type & WriteType::WriteType_TypeMask)
    {
        case WriteType::WriteType_Key:
            if ((type & WriteType::WriteType_OptionalKey) != 0)
            {
                return WriteKeyOptional(outLength, reinterpret_cast<OutputKeyNameFunction>(data), buffer, bufferLength);
            }
            else
            {
                return WriteKey(outLength, reinterpret_cast<const char*>(data), buffer, bufferLength);
            }
        case WriteType::WriteType_String:
            return WriteString(outLength, reinterpret_cast<OutputStringValueFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_Int64:
            return WriteSimpleValue(outLength, reinterpret_cast<OutputInt64ValueFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_Uint64:
            return WriteSimpleValue(outLength, reinterpret_cast<OutputUint64ValueFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_Double:
            return WriteSimpleValue(outLength, reinterpret_cast<OutputDoubleValueFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_Boolean:
            return WriteSimpleValue(outLength, reinterpret_cast<OutputBooleanValueFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_ObjectBegin:
            return WriteObjectBegin(outLength, reinterpret_cast<ReadyForOutputObjectFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_ObjectAny:
            return WriteObjectAnyBegin(outLength, reinterpret_cast<ReadyForOutputAnyObjectFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_ObjectRepeat:
            return WriteObjectRepeatFirst(outLength, reinterpret_cast<ReadyForOutputObjectRepeatFunction>(data), buffer, bufferLength);
        case WriteType::WriteType_ObjectEnd:
            return WriteObjectEnd(outLength, buffer, bufferLength);
        case WriteType::WriteType_ArrayBegin:
            return WriteArrayBegin(outLength, outHasItem, data, isRepeat, buffer, bufferLength);
        case WriteType::WriteType_ArrayEnd:
            return WriteArrayEnd(outLength, buffer, bufferLength);
        default:
            NN_SDK_ASSERT(false, "Unexpected data format");
            return WriteResult::Error;
    }
    // unreachable
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteKey(size_t* outLength, const char* keyName, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    FixedSizeAllocator allocator;
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);

    if (!GetKeyLength(&len, keyName, &allocator, isCommaNecessary))
    {
        return WriteResult::Error;
    }
    if (len > bufferLength)
    {
        return WriteResult::NeedMoreBuffer;
    }
    if (buffer != nullptr && !PutKeyToBuffer(&len, keyName, buffer, bufferLength, &allocator, isCommaNecessary))
    {
        return WriteResult::Error;
    }
    *outLength = len;

    // 読み取り位置情報の更新(index は更新しない)
    info.isCommaNecessary = false;
    info.isNextValue = true;
    ++info.currentPosition;

    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteKeyOptional(size_t* outLength, OutputKeyNameFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    FixedSizeAllocator allocator;
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
    const char* keyName;

    if (!func(&keyName, m_Param, info.index))
    {
        return WriteResult::Error;
    }
    if (keyName == nullptr)
    {
        // 次の次の位置を取得
        int pos = CalculateNextPosition(info.currentPosition + 1);
        if (pos < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = pos;
        // isCommaNecessary はそのまま
        *outLength = 0;
    }
    else
    {
        if (!GetKeyLength(&len, keyName, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutKeyToBuffer(&len, keyName, buffer, bufferLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
        *outLength = len;

        // 読み取り位置情報の更新
        info.isCommaNecessary = false;
        info.isNextValue = true;
        ++info.currentPosition;
    }
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteString(size_t* outLength, OutputStringValueFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    FixedSizeAllocator allocator;
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
    const char* value = nullptr;
    nn::util::optional<size_t> valueLength(static_cast<size_t>(0));
    if (!func(&valueLength, &value, m_Param, info.index))
    {
        return WriteResult::Error;
    }
    if (!valueLength)
    {
        len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
    }
    else
    {
        if (!GetValueLength(&len, value, *valueLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutValueToBuffer(&len, value, *valueLength, buffer, bufferLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
    }
    *outLength = len;

    // 読み取り位置情報の更新
    info.isCommaNecessary = true;
    info.isNextValue = false;
    ++info.index;
    ++info.currentPosition;

    return WriteResult::Succeeded;
}

// この関数の呼び出しはこのファイルでしか行わないのでここで定義
template <typename FunctionType>
JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteSimpleValue(size_t* outLength, FunctionType func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    typedef typename TypeFromOutputFunction<FunctionType>::Type ValueType;
    FixedSizeAllocator allocator;
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
    nn::util::optional<ValueType> value;
    if (!func(&value, m_Param, info.index))
    {
        return WriteResult::Error;
    }
    if (!value)
    {
        len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
    }
    else
    {
        if (!GetValueLength(&len, *value, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutValueToBuffer(&len, *value, buffer, bufferLength, &allocator, isCommaNecessary))
        {
            return WriteResult::Error;
        }
    }
    *outLength = len;

    // 読み取り位置情報の更新
    info.isCommaNecessary = true;
    info.isNextValue = false;
    ++info.index;
    ++info.currentPosition;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteObjectBegin(size_t* outLength, ReadyForOutputObjectFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isNull = false;

    // scope: データ書き込み
    {
        FixedSizeAllocator allocator;
        bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
        if (func != nullptr && !func(&isNull, m_Param, info.index))
        {
            return WriteResult::Error;
        }
        if (isNull)
        {
            len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
        else
        {
            if (!GetObjectBeginLength(&len, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutObjectBeginToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
    }

    if (isNull)
    {
        // depth を更新せず位置を ObjectEnd の次に進める
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
    }
    else
    {
        // depth を更新
        if (!PushDepthInfo(info, nullptr, WriteType::WriteType_ObjectEnd))
        {
            return WriteResult::Error;
        }
    }
    *outLength = len;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteObjectAnyBegin(size_t* outLength, ReadyForOutputAnyObjectFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    nn::util::optional<int> definitionLength;
    const WriteDataDefinition* definitionArray;

    // scope: データ書き込み
    {
        FixedSizeAllocator allocator;
        bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
        if (!func(&definitionLength, &definitionArray, m_Param, info.index))
        {
            return WriteResult::Error;
        }
        if (!definitionLength)
        {
            len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
        else
        {
            if (!GetObjectBeginLength(&len, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutObjectBeginToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
    }

    if (!definitionLength)
    {
        // depth を更新せず位置を ObjectEnd の次に進める
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
    }
    else
    {
        NN_SDK_ASSERT(definitionArray != nullptr, "The definition data returned by ReadyForOutputAnyObjectFunction is nullptr");
        // depth を更新
        auto& infoChild = m_DepthInfo[++m_CurrentDepth];
        infoChild.subArray = definitionArray;
        infoChild.subArrayLength = *definitionLength;
        infoChild.currentPosition = 0;
        infoChild.index = 0;
        infoChild.isObject = true;
        infoChild.isCommaNecessary = false;
        infoChild.isNextValue = false;
        infoChild.repeatFunction = nullptr;
    }
    *outLength = len;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteObjectRepeatFirst(size_t* outLength, ReadyForOutputObjectRepeatFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    size_t len = 0;
    auto& info = m_DepthInfo[m_CurrentDepth];
    int definitionLength;
    const WriteDataDefinition* definitionArray;
    nn::util::optional<bool> hasItem = false;
    const char* keyName = nullptr;

    // scope: データ書き込み
    {
        FixedSizeAllocator allocator;
        bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
        if (!func(&hasItem, &keyName, &definitionLength, &definitionArray, m_Param, info.index, 0))
        {
            return WriteResult::Error;
        }
        if (hasItem == nn::util::nullopt)
        {
            len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
        else
        {
            size_t lenKeyName = 0;
            // 最初のみオブジェクト開始記号を書き込む
            if (!GetObjectBeginLength(&len, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (*hasItem)
            {
                NN_SDK_ASSERT(keyName != nullptr, "Unexpected: keyName is must be valid (nullptr is not allowed)");
                // 先頭なので ',' は不要
                if (!GetKeyLength(&lenKeyName, keyName, &allocator, false))
                {
                    return WriteResult::Error;
                }
            }
            if (len + lenKeyName > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }

            if (buffer != nullptr && !PutObjectBeginToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (*hasItem)
            {
                if (buffer != nullptr && !PutKeyToBuffer(&lenKeyName, keyName, static_cast<uint8_t*>(buffer) + len, bufferLength - len, &allocator, false))
                {
                    return WriteResult::Error;
                }
            }
            len += lenKeyName;
        }
    }

    if (hasItem == nn::util::nullopt)
    {
        // depth を更新せず位置を次に進める
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
    }
    else
    {
        // depth を更新
        auto& infoChild = m_DepthInfo[++m_CurrentDepth];
        infoChild.subArray = definitionArray;
        infoChild.subArrayLength = definitionLength;
        infoChild.currentPosition = (*hasItem ? 0 : definitionLength); // 要素がない場合は即終了とする
        infoChild.index = 0;
        infoChild.isObject = true;
        infoChild.isCommaNecessary = false;
        infoChild.isNextValue = true; // キー書き込み済みなので値の書き込みに進める
        infoChild.repeatObjectFunction = func;
    }

    *outLength = len;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteObjectRepeatNext(size_t* outLength, ReadyForOutputObjectRepeatFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    size_t len = 0;
    auto& info = m_DepthInfo[m_CurrentDepth];
    int definitionLength;
    const WriteDataDefinition* definitionArray;
    nn::util::optional<bool> hasItem = false;
    const char* keyName = nullptr;
    int baseIndex;

    // scope: 親要素のインデックス値を取得
    {
        NN_SDK_ASSERT(m_CurrentDepth > 0, "Unexpected: WriteObjectRepeat called with root data");
        auto& parentInfo = m_DepthInfo[m_CurrentDepth - 1];
        baseIndex = parentInfo.index;
    }

    // scope: データ書き込み
    {
        FixedSizeAllocator allocator;
        bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
        if (!func(&hasItem, &keyName, &definitionLength, &definitionArray, m_Param, baseIndex, info.index))
        {
            return WriteResult::Error;
        }
        NN_SDK_ASSERT(hasItem != nn::util::nullopt, "Unexpected: nullopt is not allowed for non-first item");
        if (hasItem == nn::util::nullopt)
        {
            return WriteResult::Error;
        }
        else if (*hasItem)
        {
            // 要素がある場合のみキー名を書き込む
            NN_SDK_ASSERT(keyName != nullptr, "Unexpected: keyName is must be valid (nullptr is not allowed)");
            if (!GetKeyLength(&len, keyName, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }

            if (buffer != nullptr && !PutKeyToBuffer(&len, keyName, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
    }

    // WriteCurrentData で使用するため要素の有無にかかわらずセットする
    info.isNextValue = true;
    info.isCommaNecessary = false;
    if (!*hasItem)
    {
        // もう要素がないので末尾に進める
        info.currentPosition = info.subArrayLength;
    }
    else
    {
        // subArray を更新しつつその先頭要素を読み取るようにセットする
        info.subArray = definitionArray;
        info.subArrayLength = definitionLength;
        info.currentPosition = 0;
    }

    *outLength = len;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteObjectEnd(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CurrentDepth > 0, "Invalid data definition: ObjectEnd without ObjectBegin");
    if (m_CurrentDepth == 0)
    {
        return WriteResult::Error;
    }
    size_t len;
    {
        FixedSizeAllocator allocator;
        if (!GetObjectEndLength(&len, &allocator))
        {
            return WriteResult::Error;
        }
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutObjectEndToBuffer(&len, buffer, bufferLength, &allocator))
        {
            return WriteResult::Error;
        }
    }

    // depth を更新して次の位置へ
    {
        auto& info = m_DepthInfo[--m_CurrentDepth];
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
        ++info.index;
    }
    *outLength = len;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteArrayBegin(size_t* outLength, bool* outHasItem, void* func, bool isRepeat, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    size_t len;
    auto& info = m_DepthInfo[m_CurrentDepth];
    bool isNull;
    bool hasItem = true;

    // scope: データ書き込み
    {
        FixedSizeAllocator allocator;
        bool isCommaNecessary = (m_CurrentDepth > 0 && info.isCommaNecessary);
        if (isRepeat)
        {
            nn::util::optional<bool> value;
            if (!reinterpret_cast<ReadyForOutputArrayRepeatFunction>(func)(&value, m_Param, info.index, 0))
            {
                return WriteResult::Error;
            }
            isNull = (value == nn::util::nullopt);
            if (!isNull)
            {
                hasItem = *value;
            }
        }
        else
        {
            isNull = false;
            if (func != nullptr && !reinterpret_cast<ReadyForOutputArrayFunction>(func)(&isNull, m_Param, info.index))
            {
                return WriteResult::Error;
            }
        }
        if (isNull)
        {
            len = LengthForNullValue + (isCommaNecessary ? 1 : 0);
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutNullToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
        else
        {
            if (!GetArrayBeginLength(&len, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
            if (len > bufferLength)
            {
                return WriteResult::NeedMoreBuffer;
            }
            if (buffer != nullptr && !PutArrayBeginToBuffer(&len, buffer, bufferLength, &allocator, isCommaNecessary))
            {
                return WriteResult::Error;
            }
        }
    }

    if (isNull)
    {
        // depth を更新せず位置を ArrayEnd の次に進める
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
    }
    else
    {
        // depth を更新
        if (!PushDepthInfo(info, isRepeat ? func : nullptr, WriteType::WriteType_ArrayEnd))
        {
            return WriteResult::Error;
        }
    }
    *outLength = len;
    *outHasItem = hasItem;
    return WriteResult::Succeeded;
}

JsonStructuredWriter::WriteResult JsonStructuredWriter::WriteArrayEnd(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CurrentDepth > 0, "Invalid data definition: ArrayEnd without ArrayBegin");
    if (m_CurrentDepth == 0)
    {
        return WriteResult::Error;
    }
    size_t len;
    {
        FixedSizeAllocator allocator;
        if (!GetArrayEndLength(&len, &allocator))
        {
            return WriteResult::Error;
        }
        if (len > bufferLength)
        {
            return WriteResult::NeedMoreBuffer;
        }
        if (buffer != nullptr && !PutArrayEndToBuffer(&len, buffer, bufferLength, &allocator))
        {
            return WriteResult::Error;
        }
    }

    // depth を更新して次の位置へ
    {
        auto& info = m_DepthInfo[--m_CurrentDepth];
        int nextPosition = CalculateNextPosition(info.currentPosition);
        if (nextPosition < 0)
        {
            return WriteResult::Error;
        }
        info.currentPosition = nextPosition;
        info.isCommaNecessary = true;
        info.isNextValue = false;
        ++info.index;
    }
    *outLength = len;
    return WriteResult::Succeeded;
}


int JsonStructuredWriter::CalculateNextPosition(int startPosition) const NN_NOEXCEPT
{
    auto& info = m_DepthInfo[m_CurrentDepth];
    auto& data = info.subArray[startPosition];
    auto dataType = (data.type & WriteType::WriteType_TypeMask);
    if (dataType != WriteType::WriteType_ObjectBegin && dataType != WriteType::WriteType_ArrayBegin)
    {
        return startPosition + 1;
    }
    ++startPosition;
    // 対応する ObjectEnd/ArrayEnd を探す
    int nestCount = 0;
    while (startPosition < info.subArrayLength)
    {
        uint8_t type = (info.subArray[startPosition].type & WriteType::WriteType_TypeMask);
        if (type == WriteType::WriteType_ObjectEnd || type == WriteType::WriteType_ArrayEnd)
        {
            if (nestCount == 0)
            {
                // Begin と End の種類が一致していない場合は -1 を返す
                // (「!=」の lhs と rhs がそれぞれ Object である場合は true、Array である場合は false になる)
                if ((dataType == WriteType::WriteType_ObjectBegin) != (type == WriteType::WriteType_ObjectEnd))
                {
                    return -1;
                }
                // End の次の位置を返す
                return startPosition + 1;
            }
            // 厳密な対応はここでは見ない
            --nestCount;
        }
        else if (type == WriteType::WriteType_ObjectBegin || type == WriteType::WriteType_ArrayBegin)
        {
            if (nestCount == MaxDepth)
            {
                // 入れ子が深すぎる
                NN_SDK_ASSERT(false, "Invalid data definition: Too much recursion");
                return false;
            }
            ++nestCount;
        }
        ++startPosition;
    }
    return -1;
}

bool JsonStructuredWriter::PushDepthInfo(const DepthInfo& currentInfo, void* repeatFunctionPtr, WriteType expectEndType) NN_NOEXCEPT
{
    // depth を更新
    // (対応する expectEndType を探す)
    int nestCount = 0;
    int pos = currentInfo.currentPosition + 1;
    bool found = false;
    while (pos < currentInfo.subArrayLength)
    {
        uint8_t type = (currentInfo.subArray[pos].type & WriteType::WriteType_TypeMask);
        if (type == WriteType::WriteType_ObjectEnd || type == WriteType::WriteType_ArrayEnd)
        {
            if (nestCount == 0)
            {
                NN_SDK_ASSERT(type == expectEndType, "Invalid data definition: ObjectBegin/ArrayBegin found but no end-type found (confused for another end type?)");
                auto& infoChild = m_DepthInfo[++m_CurrentDepth];
                infoChild.subArray = &currentInfo.subArray[currentInfo.currentPosition + 1]; // Start を含めない
                infoChild.subArrayLength = pos - (currentInfo.currentPosition + 1); // End を含めない
                infoChild.currentPosition = 0;
                infoChild.index = 0;
                infoChild.isObject = (expectEndType != WriteType::WriteType_ArrayEnd);
                infoChild.isCommaNecessary = false;
                infoChild.isNextValue = false;
                infoChild.repeatFunctionPtr = repeatFunctionPtr;
                found = true;
                break;
            }
            // 厳密な対応はここでは見ない
            --nestCount;
        }
        else if (type == WriteType::WriteType_ObjectBegin || type == WriteType::WriteType_ArrayBegin)
        {
            if (nestCount == MaxDepth)
            {
                // 入れ子が深すぎる
                NN_SDK_ASSERT(false, "Invalid data definition: Too much recursion");
                return false;
            }
            ++nestCount;
        }
        ++pos;
    }
    if (!found)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: ArrayBegin found but no ArrayEnd found");
        return false;
    }
    return true;
}

}}}}}
