﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

namespace
{
    static const char SystemSaveDataMountPoint[] = "pctlss";
    static const nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000100; // refs. SIGLO-26689
    static const size_t SystemSaveDataTotalSize = 512 * 1024;
    // Commit時における差分サイズ + 32KiB(ファイル・ディレクトリの管理テーブル 16KiB + 16KiB) が
    // この値を超えないようにジャーナルサイズを設定する
    // (320KiB のサイズのファイル(実際に発生しうる差分はほとんどの場合 1～2KiB 程度)が
    // あるため、念のため大きく確保しておく)
    static const size_t SystemSaveDataJournalSize = 512 * 1024;
    NN_STATIC_ASSERT(SystemSaveDataJournalSize <= SystemSaveDataTotalSize);

    static int g_FileOpened = 0;
    nn::os::SdkMutexType g_MutexForLockCommit = NN_OS_SDK_MUTEX_INITIALIZER();

    static nn::Result _InitializeSystemSaveData() NN_NOEXCEPT
    {
        nn::fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(nn::fs::MountSystemSaveData(SystemSaveDataMountPoint, SystemSaveDataId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Initialize save data.\n");
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(SystemSaveDataId, SystemSaveDataTotalSize, SystemSaveDataJournalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(SystemSaveDataMountPoint, SystemSaveDataId));
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
}

void FileSystem::InitializeFileSystem() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(_InitializeSystemSaveData());
}

void FileSystem::FinalizeFileSystem() NN_NOEXCEPT
{
    nn::fs::Unmount(SystemSaveDataMountPoint);
}

nn::Result FileSystem::OpenRead(FileStream* outStream, const char* fileName) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, fileName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    NN_RESULT_DO(nn::fs::OpenFile(&outStream->m_FileHandle, name,
        nn::fs::OpenMode::OpenMode_Read));
    outStream->m_bOpened = true;
    ++g_FileOpened;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::OpenWrite(FileStream* outStream, const char* fileName) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, fileName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    NN_RESULT_TRY(nn::fs::CreateFile(name, 0))
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            // ファイルサイズ未指定の場合はどんなサイズのファイルであっても許容できる
        }
    NN_RESULT_END_TRY
    NN_RESULT_DO(nn::fs::OpenFile(&outStream->m_FileHandle, name,
        nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    outStream->m_bOpened = true;
    ++g_FileOpened;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::OpenWrite(FileStream* outStream, const char* fileName, int64_t fileSize) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, fileName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    auto result = nn::fs::OpenFile(&outStream->m_FileHandle, name,
        nn::fs::OpenMode::OpenMode_Write);
    // ファイルが存在する場合は Success になるが、
    // サイズが意図しないものになっている場合はファイルを作り直す
    if (result.IsSuccess())
    {
        int64_t size = 0;
        NN_RESULT_DO(nn::fs::GetFileSize(&size, outStream->m_FileHandle));
        if (size != fileSize)
        {
            nn::fs::CloseFile(outStream->m_FileHandle);
            NN_RESULT_DO(nn::fs::DeleteFile(name));
            result = nn::fs::ResultPathNotFound();
        }
    }
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateFile(name, fileSize));
            NN_RESULT_DO(nn::fs::OpenFile(&outStream->m_FileHandle, name,
                nn::fs::OpenMode::OpenMode_Write));
        }
    NN_RESULT_END_TRY
    outStream->m_bOpened = true;
    ++g_FileOpened;
    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::Delete(const char* fileName) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, fileName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    return nn::fs::DeleteFile(name);
}

nn::Result FileSystem::CreateDirectory(const char* directoryName) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, directoryName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    return nn::fs::CreateDirectory(name);
}

nn::Result FileSystem::DeleteDirectory(const char* directoryName, bool recursive) NN_NOEXCEPT
{
    char name[128] = {};
    int length = nn::util::SNPrintf(name, sizeof (name), "%s:/%s", SystemSaveDataMountPoint, directoryName);

    NN_SDK_ASSERT(static_cast<size_t>(length) < sizeof (name));
    NN_UNUSED(length);

    NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
    return recursive ? nn::fs::DeleteDirectoryRecursively(name) : nn::fs::DeleteDirectory(name);
}

nn::Result FileSystem::Commit() NN_NOEXCEPT
{
    // ロックされていない、かつオープン済みファイルが 0 である場合にコミットを行う
    while (NN_STATIC_CONDITION(true))
    {
        {
            NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
            if (g_FileOpened == 0)
            {
                NN_RESULT_DO(nn::fs::CommitSaveData(SystemSaveDataMountPoint));
                break;
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
    }
    NN_RESULT_SUCCESS;
}


nn::Result FileStream::Read(size_t* outReadSize, int64_t offset, void* outBuffer, size_t bufferSize) NN_NOEXCEPT
{
    return nn::fs::ReadFile(outReadSize, m_FileHandle, offset, outBuffer, bufferSize);
}

nn::Result FileStream::Write(int64_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return nn::fs::WriteFile(m_FileHandle, offset, buffer, bufferSize,
        nn::fs::WriteOption::MakeValue(0));
}

nn::Result FileStream::Flush() NN_NOEXCEPT
{
    return nn::fs::FlushFile(m_FileHandle);
}

nn::Result FileStream::GetFileSize(int64_t* outFileSize) NN_NOEXCEPT
{
    return nn::fs::GetFileSize(outFileSize, m_FileHandle);
}

void FileStream::Close() NN_NOEXCEPT
{
    if (m_bOpened)
    {
        NN_UTIL_LOCK_GUARD(g_MutexForLockCommit);
        nn::fs::CloseFile(m_FileHandle);
        m_bOpened = false;
        --g_FileOpened;
    }
}

}}}}}
