﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>

#include <nn/pcm/pcm_Api.h>
#include <nn/pcm/pcm_IManager.sfdl.h> // for IManager

#include "pcm_GetManagerByHipc.h"

namespace nn {
namespace pcm {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::pcm::IManager> g_Manager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_Manager);

        g_Manager = nn::pcm::GetManagerByHipc();

        bool isServiceEnabled;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->IsServiceEnabled(&isServiceEnabled));
        NN_ABORT_UNLESS(isServiceEnabled, "Tried to initialize nn::pcm library while the service is disabled by debug settings.\n");
    }

    g_InitializeCount++;
}

void InitializeWith(nn::sf::SharedPointer<nn::pcm::IManager> manager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_Manager = manager;

    g_InitializeCount++;
}

nn::sf::SharedPointer<IManager> GetInternalManager() NN_NOEXCEPT
{
    return g_Manager;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_Manager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_Manager = nullptr;
    }
}

bool IsSupported(MeasuringPoint point) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    bool isSupported;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->IsSupported(&isSupported, static_cast<std::int32_t>(point)));
    return isSupported;
}

int ReadCurrentPower(MeasuringPoint point) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int power;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->ReadCurrentPower(&power, static_cast<std::int32_t>(point)));
    return power;
}

int ReadCurrentVoltage(MeasuringPoint point) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int voltage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->ReadCurrentVoltage(&voltage, static_cast<std::int32_t>(point)));
    return voltage;
}

} // pcm
} // nn
