﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>
#include <nn/dd.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>
#include <nn/pcie/driver/pcie_Types.h>

#include "pcie_CreateManager.h"

#define NN_PCIE_ACCESS_BY_HIPC

namespace nn { namespace pcie {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<nn::pcie::detail::ILoggedStateManager> g_LoggedStateManager;

// Initialize の参照カウント
int g_InitializeLoggedStateCount = 0;

// 参照カウントを守る Mutex
nn::os::Mutex g_InitializeLoggedStateCountMutex(false);

}

Result InitializeLoggedState() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeLoggedStateCountMutex);

    if (g_InitializeLoggedStateCount == 0)
    {

        NN_SDK_ASSERT( !g_LoggedStateManager);

        // マクロによって、HIPC/DFC を切り替える
#ifdef NN_PCIE_ACCESS_BY_HIPC
        g_LoggedStateManager = nn::pcie::CreateLoggedStateManagerByHipc();
#else
        g_LoggedStateManager = nn::pcie::CreateLoggedStateManagerByDfc();
#endif
    }

    g_InitializeLoggedStateCount++;

    return ResultSuccess();
}

Result FinalizeLoggedState() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InitializeLoggedStateCountMutex);

    NN_SDK_ASSERT(g_InitializeLoggedStateCount > 0);

    g_InitializeLoggedStateCount--;

    if (g_InitializeLoggedStateCount == 0)
    {

        NN_SDK_ASSERT(g_LoggedStateManager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_LoggedStateManager = nullptr;
    }

    return ResultSuccess();
}

Result GetLoggedState(LoggedState* pOutput, bool* pOutIsErrorLogged, bool clearAccumulated) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::sf::OutBuffer loggedStateOutput(reinterpret_cast<char *>(pOutput), sizeof(LoggedState));
    NN_ABORT_UNLESS_NOT_NULL(pOutput);
    NN_SDK_ASSERT(g_LoggedStateManager);
    result = g_LoggedStateManager->GetLoggedState(loggedStateOutput, pOutIsErrorLogged, clearAccumulated);
    return result;
}

void PrintLoggedState(nn::pcie::LoggedState& loggedState, void (*printfCallback) (const char *str, ...))
{
    const char* speedNames[] = {"BusSpeed_Invalid", "BusSpeed_Auto", "BusSpeed_EGen1", "BusSpeed_EGen2", "BusSpeed_EGen3"};
    const char* pmStateNames[] = {"PmState_D0", "PmState_D1", "PmState_D2", "PmState_D3Hot", "PmState_D3Off"};
    printfCallback("afiInitiatorSlaveReadErrorCount   %d\n", loggedState.rootComplex.globalCounts.afiInitiatorSlaveReadErrorCount);
    printfCallback("afiInitiatorSlaveWriteErrorCount  %d\n", loggedState.rootComplex.globalCounts.afiInitiatorSlaveWriteErrorCount);
    printfCallback("afiInitiatorDecodeReadErrorCount  %d\n", loggedState.rootComplex.globalCounts.afiInitiatorDecodeReadErrorCount);
    printfCallback("afiInitiatorDecodeWriteErrorCount %d\n", loggedState.rootComplex.globalCounts.afiInitiatorDecodeWriteErrorCount);
    printfCallback("afiTargetSlaveReadErrorCount      %d\n", loggedState.rootComplex.globalCounts.afiTargetSlaveReadErrorCount);
    printfCallback("afiTargetSlaveWriteErrorCount     %d\n", loggedState.rootComplex.globalCounts.afiTargetSlaveWriteErrorCount);
    printfCallback("afiTargetDecodeReadErrorCount     %d\n", loggedState.rootComplex.globalCounts.afiTargetDecodeReadErrorCount);
    printfCallback("afiTargetDecodeWriteErrorCount    %d\n", loggedState.rootComplex.globalCounts.afiTargetDecodeWriteErrorCount);
    printfCallback("afiTargetWriteErrorCount          %d\n", loggedState.rootComplex.globalCounts.afiTargetWriteErrorCount);
    printfCallback("afiDfpciDecodeReadErrorCount      %d\n", loggedState.rootComplex.globalCounts.afiDfpciDecodeReadErrorCount);
    printfCallback("afiDfpciDecodeWriteErrorCount     %d\n", loggedState.rootComplex.globalCounts.afiDfpciDecodeWriteErrorCount);
    printfCallback("afiFpciReadTimeoutCount           %d\n", loggedState.rootComplex.globalCounts.afiFpciReadTimeoutCount);
    printfCallback("afiFpciWriteTimeoutCount          %d\n", loggedState.rootComplex.globalCounts.afiFpciWriteTimeoutCount);
    printfCallback("afiPePresentSenseCount            %d\n", loggedState.rootComplex.globalCounts.afiPePresentSenseCount);
    printfCallback("afiPeClockReqSenseCount           %d\n", loggedState.rootComplex.globalCounts.afiPeClockReqSenseCount);
    printfCallback("afiClockClampSenseCount           %d\n", loggedState.rootComplex.globalCounts.afiClockClampSenseCount);
    printfCallback("afiReadyForPowerDownSenseCount    %d\n", loggedState.rootComplex.globalCounts.afiReadyForPowerDownSenseCount);
    printfCallback("afiPeerToPeerErrorCount           %d\n", loggedState.rootComplex.globalCounts.afiPeerToPeerErrorCount);
    printfCallback("intVectorWakeCount                %d\n", loggedState.rootComplex.globalCounts.intVectorWakeCount);
    printfCallback("intVectorMsiCount                 %d\n", loggedState.rootComplex.globalCounts.intVectorMsiCount);
    printfCallback("intVectorIntCount                 %d\n", loggedState.rootComplex.globalCounts.intVectorIntCount);
    printfCallback("pllResistorCalibrationValue       %d\n", loggedState.rootComplex.pllResistorCalibrationValue);

    for(int port = 0; port < nn::pcie::MaxSwitchPortCount; port++)
    {
        printfCallback("Port %d:\n", port);
        printfCallback("  speed                           %s\n",   speedNames[loggedState.rootComplex.ports[port].speed]);
        printfCallback("  flagSet                         0x%x\n", loggedState.rootComplex.ports[port].flagSet);
        printfCallback("  portResetTimeInUs               %d\n",   loggedState.rootComplex.ports[port].portResetTimeInUs);
        printfCallback("  irqCount                        %d\n",   loggedState.rootComplex.ports[port].irqCount);
        printfCallback("  afiMsgPmeCount                  %d\n",   loggedState.rootComplex.ports[port].afiMsgPmeCount);
        printfCallback("  afiMsgIntxCount                 %d\n",   loggedState.rootComplex.ports[port].afiMsgIntxCount);
        printfCallback("  afiMsgPcieFatalErrorCount       %d\n",   loggedState.rootComplex.ports[port].afiMsgPcieFatalErrorCount);
        printfCallback("  afiMsgPcieNonFatalErrorCount    %d\n",   loggedState.rootComplex.ports[port].afiMsgPcieNonFatalErrorCount);
        printfCallback("  afiMsgPcieCorrectableErrorCount %d\n",   loggedState.rootComplex.ports[port].afiMsgPcieCorrectableErrorCount);
        printfCallback("  afiMsgPcieRootPortIntCount      %d\n",   loggedState.rootComplex.ports[port].afiMsgPcieRootPortIntCount);
        printfCallback("  afiMsgHotplugCount              %d\n",   loggedState.rootComplex.ports[port].afiMsgHotplugCount);
    }

    for(int functionIndex = 0; functionIndex < nn::pcie::MaxSwitchPortCount; functionIndex++)
    {
        if(loggedState.functions[functionIndex].functionHandle == InvalidFunctionHandle) continue;
        printfCallback("Function %d\n", functionIndex);
        printfCallback("  functionHandle   0x%x\n", loggedState.functions[functionIndex].functionHandle);
        printfCallback("  busNum           0x%x\n", loggedState.functions[functionIndex].busNum);
        printfCallback("  deviceNum        0x%x\n", loggedState.functions[functionIndex].deviceNum);
        printfCallback("  funcNum          0x%x\n", loggedState.functions[functionIndex].funcNum);
        printfCallback("  vendorId         0x%x\n", loggedState.functions[functionIndex].vendorId);
        printfCallback("  deviceId         0x%x\n", loggedState.functions[functionIndex].deviceId);
        printfCallback("  headerType       0x%x\n", loggedState.functions[functionIndex].headerType);
        printfCallback("  classCode        0x%x\n", loggedState.functions[functionIndex].classCode);
        printfCallback("  revision         0x%x\n", loggedState.functions[functionIndex].revision);
        printfCallback("  speed            %s\n",   speedNames[loggedState.functions[functionIndex].speed]);
        printfCallback("  maxNumMsiVectors 0x%x\n", loggedState.functions[functionIndex].maxNumMsiVectors);
        printfCallback("  pmState          %s\n", pmStateNames[loggedState.functions[functionIndex].pmState]);
        printfCallback("  isAcquired       %s\n", loggedState.functions[functionIndex].isAcquired ? "true" : "false");
    }
}

Result GetLoggedStateEvent(nn::os::SystemEventType* pOutLoggedStateEvent,
                           nn::os::EventClearMode eventClearMode,
                           bool isStatisticReported) NN_NOEXCEPT
{
    Result result;
    nn::sf::NativeHandle sfLoggedStateEventHandle;
    NN_ABORT_UNLESS_NOT_NULL(pOutLoggedStateEvent);
    NN_SDK_ASSERT(g_LoggedStateManager);
    if((result = g_LoggedStateManager->GetLoggedStateEvent(&sfLoggedStateEventHandle, isStatisticReported)).IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutLoggedStateEvent, sfLoggedStateEventHandle.GetOsHandle(), sfLoggedStateEventHandle.IsManaged(), eventClearMode);
        sfLoggedStateEventHandle.Detach();
    }
    return result;
}

} // namespace pcie
} // namespace nn
