﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file    pci_TK1RootComplex.cpp
 * @brief   NVIDIA Tegra K1 PCIe Root Complex Implementation
 */

#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

#include "../pcie_PrivateIncludes.h"
#include "pcie_Tegra124RootComplex.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

Tegra124RootComplex::Tegra124RootComplex(Driver *pDriver, Config *pCfg) :
    RootComplex(pDriver, pCfg)
{
    m_ControllerType = ControllerType_Tegra124;
    for (int32_t index = 0; index < AFI_INT_TOTAL_VECTORS; index++)
    {
        m_port0IntIrqMgrs[index].SetPoolId(intPort0IrqMgrBasePoolId + index);
        m_port1IntIrqMgrs[index].SetPoolId(intPort1IrqMgrBasePoolId + index);
    }
}

Tegra124RootComplex::~Tegra124RootComplex()
{

}

void* Tegra124RootComplex::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Tegra124RootComplex");
}

void Tegra124RootComplex::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    nn::pcie::detail::MemoryFree(p, "Tegra124RootComplex");
}

Result Tegra124RootComplex::SetPowerGate(int32_t bitpos, bool trueToPower)
{
    uint32_t value;
    uint32_t mask = 1 << bitpos;
    uint32_t nextValue = (trueToPower) ? mask : 0;
    uint32_t timeoutInMs = 250;

    value = NN_PCI_APBDEV_PMC_PWRGATE_STATUS_REG & mask;

    // do the write only if no change, since the process is somewhat time consuming
    if ((value ^ nextValue) != 0)
    {
        NN_PCI_APBDEV_PMC_PWRGATE_TOGGLE_REG = APBDEV_PMC_PWRGATE_TOGGLE_START | bitpos;

        // wait for PMC to do it
        uint32_t waitingTime;
        for (waitingTime = 0; waitingTime < timeoutInMs; waitingTime += 1)
        {
            if (((NN_PCI_APBDEV_PMC_PWRGATE_STATUS_REG & mask) ^ nextValue) != 0)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
            else
            {
                break;
            }
        }
        if (waitingTime == timeoutInMs)
        {
            return ResultOperationTimeout();
        }
    }

    return ResultSuccess();
}

Result Tegra124RootComplex::SetReset(ResetType t, bool trueToPutIntoReset)
{
    uint32_t mask = 0;

    switch (t)
    {
    case PCIEXCLK_RST:
        mask = 1 << 10;
        break;
    case AFI_RST:
        mask = 1 << 8;
        break;
    case PCIE_RST:
        mask = 1 << 6;
        break;
    default:
        return ResultBadInternalArgument();
        break;
    }

    if (trueToPutIntoReset)
    {
        NN_PCI_CLK_RST_CONTROLLER_RST_DEV_U_SET_REG = mask;
    }
    else
    {
        NN_PCI_CLK_RST_CONTROLLER_RST_DEV_U_CLR_REG = mask;
    }

    return ResultSuccess();
}

Result Tegra124RootComplex::InitPCIePll()
{



    // enable PLLE_CML0_OEN (PCIe)
    NN_PCI_CLK_RST_CONTROLLER_PLLE_AUX_REG |= (1 << 0);

    // for now assume PLL already setup by boot



    return ResultSuccess();
}


Result Tegra124RootComplex::SetPhyEnable(bool enabled)
{
    Result result = ResultSuccess();

    if (enabled)
    {
        /* clear REFCLK_SEL fields to begin */
        NN_PCI_XUSB_PADCTL_IOPHY_PLL_P0_CTL1_REG &= ~XUSB_PADCTL_IOPHY_PLL_P0_CTL1_REFCLK_SEL_MASK;


        NN_PCI_XUSB_PADCTL_IOPHY_PLL_P0_CTL2_REG |= (XUSB_PADCTL_IOPHY_PLL_P0_CTL2_REFCLKBUF_EN |
                                                     XUSB_PADCTL_IOPHY_PLL_P0_CTL2_TXCLKREF_EN |
                                                     XUSB_PADCTL_IOPHY_PLL_P0_CTL2_TXCLKREF_SEL);

        /* Release PLL from reset, wait for lock */
        NN_PCI_XUSB_PADCTL_IOPHY_PLL_P0_CTL1_REG |= XUSB_PADCTL_IOPHY_PLL_P0_CTL1_PLL_RST;
        NN_PCIE_RETURN_UPON_ERROR(WaitPolledRegister32(&NN_PCI_XUSB_PADCTL_IOPHY_PLL_P0_CTL1_REG,
                                                       XUSB_PADCTL_IOPHY_PLL_P0_CTL1_PLL0_LOCKDET, false,
                                                       PllPollIntervalInUs, MaxPllWaitTimeInUs,
                                                       "Phy"));
    }
    else
    {
        /* Put  PLL into reset */
        NN_PCI_XUSB_PADCTL_IOPHY_PLL_P0_CTL1_REG &= ~XUSB_PADCTL_IOPHY_PLL_P0_CTL1_PLL_RST;
    }

    return result;
}

Result Tegra124RootComplex::DoPortOperation(DeviceNumber portNum, PortOperation op)
{
    Result result = ResultSuccess();
    volatile uint32_t *pPexCtrl = (portNum == 0) ? &NN_PCI_AFI_PEX0_CTRL_REG : &NN_PCI_AFI_PEX1_CTRL_REG;
    //volatile uint32_t *pPexSts = (portNum == 0) ? &NN_PCI_AFI_PEX0_STATUS_REG : &NN_PCI_AFI_PEX1_STATUS_REG;

    switch (op)
    {
    case PORT_OP_DISABLE:
        // put into reset
        *pPexCtrl &= ~AFI_PEX_CTRL_RST_L;
        // disable reference clock
        *pPexCtrl &= ~AFI_PEX_CTRL_REFCLK_EN;
        break;
    case PORT_OP_ENABLE:
        // enable reference clock
        *pPexCtrl |= (AFI_PEX_CTRL_CLKREQ_EN | AFI_PEX_CTRL_REFCLK_EN) | AFI_PEX_CTRL_REFCLK_OVR_EN;
        break;
    case PORT_OP_RESET:
        *pPexCtrl &= ~AFI_PEX_CTRL_RST_L;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2));
        *pPexCtrl |= AFI_PEX_CTRL_RST_L;
        break;
    case PORT_OP_GET_STATE:
        uint32_t regVal32;
        ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET, sizeof(uint32_t), &regVal32);
        if (!(regVal32 & RP_VEND_XP_DL_UP))
        {
            result = ResultPortLinkDown();
        }

        ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LINK_CONTROL_STATUS_OFFSET, sizeof(uint32_t), &regVal32);
        if (!(regVal32 & RP_LINK_CONTROL_STATUS_DL_LINK_ACTIVE))
        {
            result = ResultPortLinkDown();
        }
        break;
    case PORT_OP_INIT:
        // Initialize command register
        WriteConfigSpace(0, portNum, 0, StandardConfigOffset_Command, sizeof(uint16_t), 0);

        // Clear primary, secondary and subordinate bus numbers
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrimaryBusNumber, sizeof(uint32_t), 0);

        // Clear primary, secondary and subordinate bus numbers
        WriteConfigSpace(0, portNum, 0, StandardConfigOffset_CacheLineSize, sizeof(uint32_t), 0);

        // Set IO base and limits to default zero, also clear secondary-side device status
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoBase, sizeof(uint32_t), 0xf9000000);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoBaseUpper16, sizeof(uint16_t), 0);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_IoLimitUpper16, sizeof(uint16_t), 0);

        // Set MEM base and limit to defaults
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_MemoryBase, sizeof(uint16_t), 0x00000010);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_MemoryLimit, sizeof(uint16_t), 0);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchMemoryBase, sizeof(uint16_t), 0x00000010);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchMemoryLimit, sizeof(uint16_t), 0);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchBaseUpper32, sizeof(uint32_t), 0);
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_PrefetchLimitUpper32, sizeof(uint32_t), 0);

        // Set interrupt line and pin
        WriteConfigSpace(0, portNum, 0, Type0ConfigOffset_InterruptLine, sizeof(uint8_t), 0);
        WriteConfigSpace(0, portNum, 0, Type0ConfigOffset_InterruptPin, sizeof(uint8_t), 0);

        // Set bridge control to default
        WriteConfigSpace(0, portNum, 0, Type1ConfigOffset_BridgeControl, sizeof(uint16_t), 0);

        // Clear stale status
        WriteConfigSpace(0, portNum, 0, StandardConfigOffset_Status, sizeof(uint16_t),
                         StandardConfigValue_StatusDetectedParity | StandardConfigValue_StatusSigSystemError |
                         StandardConfigValue_StatusRecMasterAbort | StandardConfigValue_StatusRecTargetAbort |
                         StandardConfigValue_StatusSigTargetAbort | StandardConfigValue_StatusParity);

        // Make present status available to AFI
        ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t), &regVal32);
        regVal32 &= ~RP_PRIV_MISC_PRSNT_MAP_EP_ABSNT;
        regVal32 |= RP_PRIV_MISC_PRSNT_MAP_EP_PRSNT;
        WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t), regVal32);
        break;
    default:
        result = ResultBadInternalArgument();
        break;
    }
    return result;
}

// Enable various features of root port, not appropriate until link up
Result Tegra124RootComplex::EnablePortFeatures(DeviceNumber portNum)
{
    Result result = ResultSuccess();
    uint32_t regVal32;

    // Power mangagement settings
    // Enable clock clamping and change threshold for TMS clock clamping
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_PRIV_MISC_OFFSET, sizeof(uint32_t),
                     RP_PRIV_MISC_TMS_CLK_CLAMP_ENABLE | RP_PRIV_MISC_CTLR_CLK_CLAMP_ENABLE |
                     (0xf << RP_PRIV_MISC_TMS_CLK_CLAMP_THRESHOLD_SHIFT) |
                     (0xf << RP_PRIV_MISC_CTLR_CLK_CLAMP_THRESHOLD_SHIFT));

    // Enable ASPM - L1 state support by default
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 |= RP_VEND_XP1_LINK_PVT_CTL_L1_ASPM_SUPPORT;
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_OFFSET, sizeof(uint32_t), regVal32);

    // LTSSM wait for DLLP to finish before entering L1 or L2/L3
    // to avoid truncating of PM mesgs resulting in reciever errors
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_BIST_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 |= RP_VEND_XP_BIST_GOTO_L1_L2_AFTER_DLLP_DONE;
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_XP_BIST_OFFSET, sizeof(uint32_t), regVal32);

    // unhide AER capability
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_CTL1_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 |= RP_VEND_CTL1_ERPT;
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_VEND_CTL1_OFFSET, sizeof(uint32_t), regVal32);

    // work-around: WAR for Eye diagram failure on lanes for T124 platforms */
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_1_R2_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 |= RP_ECTL_1_R2_TX_CMADJ_1C;
    regVal32 |= RP_ECTL_1_R2_TX_DRV_CNTL_1C;
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_ECTL_1_R2_OFFSET, sizeof(uint32_t), regVal32);
    // work-around: Avoid warning during enumeration for invalid IRQ of RP */
    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_INTR_BCR_OFFSET, sizeof(uint32_t), &regVal32);
    regVal32 |= RP_INTR_BCR_INTR_LINE;
    WriteConfigSpace(0, portNum, 0, T_PCIE2_RP_INTR_BCR_OFFSET, sizeof(uint32_t), regVal32);

    return result;
}

Result Tegra124RootComplex::SetAspmEnable(EndpointFunction *pEf, bool isEnabled)
{
    return ResultSuccess();
}

Result Tegra124RootComplex::GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize)
{
    *pOutBase = IoVaHeapBase;
    *pOutSize = IoVaHeapSize;
    return ResultSuccess();
}

Result Tegra124RootComplex::PortControl(DeviceNumber devNum, PortCommand portCommand)
{
    NN_UNUSED(devNum);
    NN_UNUSED(portCommand);
    return ResultSuccess();
}

Result Tegra124RootComplex::InitAFIBars()
{
    Result result = ResultSuccess();
    do
    {
        // Disable unused AFI BARs
        NN_PCI_AFI_AXI_BAR2_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR3_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR4_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR5_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR6_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR7_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR8_SZ_REG = 0;
        NN_PCI_AFI_AXI_BAR2_START_REG = 0;
        NN_PCI_AFI_AXI_BAR3_START_REG = 0;
        NN_PCI_AFI_AXI_BAR4_START_REG = 0;
        NN_PCI_AFI_AXI_BAR5_START_REG = 0;
        NN_PCI_AFI_AXI_BAR6_START_REG = 0;
        NN_PCI_AFI_AXI_BAR7_START_REG = 0;
        NN_PCI_AFI_AXI_BAR8_START_REG = 0;
        NN_PCI_AFI_FPCI_BAR2_REG = 0;
        NN_PCI_AFI_FPCI_BAR3_REG = 0;
        NN_PCI_AFI_FPCI_BAR4_REG = 0;
        NN_PCI_AFI_FPCI_BAR5_REG = 0;
        NN_PCI_AFI_FPCI_BAR6_REG = 0;
        NN_PCI_AFI_FPCI_BAR7_REG = 0;
        NN_PCI_AFI_FPCI_BAR8_REG = 0;

        // Disable MSI for now
        NN_PCI_AFI_MSI_BAR_SZ_REG      = 0;
        NN_PCI_AFI_MSI_FPCI_BAR_ST_REG = 0;
        NN_PCI_AFI_MSI_AXI_BAR_ST_REG  = 0;

        // AFI BAR0: m_ConfigSpaceStart
        NN_PCI_AFI_FPCI_BAR0_REG       = AFI_FPCI_BAR_TYPE1_EXT_CFG;
        NN_PCI_AFI_AXI_BAR0_SZ_REG     = (Params_ConfigSpaceSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR0_START_REG  = m_ConfigSpaceStartPhys;

        // AFI BAR1: PCI_IO_SPACE
        NN_PCI_AFI_FPCI_BAR1_REG       = AFI_FPCI_BAR_MMMIO_CFG;
        NN_PCI_AFI_AXI_BAR1_SZ_REG     = (Params_IoSpaceSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR1_START_REG  = m_IoSpaceStartPhys;

        // AFI BAR2: PCI_PREF_MEM
        NN_PCI_AFI_FPCI_BAR2_REG       = (((m_PrefetchMemStartPhys >> 12) & 0xfffff) << 4) | 0x1;
        NN_PCI_AFI_AXI_BAR2_SZ_REG     = (Params_PrefetchMemSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR2_START_REG  = m_PrefetchMemStartPhys;

        // AFI BAR3: PCI_NONPREF_MEM
        NN_PCI_AFI_FPCI_BAR3_REG       = (((m_NoPrefetchMemStartPhys >> 12) & 0xfffff) << 4) | 0x1;
        NN_PCI_AFI_AXI_BAR3_SZ_REG     = (Params_NoPrefetchMemSize >> AFI_AXI_BAR_SZ_SHIFT);
        NN_PCI_AFI_AXI_BAR3_START_REG  = m_NoPrefetchMemStartPhys;

    } while (false);
    return result;
}

Result Tegra124RootComplex::InitAFIConfigReg()
{
    Result result = ResultSuccess();
    uint32_t pciCfgRegVal = 0;
    uint32_t afiFuseRegVal = 0;
    uint32_t xbar = 0;
    uint32_t laneMap = 0;
    for (int32_t portIndex = 0; portIndex < m_Config.numPorts; portIndex++)
    {
        laneMap |= ((m_Config.ports[portIndex].laneCount) << (portIndex * 8));
    }
    switch (laneMap)
    {
    case 0x0000104:
        xbar = AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_X4_X1;
        break;
    case 0x0000102:
        xbar = AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_X2_X1;
        break;
    default:
        result = ResultBadControllerConfig();
        break;
    }

    // Apply XBAR
    pciCfgRegVal = NN_PCI_AFI_PCIE_CONFIG_REG;
    pciCfgRegVal &= ~AFI_PCIE_CONFIG_SM2TMS0_XBAR_CONFIG_MASK;
    pciCfgRegVal |= xbar;

    // Initialize ports
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        DoPortOperation(portNum, PORT_OP_INIT);
        if (m_Config.ports[portNum].isEnabled)
        {
            pciCfgRegVal &= ~(1 << ((portNum) + 1));
        }
        else
        {
            pciCfgRegVal |= (1 << ((portNum) + 1));
        }
    }
    NN_PCI_AFI_PCIE_CONFIG_REG = pciCfgRegVal;

    // Allow PADS CLKREQ to control PLLE,
    //   All CLKREQs going inactive indicate DISABLE to PLLE */
    uint32_t plleControlRegVal = NN_PCI_AFI_PLLE_CONTROL_REG;
    plleControlRegVal |= AFI_PLLE_CONTROL_PADS2PLLE_CONTROL_EN;
    plleControlRegVal &= ~AFI_PLLE_CONTROL_BYPASS_PADS2PLLE_CONTROL;
    NN_PCI_AFI_PLLE_CONTROL_REG = plleControlRegVal;

    // Speed limiting
    afiFuseRegVal = NN_PCI_AFI_FUSE_REG;
    if (m_Config.speedLimit < BusSpeed_EGen2)
    {
        afiFuseRegVal |= AFI_FUSE_PCIE_T0_GEN2_DIS;
    }
    else
    {
        afiFuseRegVal &= ~AFI_FUSE_PCIE_T0_GEN2_DIS;
    }
    NN_PCI_AFI_FUSE_REG = afiFuseRegVal;

    return result;
}

Result Tegra124RootComplex::EnableAFI()
{
    Result result = ResultSuccess();

    // Top level PCIe enable
    NN_PCI_AFI_CONFIGURATION_REG |= AFI_CONFIGURATION_EN_FPCI;

    // Setup interrupts
    NN_PCI_AFI_AFI_INTR_ENABLE_REG  = AFI_INTR_EN_INI_SLVERR | AFI_INTR_EN_INI_DECERR |
        AFI_INTR_EN_TGT_SLVERR | AFI_INTR_EN_TGT_WRERR  | AFI_INTR_EN_PRSNT_SENSE;
    NN_PCI_AFI_SM_INTR_ENABLE_REG = 0xffffffff & ~AFI_SM_INTR_INTA_DEASSERT;
    NN_PCI_AFI_INTR_MASK_REG = AFI_INTR_MASK_INT_MASK | AFI_INTR_MASK_MSI_MASK;

    // No exceptions
    NN_PCI_AFI_FPCI_ERROR_MASKS_REG = 0;

    return result;
}

Result Tegra124RootComplex::InitMSI()
{
    Result result = ResultSuccess();

    if ((m_pMsiBuffer = nn::pcie::detail::MemoryAllocAligned(Params_MsiBufferSize,
                                                             Params_MsiBufferSize,
                                                             "Tegra124RootComplex Msi buffer")) != NULL)
    {
        memset(m_pMsiBuffer, 0, Params_MsiBufferSize);
        nn::dd::FlushDataCache(m_pMsiBuffer, Params_MsiBufferSize);
#if !defined NN_PCI_SMMU_DISABLED
        NN_PCIE_ABORT_UPON_ERROR(
            nn::dd::MapDeviceAddressSpaceNotAligned(
                &m_DeviceAddressSpace,
                nn::dd::GetCurrentProcessHandle(),
                reinterpret_cast<uintptr_t>(m_pMsiBuffer),
                Params_MsiBufferSize,
                IoVaInternalBase,
                nn::dd::MemoryPermission_ReadWrite));
#endif
        NN_PCI_AFI_MSI_FPCI_BAR_ST_REG = (IoVaInternalBase >> 8);
        NN_PCI_AFI_MSI_AXI_BAR_ST_REG  = IoVaInternalBase;
        NN_PCI_AFI_MSI_BAR_SZ_REG      = 1;

        // These get enabled on-demand later
        NN_PCI_AFI_MSI_EN_VEC0_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC1_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC2_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC3_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC4_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC5_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC6_REG     = 0;
        NN_PCI_AFI_MSI_EN_VEC7_REG     = 0;
    }
    else
    {
        result = ResultMemAllocFailure();
    }

    return result;
}

void Tegra124RootComplex::ResolveMemMap()
{
    Result result = ResultSuccess();

    // Get virtual mapping of bases
    NN_PCIE_ABORT_IF_ZERO(m_PcieA1Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA1Addr, NvPhysIo_PcieA1Size));
    NN_PCIE_ABORT_IF_ZERO(m_PcieA2Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA2Addr, NvPhysIo_PcieA2Size));
    NN_PCIE_ABORT_IF_ZERO(m_PcieA3Base = nn::dd::QueryIoMappingAddress(NvPhysIo_PcieA3Addr, NvPhysIo_PcieA3Size));
    NN_PCIE_ABORT_IF_ZERO(m_ClkRstBase = nn::dd::QueryIoMappingAddress(NvPhysIo_ClkRstAddr, NvPhysIo_ClkRstSize));
    NN_PCIE_ABORT_IF_ZERO(m_RtcBase = nn::dd::QueryIoMappingAddress(NvPhysIo_RtcAddr, NvPhysIo_RtcSize));
    NN_PCIE_ABORT_IF_ZERO(m_XusbPadctlBase = nn::dd::QueryIoMappingAddress(NvPhysIo_XusbPadctlAddr, NvPhysIo_XusbPadctlSize));

    // Derived base addresses of specific blocks
    m_PcieCtrl0CfgBase = m_PcieA1Base + 0x00000000;
    m_PcieCtrl1CfgBase = m_PcieA1Base + 0x00001000;
    m_PciePca0Base     = m_PcieA1Base + 0x00002000;
    m_PciePca1Base     = m_PcieA1Base + 0x00002100;
    m_PciePadsBase     = m_PcieA1Base + 0x00003000;
    m_PcieAfiBase      = m_PcieA1Base + 0x00003800;
    m_PmcBase          = m_RtcBase    + 0x400;

    // Definition of fundamental config/IO/MEM regions
    m_ConfigSpaceStart       = reinterpret_cast<uint8_t *>(m_PcieA2Base);
    m_ConfigSpaceStartPhys   = NvPhysIo_PcieA2Addr;
    m_ConfigSpaceEnd         = m_ConfigSpaceStart + Params_ConfigSpaceSize;

    m_IoSpaceStart           = reinterpret_cast<uint8_t *>(m_ConfigSpaceEnd);
    m_IoSpaceStartPhys       = NvPhysIo_PcieA2Addr + Params_ConfigSpaceSize;

    m_PrefetchMemStart       = reinterpret_cast<uint8_t *>(m_PcieA3Base);
    m_PrefetchMemStartPhys   = NvPhysIo_PcieA3Addr;

    m_NoPrefetchMemStart     = reinterpret_cast<uint8_t *>(m_PcieA3Base + Params_PrefetchMemSize);
    m_NoPrefetchMemStartPhys = NvPhysIo_PcieA3Addr + Params_PrefetchMemSize;

    NN_PCIE_LOG_VERBOSE("Tegra124RootComplex::ResolveMemMap():\n");
    NN_PCIE_LOG_VERBOSE("  m_ConfigSpaceStart       = 0x%p\n", m_ConfigSpaceStart);
    NN_PCIE_LOG_VERBOSE("  m_ConfigSpaceStartPhys   = 0x%p\n", reinterpret_cast<void*>(m_ConfigSpaceStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_IoSpaceStart           = 0x%p\n", m_IoSpaceStart);
    NN_PCIE_LOG_VERBOSE("  m_IoSpaceStartPhys       = 0x%p\n", reinterpret_cast<void*>(m_IoSpaceStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_PrefetchMemStart       = 0x%p\n", m_PrefetchMemStart);
    NN_PCIE_LOG_VERBOSE("  m_PrefetchMemStartPhys   = 0x%p\n", reinterpret_cast<void*>(m_PrefetchMemStartPhys));
    NN_PCIE_LOG_VERBOSE("  m_NoPrefetchMemStart     = 0x%p\n", m_NoPrefetchMemStart);
    NN_PCIE_LOG_VERBOSE("  m_NoPrefetchMemStartPhys = 0x%p\n", reinterpret_cast<void*>(m_NoPrefetchMemStartPhys));

#if !defined NN_PCI_SMMU_DISABLED
    NN_PCIE_ABORT_UPON_ERROR(nn::dd::CreateDeviceAddressSpace(&m_DeviceAddressSpace,
                                                              0x100000000ull));
    NN_PCIE_ABORT_UPON_ERROR(nn::dd::AttachDeviceAddressSpace(&m_DeviceAddressSpace,
                                                              nn::dd::DeviceName::DeviceName_Afi));
#endif

}


Result Tegra124RootComplex::InitInterruptVectors()
{
    Result result = ResultSuccess();

    do
    {
        // MSI
        nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi],
                                         MsiInterruptName,
                                         nn::os::EventClearMode_ManualClear);
        NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi], IntVector_Msi));

        // INT
        nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Int],
                                         MainInterruptName,
                                         nn::os::EventClearMode_ManualClear);
        NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Int], IntVector_Int));

        // WAKE
        nn::os::InitializeInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake],
                                         WakeInterruptName,
                                         nn::os::EventClearMode_ManualClear);
        NN_PCIE_BREAK_UPON_ERROR(m_pDriver->RegisterInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake], IntVector_Wake));
    }while (false);

    return result;
}

uint32_t Tegra124RootComplex::GetEndpointFunctionEnableMask(int32_t baseResourceIndex)
{
    uint32_t enableMask = 0;
    for(int32_t i=0; i < nn::pcie::MaxIrqVectorsPerDeviceFunction; i++)
    {
        int32_t resourceIndex = baseResourceIndex + i;
        int32_t vecBit = 1 << (resourceIndex % AFI_MSI_VECTORS_PER_REGISTER);
        volatile uint32_t *pVecEnReg = &NN_PCI_AFI_MSI_EN_VEC0_REG + (resourceIndex >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
        uint32_t vectorEnMask = *pVecEnReg;
        if(vectorEnMask & vecBit) enableMask |= (1 << i);
    }
    return enableMask;
}

volatile uint32_t* Tegra124RootComplex::ComputeConfigSpaceAddress(BusNumber busNum, DeviceNumber devNum,
                                                                  FunctionNumber funcNum, int32_t where)
{
    volatile uint32_t *pDWordReg = NULL;

    if (busNum == 0)
    {
        if (funcNum == 0)
        {
            switch (devNum)
            {
            case 0:
                pDWordReg = (volatile uint32_t *)(m_PcieCtrl0CfgBase + (where & ~3));
                break;
            case 1:
                pDWordReg = (volatile uint32_t *)(m_PcieCtrl1CfgBase + (where & ~3));
                break;
            default:
                break;
            }
        }
    }
    else
    {
        //[27:24] extended register number
        //[23:16] bus number
        //[15:11] device number
        //[10: 8] function number
        //[ 7: 0] register number
        uint32_t offset = ((where & 0xf00) << 16) | (busNum << 16) | (devNum << 11) |
            ((funcNum & 7) << 8) | (where & 0xfc);
        pDWordReg = (volatile uint32_t *)(m_ConfigSpaceStart + offset);

        if (((ResourceAddr)pDWordReg) >= m_ConfigSpaceEnd)
        {
            pDWordReg = NULL;
        }
    }
    return pDWordReg;
}

Result Tegra124RootComplex::ReadConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                            int32_t where, uint32_t size, void *pVal)
{
    Result result = ResultSuccess();
    volatile uint32_t *pDWordReg = NULL;

    if ((pDWordReg = ComputeConfigSpaceAddress(busNum, devNum, funcNum, where)) != NULL)
    {
        uint32_t dWordValue = *pDWordReg;
        switch (size)
        {
        case 1:
            *((uint8_t *)pVal) = (uint8_t)((dWordValue >> (8 * (where & 3))) & 0xff);
            break;
        case 2:
            *((uint16_t *)pVal) = (uint16_t)((dWordValue >> (8 * (where & 3))) & 0xffff);
            break;
        case 4:
            *((uint32_t *)pVal) = dWordValue;
            break;
        default:
            result = ResultInvalidRegisterSize();
            break;
        }
    }
    else
    {
        memset(pVal, 0, size);
        result = ResultNoDevice();
    }

    return result;
}

Result Tegra124RootComplex::WriteConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                             int32_t where, uint32_t size, uint32_t val)
{
    Result result = ResultSuccess();
    volatile uint32_t *pDWordReg = NULL;

    if ((pDWordReg = ComputeConfigSpaceAddress(busNum, devNum, funcNum, where)) != NULL)
    {
        uint32_t dWordValue;
        switch (size)
        {
        case 1:
            dWordValue = *pDWordReg & ~(0xff << ((where & 0x3) * 8));
            dWordValue |= (val << ((where & 0x3) * 8));
            *pDWordReg = dWordValue;
            break;
        case 2:
            dWordValue = *pDWordReg & ~(0xffff << ((where & 0x3) * 8));
            dWordValue |= (val << ((where & 0x3) * 8));
            *pDWordReg = dWordValue;
            break;
        case 4:
            *pDWordReg = val;
            break;
        default:
            result = ResultInvalidRegisterSize();
            break;
        }
    }
    else
    {
        result = ResultNoDevice();
    }

    return result;
}

void Tegra124RootComplex::GetRegions(Region *pIo, Region *pPrefetchMem, Region *pNoPrefetchMem)
{
    pIo->resourceAddr            = m_IoSpaceStart;
    pIo->busAddr                 = m_IoSpaceStartPhys;
    pIo->size                    = Params_IoSpaceSize;

    pPrefetchMem->resourceAddr   = m_PrefetchMemStart;
    pPrefetchMem->busAddr        = m_PrefetchMemStartPhys;
    pPrefetchMem->size           = Params_PrefetchMemSize;

    pNoPrefetchMem->resourceAddr = m_NoPrefetchMemStart;
    pNoPrefetchMem->busAddr      = m_NoPrefetchMemStartPhys;
    pNoPrefetchMem->size         = Params_NoPrefetchMemSize;
}


Result Tegra124RootComplex::AcquireIrq(EndpointFunction *pEp, IrqOptions *pOptions, IrqProfile *pRetProfile)
{
    Result result = ResultBadInternalArgument();
    IrqHandle irqHandle = -1;
    if (pOptions->irqType == IrqType_Msi)
    {
        result = m_msiIrqMgr.Acquire(pEp, &irqHandle);
    }
    else if (pOptions->irqType == IrqType_IntX)
    {
        Bus *pBus = pEp->GetBus();
        BridgeFunction *pBf = pBus->GetRootBridgeFunction();
        if (pBf != NULL)
        {
            DeviceNumber devNum = pBf->GetDevNum();
            uint8_t pinIndex = pOptions->type.intx.intPin - 1;
            if ((devNum < m_Config.numPorts) &&
                (pinIndex < AFI_INT_TOTAL_VECTORS))
            {
                switch (devNum)
                {
                case 0:
                    result = m_port0IntIrqMgrs[pinIndex].Acquire(pEp, &irqHandle);
                    break;
                case 1:
                    result = m_port1IntIrqMgrs[pinIndex].Acquire(pEp, &irqHandle);
                    break;
                default:
                    break;
                }
            }
        }
    }
    if (result.IsSuccess())
    {
        GetIrqProfile(irqHandle, pRetProfile);
    }
    return result;
}

Result Tegra124RootComplex::ReleaseIrq(IrqHandle irqHandle)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        for(int32_t index=0; index < nn::pcie::MaxIrqVectorsPerDeviceFunction; index++)
        {
            SetIrqEnable(irqHandle, index, false);
        }
        result = m_msiIrqMgr.Release(irqHandle);
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        int32_t index = poolId - intPort0IrqMgrBasePoolId;
        m_port0IntIrqMgrs[index].Release(irqHandle);
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        int32_t index = poolId - intPort1IrqMgrBasePoolId;
        m_port1IntIrqMgrs[index].Release(irqHandle);
    }
    return result;
}

Result Tegra124RootComplex::SetIrqEnable(IrqHandle irqHandle, int32_t index, bool irqEnabled)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        int32_t resourceIndex;
        if ((result = m_msiIrqMgr.IrqHandleLookup(&resourceIndex, irqHandle)).IsSuccess())
        {
            resourceIndex += index;
            if ((resourceIndex < AFI_MSI_TOTAL_VECTORS) && (resourceIndex >= 0))
            {
                uint32_t vecMask = 1 << (resourceIndex % AFI_MSI_VECTORS_PER_REGISTER);
                volatile uint32_t *pVecEnReg = &NN_PCI_AFI_MSI_EN_VEC0_REG + (resourceIndex >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
                *pVecEnReg &= ~vecMask;
                *pVecEnReg |= ((irqEnabled) ? vecMask : 0);
                result = ResultSuccess();
            }
        }
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        result = ResultSuccess();
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        result = ResultSuccess();
    }
    return result;
}

Result Tegra124RootComplex::GetIrqProfile(IrqHandle irqHandle, IrqProfile *pRetProfile)
{
    Result result = ResultBadInternalArgument();
    int32_t poolId = m_msiIrqMgr.GetPoolId(irqHandle);
    if (poolId == msiIrqMgrPoolId)
    {
        int32_t resourceIndex;
        if ((result = m_msiIrqMgr.IrqHandleLookup(&resourceIndex, irqHandle)).IsSuccess())
        {
            if ((resourceIndex < AFI_MSI_TOTAL_VECTORS) && (resourceIndex >= 0))
            {
                pRetProfile->irqHandle  = irqHandle;
                pRetProfile->irqType    = IrqType_Msi;
                pRetProfile->enableMask = GetEndpointFunctionEnableMask(resourceIndex);
                pRetProfile->type.msi.messageAddress = (BusAddress)NN_PCI_AFI_MSI_AXI_BAR_ST_REG;
                pRetProfile->type.msi.messageData    = (uint16_t)resourceIndex;
                result = ResultSuccess();
            }
        }
    }
    else if ((poolId >= intPort0IrqMgrBasePoolId) &&
             (poolId < (intPort0IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        pRetProfile->enableMask = 1;
        pRetProfile->irqHandle  = irqHandle;
        pRetProfile->irqType    = IrqType_IntX;
        result = ResultSuccess();
    }
    else if ((poolId >= intPort1IrqMgrBasePoolId) &&
             (poolId < (intPort1IrqMgrBasePoolId + AFI_INT_TOTAL_VECTORS)))
    {
        pRetProfile->enableMask = 1;
        pRetProfile->irqHandle  = irqHandle;
        pRetProfile->irqType    = IrqType_IntX;
        result = ResultSuccess();
    }
    return result;
}

void Tegra124RootComplex::HandleInterrupt(IntVector intVector)
{
    switch (intVector)
    {
    case IntVector_Wake:
        break;
    case IntVector_Msi:
        HandleMSIInterrupt();
        break;
    case IntVector_Int:
        HandleINTInterrupt();
        break;
    default:
        break;
    }
}

void Tegra124RootComplex::SetErrorReportingEnable(bool enable)
{
    if (enable)
    {
        NN_PCI_AFI_AFI_INTR_ENABLE_REG  |= AFI_INTR_EN_TGT_DECERR | AFI_INTR_EN_DFPCI_DECERR;
    }
    else
    {
        NN_PCI_AFI_AFI_INTR_ENABLE_REG  &= (~AFI_INTR_EN_TGT_DECERR | AFI_INTR_EN_DFPCI_DECERR);
    }
}

void Tegra124RootComplex::HandleINTInterrupt()
{
    const char *debugIntCodeString = NULL;
    uint32_t code = NN_PCI_AFI_INTR_CODE_REG & AFI_INTR_CODE_MASK;
    uint32_t sigRegVal = NN_PCI_AFI_INTR_SIGNATURE_REG;
    NN_PCI_AFI_INTR_CODE_REG = 0;
    switch (code)
    {
    case  AFI_INTR_CODE_INI_SLVERR:
        debugIntCodeString = "AFI_INTR_CODE_INI_SLVERR";
        break;
    case  AFI_INTR_CODE_INI_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_INI_DECERR";
        break;
    case  AFI_INTR_CODE_TGT_SLVERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_SLVERR";
        break;
    case  AFI_INTR_CODE_TGT_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_DECERR";
        break;
    case  AFI_INTR_CODE_TGT_WRERR:
        debugIntCodeString = "AFI_INTR_CODE_TGT_WRERR";
        break;
    case  AFI_INTR_CODE_SM_MSG:
        //debugIntCodeString = "AFI_INTR_CODE_SM_MSG";
        HandleSidebandMessageInterrupt(sigRegVal);
        break;
    case  AFI_INTR_CODE_DFPCI_DECERR:
        debugIntCodeString = "AFI_INTR_CODE_DFPCI_DECERR";
        break;
    case  AFI_INTR_CODE_FPCI_TIMEOUT:
        debugIntCodeString = "AFI_INTR_CODE_FPCI_TIMEOUT";
        break;
    case  AFI_INTR_CODE_PE_PRSNT_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_PE_PRSNT_SENSE";
        break;
    case  AFI_INTR_CODE_PE_CLKREQ_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_PE_CLKREQ_SENSE";
        break;
    case  AFI_INTR_CODE_CLKCLAMP_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_CLKCLAMP_SENSE";
        break;
    case  AFI_INTR_CODE_RDY4PD_SENSE:
        debugIntCodeString = "AFI_INTR_CODE_RDY4PD_SENSE";
        break;
    case  AFI_INTR_P2P_ERROR:
        debugIntCodeString = "AFI_INTR_P2P_ERROR";
        break;
    default:
        break;
    }

    //NN_PCI_AFI_INTR_SIGNATURE_REG = 0;
    //NN_PCI_AFI_INTR_CODE_REG = 0;

    if (debugIntCodeString != NULL)
    {
        NN_PCIE_LOG_INFO(" %s.\n", debugIntCodeString);
    }
}

void Tegra124RootComplex::DispatchIntx(PortIntxIrqMgrType *pMgr)
{
    IrqManagerEntry entryList[MaxFunctionsPerIrq];
    int32_t numEntries = pMgr->GetAllAcquired(entryList);
    for (int32_t index = 0; index < numEntries; index++)
    {
        IrqManagerEntry *pE = entryList + index;
        EndpointFunction *pEp = (EndpointFunction *)pE->context;
        pEp->HandleIrq(IrqType_IntX, pE->irqHandle);
    }
}

void Tegra124RootComplex::HandleSidebandMessageInterrupt(uint32_t sigRegVal)
{
    uint32_t regVal32;
    uint32_t afiMsg = NN_PCI_AFI_MSG_REG;
    DeviceNumber devNum = (afiMsg & AFI_MSG_1_MASK) ? 1 : 0;

    if ((AFI_MSG_1_ENDPOINT_PME_MASK | AFI_MSG_0_ENDPOINT_PME_MASK) & afiMsg)
    {
        // clear PME status
        ReadConfigSpace(0, devNum, 0, T_PCIE2_RP_RSR_OFFSET, sizeof(uint32_t), &regVal32);
        regVal32 |= RP_RSR_PMESTAT_MASK;
        WriteConfigSpace(0, devNum, 0, T_PCIE2_RP_RSR_OFFSET, sizeof(uint32_t), regVal32);
    }

    // INTx dispatch
    if (AFI_MSG_0_PCIE_INTERRUPT_ALL_MASK & afiMsg)
    {
        for (uint8_t intPinIndex = 0; intPinIndex < AFI_INT_TOTAL_VECTORS; intPinIndex++)
        {
            if ((AFI_MSG_0_PCIE_INTERRUPT_A_MASK << intPinIndex) & afiMsg)
            {
                DispatchIntx(m_port0IntIrqMgrs + intPinIndex);
            }
        }
    }
    else if (AFI_MSG_1_PCIE_INTERRUPT_ALL_MASK & afiMsg)
    {
        for (uint8_t intPinIndex = 0; intPinIndex < AFI_INT_TOTAL_VECTORS; intPinIndex++)
        {
            if ((AFI_MSG_1_PCIE_INTERRUPT_A_MASK << intPinIndex) & afiMsg)
            {
                DispatchIntx(m_port1IntIrqMgrs + intPinIndex);
            }
        }
    }

    if ((AFI_MSG_1_PCIE_FATAL_ERROR_MASK | AFI_MSG_0_PCIE_FATAL_ERROR_MASK) & afiMsg)
    {
    }
    if ((AFI_MSG_1_PCIE_NONFATAL_ERROR_MASK | AFI_MSG_0_PCIE_NONFATAL_ERROR_MASK) & afiMsg)
    {
    }
    if ((AFI_MSG_1_PCIE_CORRECTABLE_ERROR_MASK | AFI_MSG_0_PCIE_CORRECTABLE_ERROR_MASK) & afiMsg)
    {
    }
    if ((AFI_MSG_1_PCIE_ROOT_PORT_INTERRUPT_MASK | AFI_MSG_0_PCIE_ROOT_PORT_INTERRUPT_MASK) & afiMsg)
    {
    }
    if ((AFI_MSG_1_HOTPLUG_MASK | AFI_MSG_0_HOTPLUG_MASK) & afiMsg)
    {
    }
}

void Tegra124RootComplex::HandleMSIInterrupt()
{
    // for all AFI_MSI_VECx_REG registers
    for (int32_t irqBaseNum = 0; irqBaseNum < AFI_MSI_TOTAL_VECTORS; irqBaseNum += AFI_MSI_VECTORS_PER_REGISTER)
    {
        volatile uint32_t *pVecReg = &NN_PCI_AFI_MSI_VEC0_REG + (irqBaseNum >> AFI_MSI_VECTORS_PER_REGISTER_POWEROFTWO);
        uint32_t vectorMask;
        while((vectorMask = *pVecReg) != 0)
        {
            // for all pending bits in the register
            for (int32_t bitNum = 0; (bitNum < AFI_MSI_VECTORS_PER_REGISTER) && (vectorMask != 0); bitNum++)
            {
                uint32_t bitMask = 1 << bitNum;
                if (bitMask & vectorMask)
                {
                    // clear pending
                    *pVecReg = bitMask;
                    vectorMask &= ~bitMask;

                    // dispatch
                    EndpointFunction *pEp = NULL;
                    int32_t offset = -1;
                    m_msiIrqMgr.ResourceIndexLookup(&pEp, &offset, irqBaseNum + bitNum);
                    if (pEp != NULL)
                    {
                        pEp->HandleIrq(IrqType_Msi, offset);
                    }
                }
            }
        }
    }
}

Result Tegra124RootComplex::Initialize()
{
    Result result = ResultSuccess();

    // Run base class initization first
    RootComplex::Initialize();

    // I/O VA Manager
    m_IoVaManager.Initialize(IoVaHeapBase, IoVaHeapSize);

    // Resolve root complex memory map
    ResolveMemMap();

    InitInterruptVectors();

    // Disable ports to begin
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        DoPortOperation(portNum, PORT_OP_DISABLE);
    }

    // AFI PCIE Config
    NN_PCIE_ABORT_UPON_ERROR(InitAFIConfigReg());

    // Power up
    result = SetPowerGate(APBDEV_PMC_PWRGATE_PCI_BIT, true);
    if (!result.IsSuccess())
    {
        NN_PCIE_LOG_ERROR("failed to enable power\n");
        return result;
    }

    // Power-up PEX clock bias
    NN_PCI_AFI_PEXBIAS_CTRL_REG = 0;

    // AFI BARs (not the same as PCI device function BARs)
    NN_PCIE_ABORT_UPON_ERROR(InitAFIBars());

    // MSI
    NN_PCIE_ABORT_UPON_ERROR(InitMSI());

    // Aperatures as MMIO
    NN_PCI_APBDEV_PMC_GLB_AMAP_CFG_REG &= ~(APBDEV_PMC_GLB_AMAP_CFG_PCIE_A3_DRAM |
                                            APBDEV_PMC_GLB_AMAP_CFG_PCIE_A2_DRAM |
                                            APBDEV_PMC_GLB_AMAP_CFG_PCIE_A1_DRAM);

    // Enable the phy
    result = SetPhyEnable(true);
    if (!result.IsSuccess())
    {
        return result;
    }

    // Release PCIEXCLK
    SetReset(PCIEXCLK_RST, false);

    // Enable
    NN_PCIE_ABORT_UPON_ERROR(EnableAFI());

    // Bring up port links
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        if (m_Config.ports[portNum].isEnabled)
        {
            int32_t retries = 3;
            bool linkUp = false;
            DoPortOperation(portNum, PORT_OP_ENABLE);
            do
            {
                DoPortOperation(portNum, PORT_OP_RESET);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
                if (DoPortOperation(portNum, PORT_OP_GET_STATE).IsSuccess())
                {
                    const char *pSpeed = "?";
                    uint32_t regVal32 = 0;
                    linkUp = true;
                    NN_UNUSED(pSpeed);
                    EnablePortFeatures(portNum);
                    ReadConfigSpace(0, portNum, 0, T_PCIE2_RP_LINK_CONTROL_STATUS_OFFSET, sizeof(uint32_t), &regVal32);
                    switch ((regVal32 & RP_LINK_CONTROL_STATUS_SPEED_MASK) >> RP_LINK_CONTROL_STATUS_SPEED_SHIFT)
                    {
                    case PcieValue_LinkStatusCurrentLinkSpeed_2_5GB:
                        pSpeed = "GEN1";
                        break;
                    case PcieValue_LinkStatusCurrentLinkSpeed_5_0GB:
                        pSpeed = "GEN2";
                        break;
                    default:
                        break;
                    }
                    NN_PCIE_LOG_INFO("Tegra124RootComplex::Initialize() port %d UP, speed %s.\n", portNum, pSpeed);
                    break;
                }
                else
                {
                    NN_PCIE_LOG_INFO("Tegra124RootComplex::Initialize() port %d DOWN, retrying...\n", portNum);
                }
            }while (!linkUp && retries-- > 0);
            if (!linkUp && retries <= 0)
            {
                NN_PCIE_LOG_INFO("Tegra124RootComplex::Initialize() port %d DOWN.\n", portNum);
                DoPortOperation(portNum, PORT_OP_DISABLE);
            }
        }
    }

    return result;
}

Result Tegra124RootComplex::Finalize()
{
    Result result = ResultSuccess();

    // Finalize base class first, as this operation may cause hardware to be touched
    RootComplex::Finalize();

    // Disable interrupts
    NN_PCI_AFI_AFI_INTR_ENABLE_REG = 0;
    NN_PCI_AFI_SM_INTR_ENABLE_REG = 0;
    NN_PCI_AFI_INTR_MASK_REG = 0;
    NN_PCI_AFI_FPCI_ERROR_MASKS_REG = 0;

    // Unregister interrupts
    nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Msi]);
    nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Int]);
    nn::os::FinalizeInterruptEvent(&m_InterruptEvent[InterruptEvent_Wake]);

    // Disable ports
    for (DeviceNumber portNum = 0; portNum < m_Config.numPorts; portNum++)
    {
        DoPortOperation(portNum, PORT_OP_DISABLE);
    }

    // Top level PCIe disable
    NN_PCI_AFI_CONFIGURATION_REG &= ~AFI_CONFIGURATION_EN_FPCI;

    // Teardown MSI related
    if (m_pMsiBuffer != NULL)
    {
#if !defined NN_PCI_SMMU_DISABLED
        nn::dd::
            UnmapDeviceAddressSpace(&m_DeviceAddressSpace,
                                    nn::dd::GetCurrentProcessHandle(),
                                    reinterpret_cast<uintptr_t>(m_pMsiBuffer),
                                    Params_MsiBufferSize,
                                    IoVaInternalBase);
#endif
        nn::pcie::detail::MemoryFree(m_pMsiBuffer, "Tegra124RootComplex Msi buffer");
        m_pMsiBuffer = NULL;
    }

#if !defined NN_PCI_SMMU_DISABLED
    // sMMU teardown
    nn::dd::DetachDeviceAddressSpace(&m_DeviceAddressSpace,
                                     nn::dd::DeviceName::DeviceName_Afi);
    nn::dd::DestroyDeviceAddressSpace(&m_DeviceAddressSpace);
#endif

    m_IoVaManager.Finalize();

    return result;
}

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
