﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief
 *
 * @details
 */

namespace nn { namespace pcie { namespace driver { namespace detail {


class Driver
{

public:
    // Type for list of all clients
    typedef nn::util::IntrusiveList
    < Client,nn::util::IntrusiveListMemberNodeTraits<Client, &Client::m_ListNode>> ClientListType;

     // Type for list of endpoint functions
     typedef nn::util::IntrusiveList
     < EndpointFunction,nn::util::IntrusiveListMemberNodeTraits
     <EndpointFunction, &EndpointFunction::m_EndpointFuncListNode>> EndpointFunctionListType;


    Driver();
    ~Driver();

    Result Initialize();
    Result Finalize();
    Result RegisterInterruptEvent(nn::os::InterruptEventType* event, IntVector intVector);
    Result UnregisterInterruptEvent(IntVector intVector);
    Result GetBusAddressess(ResourceAddr resourceAddr, BusAddress *pRetBusAddress);
    Result GetRootComplex(RootComplex** pRootComplex);
    void RequestLoggedStateUpdate(bool isDueToError);
    ResourceAddr AlignResourceAddressWithBus(ResourceAddr inputAddress, size_t powerOfTwoRoundedSize);
    void DoLoggedStateCallback(bool isDueToError);
    void PmStateCallout(const FunctionState* pState);
    Result ResetDevice(BusNumber busNumber, DeviceNumber deviceNumber, FunctionResetOptionMask resetOptionMask);
    void SetLoggedStateCallback(LoggedStateCallback callback, uintptr_t context);


    // Synchronization of calling client threads and interrupt thread
    nn::os::Mutex       m_AccessMutex;

    // Table of all created busses
    Bus*                m_BusTable[MaxBusCount];

    // List of all created clients
    ClientListType      m_ClientList;

    // Client handle manager
    SafeHandleManager<Client, ClassDriverHandle, nn::pcie::MaxClientCount, 0xc000> m_ClientHandleManager;

    // Endpoint function handle manager
    SafeHandleManager<EndpointFunction, FunctionHandle, MaxEndpointFunctionCount, 0xe000> m_EndpointFunctionHandleManager;

    // List of all created endpoint functions
    EndpointFunctionListType m_EndpointFunctionList;

    // Local event manager
    LocalEventManager m_LocalEventManager;

private:
    enum IntThreadMultiWait
    {
        IntThreadMultiWait_First,
        IntThreadMultiWait_FirstIntVector,
        IntThreadMultiWait_LastIntVector = IntThreadMultiWait_FirstIntVector + MaxIntVectorCount - 1,
        IntThreadMultiWait_Pm,
        IntThreadMultiWait_LocalEventTimer,
        IntThreadMultiWait_ReEvaluate,
        IntThreadMultiWait_Max,
    };
    enum LocalEventId
    {
        LocalEventId_DoEnumeration,
        LocalEventId_LoggedStateRateLimit,
        LocalEventId_Max,
    };
    const BusNumber    WiFiBusNumber = 2;
    const DeviceNumber WiFiDeviceNumber = 0;

    Result InitializeRootComplex();
    Result FinalizeRootComplex();
    Result Resume();
    Result Suspend();
    void EnumerateDepthFirst(Bus *pBus, BusNumber *pAllocBusNum, uint64_t deviceMask);
    void HandlePowerManagementEvent();
    void ResetWiFiDevice(bool isLongResetDuration);
    void ShutdownWiFiDevice();
    void InterruptThread();
    static void InterruptThreadEntry(void *pThis)
    {
        ((Driver *)pThis)->InterruptThread();
    }
    void HandleLocalEvent(LocalEventType *pEvent);
    static void HandleLocalEventStatic(void *context, LocalEventType *pEvent)
    {
        Driver* pThis = reinterpret_cast<Driver*>(context);
        pThis->HandleLocalEvent(pEvent);
    }

    // Interrupt thread
    bool                        m_IntThreadRun;
    nn::os::ThreadType          m_IntThread;
    NN_ALIGNAS(nn::os::StackRegionAlignment) uint32_t m_IntThreadStack[1024 * 8];
    nn::os::MultiWaitType       m_MultiWait;
    nn::os::MultiWaitHolderType m_InterruptHolders[IntVector_MaxTypes];
    nn::os::InterruptEventType* m_InterruptEvents[IntVector_MaxTypes];
    nn::os::MultiWaitHolderType m_ReEvaluateHolder;
    nn::os::SystemEventType     m_ReEvaluateEvent;
    nn::os::MultiWaitHolderType m_PmModuleHolder;
    nn::os::MultiWaitHolderType m_LocalEventManagerTimerHolder;
    nn::psc::PmModule           m_PmModule;

    // Host
    RootComplex*                m_pRootComplex;
    RootComplex::Region         m_IoRegion;
    RootComplex::Region         m_PrefMemRegion;
    RootComplex::Region         m_NoPrefMemRegion;
    BusNumber                   m_AllocatedBusNum;
    ResourceAddr                m_AllocatedIoAddr;
    ResourceAddr                m_IoAddrLimit;
    ResourceAddr                m_AllocatedNoPrefMemAddr;
    ResourceAddr                m_NoPrefMemAddrLimit;
    ResourceAddr                m_AllocatedPrefMemAddr;
    ResourceAddr                m_PrefMemAddrLimit;

    // Platform specific glue
    bool                        m_IsWifiResetManaged;

    // Local event handling
    LocalEventPoolType          m_LocalEventPool;
    LocalEventType              m_LocalEventStorage[4];

    // Delayed enumeration
    LocalEventType              m_EnumerationTimer;

    // Logged State management
    LocalEventType              m_LoggedStateRateLimitTimer;
    bool                        m_IsLoggedStateStatisticsRateLimitActive;
    bool                        m_IsLoggedStateErrorRateLimitActive;
    int                         m_LoggedStateStatisticsPendingCount;
    int                         m_LoggedStateErrorPendingCount;
    LoggedStateCallback         m_LoggedStateCallback;
    uintptr_t                   m_LoggedStateContext;

};


} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn

