﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Bus class
 *
 * @details Bus is conduit connecting one or more PCI functions.
 * - "upstream" direction goes towards host (root complex)
 * - Although PCIe busses are physically point-to-point, there
 * are cases where multiple devices are logically attached to
 * same bus, such as within switch construct.
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

class Device;
class RootComplex;
class BridgeFunction;

class Bus
{
public:
    Bus(Driver *pDriver, Bus *pUpStreamBus, BridgeFunction *pUpStreamBridgeFunction,
        BusNumber busNum);
    ~Bus();


    Result Initialize();
    Result Finalize();
    BusNumber GetBusNum();
    bool IsRoot();
    Bus* GetUpStreamBus();
    BridgeFunction* GetUpStreamBridgeFunction();
    BridgeFunction* GetRootBridgeFunction();
    void SetRootComplex(RootComplex *pRootComplex);
    void GetRootComplex(RootComplex **ppRootComplex);
    void AttachDevice(Device *pDevice);
    Result DetachDevice(DeviceNumber devNum);
    Device* GetAttachedDevice(DeviceNumber devNum);
    Result ReadConfigDWord(DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint32_t *pDWord);
    Result ReadConfigWord(DeviceNumber devNum, FunctionNumber funcNum,
                          int32_t where, uint16_t *pWord);
    Result ReadConfigByte(DeviceNumber devNum, FunctionNumber funcNum,
                          int32_t where, uint8_t *pByte);
    Result WriteConfigDWord(DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint32_t dword);
    Result WriteConfigWord(DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint16_t word);
    Result WriteConfigByte(DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint8_t byte);
    void RegisterResource(Function::Resource *pResToRegister);
    void UnRegisterResource(Function::Resource *pResToUnRegister);
    Result AssignResourceAddresses(ResourceAddr *pIoAddrBase, ResourceAddr ioAddrLimit,
                                   ResourceAddr *pMemNoPrefAddrBase, ResourceAddr memNoPrefAddrLimit,
                                   ResourceAddr *pPrefMemAddrBase, ResourceAddr prefMemAddrLimit);
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;


    static Bus* GetBus(Driver *pDriver, BusNumber busNum);


private:
    // List of devices on this bus
    typedef nn::util::IntrusiveList
    < Device,nn::util::IntrusiveListMemberNodeTraits<Device, &Device::m_BusDevListNode>> DevListType;

     // List of address space resources on this bus
     typedef nn::util::IntrusiveList
     < Function::Resource,nn::util::IntrusiveListMemberNodeTraits<Function::Resource, &Function::Resource::listNode>> AddrSpaceListType;


      // Constructor initialized
      Driver            *m_pDriver;
      Bus               *m_pUpStreamBus;
      BridgeFunction    *m_pUpStreamBridgeFunction;
      BusNumber         m_BusNum;

      bool              m_IsRoot;
      RootComplex       *m_pRootComplex;

      // List of devices on this bus
      DevListType       m_DevList;

      // List of address space resources on this bus
      AddrSpaceListType m_IoSpaceList;
      AddrSpaceListType m_MemNoPrefetchSpaceList;
      AddrSpaceListType m_MemPrefetchSpaceList;
};

} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn

