﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace olsc { namespace srv { namespace util {

// --------------------------------------------------------------------------
// MountContext
// --------------------------------------------------------------------------

MountContext::MountContext(MountContext&& rhs) NN_NOEXCEPT
    : m_MountManager(rhs.m_MountManager), m_Mounter(rhs.m_Mounter), m_ReferenceMode(rhs.m_ReferenceMode)
{
    rhs.m_MountManager = nullptr;
}

MountContext::MountContext(MountManagerBase* mountManager, MounterId mounter) NN_NOEXCEPT
    : m_MountManager(mountManager), m_Mounter(mounter), m_ReferenceMode(ReferenceMode::Read)
{}

MountContext& MountContext::operator=(MountContext&& rhs) NN_NOEXCEPT
{
    MountContext t(std::move(rhs));
    swap(*this, t);
    return *this;
}

MountContext::~MountContext() NN_NOEXCEPT
{
    if (m_MountManager)
    {
        m_MountManager->Release(m_Mounter, m_ReferenceMode);
    }
}

void MountContext::SetReferenceMode(ReferenceMode referenceMode) NN_NOEXCEPT
{
    m_ReferenceMode = referenceMode;
}

void swap(MountContext& lhs, MountContext& rhs) NN_NOEXCEPT
{
    std::swap(lhs.m_MountManager, rhs.m_MountManager);
    std::swap(lhs.m_Mounter, rhs.m_Mounter);
    std::swap(lhs.m_ReferenceMode, rhs.m_ReferenceMode);
}

const char* MountContext::GetMountName() const NN_NOEXCEPT
{
    return m_MountManager->GetMountName(m_Mounter);
}

const char* MountContext::GetRootPath() const NN_NOEXCEPT
{
    return m_MountManager->GetRootPath(m_Mounter);
}

// -----------------------------------------------------------------
// WriteMount
// -----------------------------------------------------------------

WriteMount::WriteMount(WriteMount&& rhs) NN_NOEXCEPT
    : MountContext(std::move(rhs))
{
}

WriteMount::WriteMount(MountManagerBase* mountManager, MounterId mounter) NN_NOEXCEPT
    : MountContext(mountManager, mounter)
{
    SetReferenceMode(ReferenceMode::ReadAndWrite);
}

WriteMount& WriteMount::operator=(WriteMount&& rhs) NN_NOEXCEPT
{
    WriteMount t(std::move(rhs));
    swap(*this, t);
    return *this;
}

Result WriteMount::Commit() NN_NOEXCEPT
{
    NN_RESULT_DO(fs::Commit(GetMountName()));
    NN_RESULT_SUCCESS;
}

void swap(WriteMount& lhs, WriteMount& rhs) NN_NOEXCEPT
{
    swap(static_cast<MountContext&>(lhs), static_cast<MountContext&>(rhs));
}

}}}} //namespace nn::olsc::srv::util
