﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/util/olsc_AutonomyUploadWaitingList.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv { namespace util {

int AutonomyUploadWaitingList::ApplicationIdList::GetCount() const NN_NOEXCEPT
{
    return m_Count;
}

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::ApplicationIdList::PopFront() NN_NOEXCEPT
{
    return RemoveAt(0);
}

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::ApplicationIdList::GetFront() const NN_NOEXCEPT
{
    if (m_Count == 0)
    {
        return nn::util::nullopt;
    }
    return m_Applications[0];
}

void AutonomyUploadWaitingList::ApplicationIdList::PushBack(ApplicationId appId) NN_NOEXCEPT
{
    auto found = Find(appId);
    if (found)
    {
        return;
    }

    if (m_Count >= MaxApplicationCount)
    {
        auto erased = RemoveAt(0);
        NN_DETAIL_OLSC_WARN("AutonomyUploadWaitingList is full. Removed top value. Removed = %016llx\n", erased->value);
        NN_UNUSED(erased);
    }

    NN_SDK_ASSERT(m_Count < MaxApplicationCount);
    m_Applications[m_Count] = appId;
    m_Count++;
}

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::ApplicationIdList::Remove(ApplicationId appId) NN_NOEXCEPT
{
    auto index = Find(appId);
    if (index)
    {
        return RemoveAt(*index);
    }
    return nn::util::nullopt;
}

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::ApplicationIdList::RemoveAt(int index) NN_NOEXCEPT
{
    if (index >= m_Count)
    {
        return nn::util::nullopt;
    }

    auto erased = m_Applications[index];
    std::memmove(&m_Applications[index], &m_Applications[index + 1], sizeof(m_Applications[0]) * (m_Count - index - 1));
    m_Count--;

    return erased;
}

nn::util::optional<int> AutonomyUploadWaitingList::ApplicationIdList::Find(ApplicationId appId) const NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; ++i)
    {
        if (m_Applications[i] == appId)
        {
            return i;
        }
    }
    return nn::util::nullopt;
}

// --------------------------------------------------------------------------------------

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::PopFront() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    return m_WaitingApplications.PopFront();
}

nn::util::optional<ApplicationId> AutonomyUploadWaitingList::GetFront() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    return m_WaitingApplications.GetFront();
}

int AutonomyUploadWaitingList::GetCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    return m_WaitingApplications.GetCount();
}

void AutonomyUploadWaitingList::Update(const ApplicationId ids[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    for (int i = 0; i < count; ++i) {
        m_WaitingApplications.PushBack(ids[i]);
    }
}

void AutonomyUploadWaitingList::Remove(ApplicationId appId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    m_WaitingApplications.Remove(appId);
}

}}}} //namespace nn::olsc::srv::util

