﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/crypto/crypto_Md5Generator.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/http/http_Result.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/olsc_SaveDataArchiveInfoCacheManager.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_TransferHelper.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os/os_Random.h>
#include <nn/time.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_FormatString.h>

#undef NN_RESULT_DO
#define NN_RESULT_DO(s) \
    do { \
        auto ___r = (s); \
        if(___r.IsFailure()) { \
            NN_DETAIL_OLSC_TRACE("%s(%d): %08x\n", __FILE__, __LINE__, ___r.GetInnerValueForDebug()); \
            NN_RESULT_THROW(___r);\
        }\
    }while(NN_STATIC_CONDITION(false))

namespace nn { namespace olsc { namespace srv {

namespace {
    void DumpSi(const char* title, const SeriesInfo& si) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        NN_DETAIL_OLSC_TRACE("%s:\n", title);
        NN_DETAIL_OLSC_TRACE("SeriesId  : %016llx, %llu\n", si.seriesId, si.seriesId);
        NN_DETAIL_OLSC_TRACE("CommitId  : %016llx, %llu\n", si.commitId, si.commitId);
#else
        NN_UNUSED(title);
        NN_UNUSED(si);
#endif
    }

    Result DecideSeriesInfoForUpload(SeriesInfo* out, const SeriesInfo& serverSi, const SeriesInfo& lastSi, const fs::SaveDataCommitId& commitId) NN_NOEXCEPT
    {
        DumpSi("serverSi", serverSi);
        DumpSi("lastSi", lastSi);

        if (serverSi != lastSi || lastSi == InvalidSeriesInfo)
        {
            os::GenerateRandomBytes(&out->seriesId, sizeof(out->seriesId));
        }
        else
        {
            *out = lastSi;
        }

        out->commitId = commitId;

        DumpSi("--> Decided", *out);
        NN_RESULT_SUCCESS;
    }

    Result EvaluateLocalSaveDataConditionForDownload(const nn::util::optional<fs::SaveDataCommitId>& pCommitId, const TransferTaskDetailInfo& detailInfo) NN_NOEXCEPT
    {
        // デバイスにセーブデータが無ければ、登録時の状況に関わらずローカルの条件はクリア
        if (!pCommitId)
        {
            NN_RESULT_SUCCESS;
        }

        // タスク登録時にはセーブデータがなかったが現在は作成されている
        if (!detailInfo.config.dlInfo.saveDataExists)
        {
            NN_RESULT_THROW(olsc::ResultLocalDataCreated());
        }

        // タスク登録時と現在とでセーブデータのハッシュが異なっている（セーブが更新されてしまっている）
        if (*pCommitId != detailInfo.config.dlInfo.currentCommitId)
        {
            NN_RESULT_THROW(olsc::ResultLocalDataUpdated());
        }

        NN_RESULT_SUCCESS;
    }

    Result EvaluateServerSeriesInfoConditionForDownload(const nn::util::optional<SaveDataArchiveInfo>& pServerSda, const TransferTaskDetailInfo& detailInfo) NN_NOEXCEPT
    {
        // サーバにデータがない
        if (!pServerSda)
        {
            NN_RESULT_THROW(olsc::ResultServerDataNotExist());
        }

        auto serverSi = pServerSda->seriesInfo;

        // タスク登録時から系列が変わってしまっている
        if (detailInfo.config.cachedSi.seriesId != serverSi.seriesId)
        {
            NN_RESULT_THROW(olsc::ResultServerSeriesIdChanged());
        }

        NN_RESULT_SUCCESS;
    }

    Result EvaluateLocalSaveDataConditionForUpload(const fs::SaveDataCommitId& commitId, const SeriesInfo& lastSi) NN_NOEXCEPT
    {
        if (lastSi && commitId == lastSi.commitId)
        {
            // 最後に同期した時点からセーブデータが更新されていなくても、タスクが積まれていれば実行する。
            // (自動 UL の場合はそもそもタスクが積まれないはず。)
            NN_DETAIL_OLSC_INFO("Uploading save data which is not changed since last synchronization.");
        }

        NN_RESULT_SUCCESS;
    }

    Result EvaluateServerSeriesInfoConditionForUpload(const nn::util::optional<SaveDataArchiveInfo>& serverSda, const TransferTaskDetailInfo& detailInfo) NN_NOEXCEPT
    {
        // 登録時の SI(SDA) なし 且つ 現在も SI(SDA) なし
        if (!detailInfo.config.cachedSi && !serverSda)
        {
            NN_RESULT_SUCCESS;
        }

        if (!serverSda)
        {
            // 登録時 SI(SDA) あり、且つ 現在は SI(SDA) なし
            NN_RESULT_THROW(olsc::ResultServerDataNotExist());
        }
        else if (!detailInfo.config.cachedSi)
        {
            // 登録時 SI(SDA) なし、且つ 現在は SI(SDA) あり
            NN_RESULT_THROW(olsc::ResultServerSeriesInfoCreated());
        }

        NN_RESULT_THROW_UNLESS(detailInfo.config.cachedSi == serverSda->seriesInfo, ResultServerSeriesInfoUpdated());
        NN_RESULT_SUCCESS;
    }

    Result EvaluateUploadTransferCondition(
        SeriesInfo* pOutSi,
        const TransferTaskDetailInfo& detailInfo,
        const fs::SaveDataInfo& saveDataInfo,
        const SeriesInfo& lastSi,
        const nn::util::optional<SaveDataArchiveInfo>& serverSda) NN_NOEXCEPT
    {
        auto serverSi = (serverSda) ? serverSda->seriesInfo : InvalidSeriesInfo;
        // 最新の SDA と登録時の SDA を評価して処理の継続可能性を判断する。
        NN_RESULT_DO(EvaluateServerSeriesInfoConditionForUpload(serverSda, detailInfo));

        // ローカルセーブデータの情報取得と評価
        fs::SaveDataCommitId commitId;
        NN_RESULT_DO(fs::GetSaveDataCommitId(&commitId, saveDataInfo.saveDataSpaceId, saveDataInfo.saveDataId));
        NN_RESULT_DO(EvaluateLocalSaveDataConditionForUpload(commitId, lastSi));

        // このアップロードタスクで使用する SI を決定
        NN_RESULT_DO(DecideSeriesInfoForUpload(pOutSi, serverSi, lastSi, commitId));

        NN_RESULT_SUCCESS;
    }

    // TODO : ネットワーク接続要求は TTA で扱うため、最終的にここからは除く
    Result EnsureNetworkConnection(nifm::NetworkConnection& connection, nn::util::Cancelable& cancelable) NN_NOEXCEPT
    {
        NN_RESULT_DO(util::SubmitAndWaitNetworkConnection(connection, nifm::RequirementPreset_InternetForSystemProcess, cancelable));
        NN_RESULT_SUCCESS;
    }

} // namespace

bool TransferTaskBase::IsSucceeded() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Done && m_LastResult.IsSuccess();
}

bool TransferTaskBase::IsRetryable() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Done && IsRetryableResult(m_LastResult);
}

bool TransferTaskBase::IsFatal() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Done && !IsRetryableResult(m_LastResult);
}

const nn::util::optional<SaveDataArchiveInfo>& TransferTaskBase::TryGetLatestSaveDataArchive() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_LatestSaveDataArchive;
}

void TransferTaskBase::CompleteTask(Result result) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Done = true;
    m_LastResult = result;

    auto contextUpdater = [&](TransferTaskContext& context)
    {
        context.lastResult = m_LastResult;
    };
    UpdateContext(contextUpdater);
}

const database::TransferTaskContextDatabase::TransferTaskContext& TransferTaskBase::GetContext() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Context;
}

void TransferTaskBase::InitializeContext(TransferTaskId id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Context.id = id;
}

void TransferTaskBase::SetupContext(size_t dataSize, uint32_t maxPartitionsCount) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Context.totalSize = dataSize;
    m_Context.totalComponentCount = maxPartitionsCount;
}

void TransferTaskBase::UpdateContext(const ContextUpdater& pred) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    pred(m_Context);
}

void TransferTaskBase::UpdateContextByChunk(size_t chunkSize) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Context.currentSize += chunkSize;
    m_Context.currentComponentIndex++;
}

// TODO : ちゃんとした Context の取得ができたら排除
void TransferTaskBase::CompleteContext() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Context.currentSize = m_Context.totalSize;
    m_Context.currentComponentIndex = m_Context.totalComponentCount;
}

void TransferTaskBase::Initialize(const TransferTaskDetailInfo& detailInfo, const SeriesInfo& lastSi) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_DetailInfo = detailInfo;
    InitializeContext(detailInfo.id);
    m_LastSi = lastSi;
}

void TransferTaskBase::Resume(const TransferTaskDetailInfo& detailInfo, const TransferTaskContext& context, const SeriesInfo& lastSi) NN_NOEXCEPT
{
    Initialize(detailInfo, lastSi);

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    m_Context = context;
}

void TransferTaskBase::Resume(const TransferTaskDetailInfo& detailInfo, const TransferTaskContext& context, const SeriesInfo& lastSi, TransferTaskSuspendedInfo&& info) NN_NOEXCEPT
{
    Initialize(detailInfo, lastSi);

    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    UpdateTransferTaskSuspendedInfo(std::move(info));
    NN_SDK_ASSERT(!info.IsValid());
}

const TransferTaskDetailInfo& TransferTaskBase::GetDetailInfo() const NN_NOEXCEPT
{
    return m_DetailInfo;
}

TransferTaskBase::TransferTaskExecutionResource& TransferTaskBase::GetExecutionResource() NN_NOEXCEPT
{
    return m_ExecutionResource;
}

const SeriesInfo& TransferTaskBase::GetLastSi() const NN_NOEXCEPT
{
    return m_LastSi;
}

void TransferTaskBase::SetLatestSaveDataArchive(const nn::util::optional<SaveDataArchiveInfo>& sda) NN_NOEXCEPT
{
    m_LatestSaveDataArchive = sda;
}

void TransferTaskBase::PrintDetailInfo() const NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    auto& detailInfo = GetDetailInfo();

    NN_DETAIL_OLSC_TRACE("[olsc] Id      : %016llx\n", detailInfo.id);
    NN_DETAIL_OLSC_TRACE("[olsc] AppId   : %016llx\n", detailInfo.appId);
    NN_DETAIL_OLSC_TRACE("[olsc] Uid     : %016llx%016llx\n", detailInfo.uid._data[0], detailInfo.uid._data[1]);
    NN_DETAIL_OLSC_TRACE("[olsc] Kind    : %s\n", detailInfo.config.kind == TransferTaskKind::Download ? "DL" : "UL");
    NN_DETAIL_OLSC_TRACE("[olsc] SeriesId: %016llx\n", detailInfo.config.cachedSi.seriesId);
#endif
}

nifm::NetworkConnection& TransferTaskBase::GetNetworkConnection() NN_NOEXCEPT
{
    return m_Connection;
}

TransferTaskSuspendedInfo TransferTaskBase::DetachTransferTaskSuspendedInfo() NN_NOEXCEPT
{
    return TransferTaskSuspendedInfo(std::move(m_SuspendedInfo));
}

void TransferTaskBase::UpdateTransferTaskSuspendedInfo(TransferTaskSuspendedInfo&& suspendedInfo) NN_NOEXCEPT
{
    m_SuspendedInfo = std::move(suspendedInfo);
}

bool TransferTaskBase::IsTaskResumable() NN_NOEXCEPT
{
    return m_SuspendedInfo.IsValid();
}

// ----------------------------------------------------------------------

Result UploadTransferTask::PrepareNewUpload(
    bool* pOutIsDiffUpload,
    SaveDataArchiveInfo* pOutSda,
    std::unique_ptr<fs::ISaveDataDivisionExporter>* pOutExporter,
    const TransferTaskDetailInfo& detailInfo,
    fs::SaveDataTransferManagerForCloudBackUp& manager,
    TransferTaskExecutionResource& resource
) NN_NOEXCEPT
{
    nn::util::optional<SaveDataArchiveInfo> serverSda;
    // 既存バックアップの情報取得
    NN_RESULT_DO(transfer::RequestFixedSaveDataArchiveInfo(
        &serverSda, resource.nsaIdTokenBuffer, detailInfo.appId,
        resource.curlHandle, resource.workBuffer.data(), resource.workBuffer.size(), this));
    SetLatestSaveDataArchive(serverSda);

    // UL するセーブを開く
    auto pSaveDataInfo = util::FindSaveData(detailInfo.uid, detailInfo.appId);
    NN_RESULT_THROW_UNLESS(pSaveDataInfo, olsc::ResultLocalDataNotExist());

    std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;
    NN_RESULT_DO(manager.OpenSaveDataExporter(&exporter, fs::SaveDataSpaceId::User, pSaveDataInfo->saveDataId));
    NN_SDK_ASSERT(exporter);

    // データ転送の条件の確認(バックアップ可能か判定)し、SI を取得
    SeriesInfo si;
    NN_RESULT_DO(EvaluateUploadTransferCondition(&si, detailInfo, *pSaveDataInfo, GetLastSi(), serverSda));

    // TODO : 新規 UL か差分 UL か判定
    *pOutIsDiffUpload = serverSda;

    // バックアップ開始の通知
    SaveDataArchiveInfo targetSda;
    NN_RESULT_DO(transfer::TransferHelper::StartWholeUpload(
        &targetSda, detailInfo.uid, detailInfo.appId,
        si, resource.nsaIdTokenBuffer, *pSaveDataInfo,
        resource.curlHandle, resource.workBuffer.data(), resource.workBuffer.size(), *this));

    NN_DETAIL_OLSC_TRACE("numPartitions = %d\n", targetSda.numOfPartitions);
    exporter->SetDivisionCount(targetSda.numOfPartitions);

    // context の初期化
    // TODO : 差分 UL の場合、差分のある CF の合計サイズを分母とする
    SetupContext(targetSda.dataSize, targetSda.numOfPartitions);

    *pOutExporter = std::move(exporter);
    *pOutSda = targetSda;
    NN_RESULT_SUCCESS;
}

Result UploadTransferTask::PrepareResumedUpload(
    SaveDataArchiveInfo* pOutSda,
    const TransferTaskDetailInfo& detailInfo,
    SaveDataArchiveId sdaId,
    fs::ISaveDataChunkIterator& iter,
    TransferTaskExecutionResource& resource
) NN_NOEXCEPT
{
    SaveDataArchiveInfo targetSda;
    // タイムアウトの延長
    int cfCount = 0;
    NN_RESULT_DO(transfer::TransferHelper::GetSuspendedSaveDataArchiveInfo(
        &targetSda, &cfCount, sdaId, resource.nsaIdTokenBuffer, resource.curlHandle,
        resource.workBuffer.data(), resource.workBuffer.size(), *this));

    // UL するセーブデータに(再開前と比較して)変更がないか確認
    auto pSaveDataInfo = util::FindSaveData(detailInfo.uid, detailInfo.appId);
    NN_RESULT_THROW_UNLESS(pSaveDataInfo, olsc::ResultLocalDataNotExist());

    fs::SaveDataCommitId commitId;
    NN_RESULT_DO(fs::GetSaveDataCommitId(&commitId, pSaveDataInfo->saveDataSpaceId, pSaveDataInfo->saveDataId));
    NN_RESULT_THROW_UNLESS(targetSda.seriesInfo.commitId == commitId, olsc::ResultLocalDataChangedWhileSuspended());

    // context の初期化
    // TODO : 差分 UL の場合、差分のある CF の合計サイズを分母とする
    SetupContext(targetSda.dataSize, targetSda.numOfPartitions);
    *pOutSda = targetSda;

    auto& cfList = (reinterpret_cast<transfer::TransferHelper::Buffer*>(resource.workBuffer.data()))->componentFileListBuffer;
    auto endCf = cfList.begin() + cfCount;

    // UL 済みの CF については iter を進める
    for (; !iter.IsEnd(); iter.Next())
    {
        auto index = iter.GetId();
        auto targetCf = std::find_if(cfList.begin(), endCf, [&index](ComponentFileInfo i){
            return i.clientArgument.chunkId == index;
            });
        // targetCf が存在しない、または targetCf が Fixed ではない場合はそこから継続
        if(targetCf == endCf || targetCf->status != ComponentFileStatus::Fixed)
        {
            NN_RESULT_SUCCESS;
        }
        // Fixed な CF だったら、 context を進める
        UpdateContextByChunk(targetCf->saveDataChunkSize);
    }
    NN_RESULT_SUCCESS;
}

Result UploadTransferTask::Upload() NN_NOEXCEPT
{
    auto& detailInfo = GetDetailInfo();
    auto& resource = GetExecutionResource();

    // TODO : ネットワーク接続要求は TTA で扱うため、最終的にここからは除く
    NN_RESULT_DO(EnsureNetworkConnection(GetNetworkConnection(), *this));

    size_t nsaIdTokenSize;
    NN_RESULT_DO(util::GetNsaIdToken(&nsaIdTokenSize, resource.nsaIdTokenBuffer.data, sizeof(resource.nsaIdTokenBuffer.data), detailInfo.uid, *this));
    NN_DETAIL_OLSC_TRACE("NsaIdToken: %s\n", resource.nsaIdTokenBuffer);

    // TODO : ポリシーチェック

    SaveDataArchiveInfo targetSda;
    std::unique_ptr<fs::ISaveDataChunkIterator> iter;

    // 中断情報が入っていたら再開
    TransferTaskSuspendedInfo suspendedInfo = {};
    if(IsTaskResumable())
    {
        suspendedInfo = DetachTransferTaskSuspendedInfo();
        NN_RESULT_DO(suspendedInfo.exporter->OpenSaveDataDiffChunkIterator(&iter));
        NN_RESULT_DO(PrepareResumedUpload(&targetSda, detailInfo, suspendedInfo.sdaId, *iter.get(), resource));
        if(targetSda.status == SaveDataArchiveStatus::Fixed)
        {
            CompleteContext();
            NN_RESULT_SUCCESS;
        }
    }
    else
    {
        bool isDiffUpload;
        std::unique_ptr<fs::SaveDataTransferManagerForCloudBackUp> manager(new fs::SaveDataTransferManagerForCloudBackUp());
        suspendedInfo.manager = std::move(manager);
        NN_RESULT_DO(PrepareNewUpload(&isDiffUpload, &targetSda, &suspendedInfo.exporter, detailInfo, *suspendedInfo.manager.get(), resource));

        suspendedInfo.id = detailInfo.id;
        suspendedInfo.sdaId = targetSda.id;
        suspendedInfo.isDifferentialUpload = isDiffUpload;

        NN_RESULT_DO(suspendedInfo.exporter->OpenSaveDataDiffChunkIterator(&iter));
    }
    NN_UTIL_SCOPE_EXIT
    {
        NN_SDK_ASSERT(suspendedInfo.IsValid());
        UpdateTransferTaskSuspendedInfo(std::move(suspendedInfo));
    };

    auto contextUpdater = [&](size_t transferredSize, size_t offsetNumber)
    {
        UpdateContext([&](TransferTaskContext& context) {
            context.currentSize += transferredSize;
            context.currentComponentIndex = static_cast<int>(offsetNumber);
        });
    };

    // UL が必要な分割データを Export
    NN_RESULT_DO(transfer::TransferHelper::ExportSaveData(
        targetSda.id, suspendedInfo.exporter, iter, resource.nsaIdTokenBuffer, resource.curlHandle,
        contextUpdater, detailInfo.uid, resource.workBuffer.data(), resource.workBuffer.size(), *this));

    // ExportSaveData で更新されている可能性があるため再取得
    NN_RESULT_DO(util::GetNsaIdToken(&nsaIdTokenSize, resource.nsaIdTokenBuffer.data, sizeof(resource.nsaIdTokenBuffer.data), detailInfo.uid, *this));

    // バックアップ完了通知
    // TODO : 新規 UL か差分 UL かで呼び分け
    NN_RESULT_DO(transfer::TransferHelper::FinishWholeUpload(
        targetSda.id, suspendedInfo.exporter, resource.nsaIdTokenBuffer, resource.curlHandle,
        resource.workBuffer.data(), resource.workBuffer.size(), *this));

    // バックアップの情報取得
    nn::util::optional<SaveDataArchiveInfo> serverSda;
    NN_RESULT_DO(transfer::RequestSaveDataArchiveInfo(
        &serverSda, resource.nsaIdTokenBuffer, targetSda.id,
        resource.curlHandle, resource.workBuffer.data(), resource.workBuffer.size(), this));
    SetLatestSaveDataArchive(serverSda);

    NN_RESULT_THROW_UNLESS(serverSda, olsc::ResultExpectedServerDataNotExist());
    CompleteContext();
    NN_RESULT_SUCCESS;
}

void UploadTransferTask::Execute() NN_NOEXCEPT
{
    PrintDetailInfo();

    CompleteTask(Upload());
}


bool UploadTransferTask::IsRetryableResult(Result result) const NN_NOEXCEPT
{
    NN_UNUSED(result);

    // TODO: 下記エラーの場合のみ true を返す
    // - (UL 再開時の) セーブデータ更新済
    // - UL タイムアウト延長に失敗(タイムアウト済み or 削除済み)
    return false;
}

// --------------------------------------------------------------------

Result DownloadTransferTask::Download() NN_NOEXCEPT
{
    auto& detailInfo = GetDetailInfo();
    auto& resource = GetExecutionResource();

    // TODO : ネットワーク接続要求は TTA で扱うため、最終的にここからは除く
    NN_RESULT_DO(EnsureNetworkConnection(GetNetworkConnection(), *this));

    size_t nsaIdTokenSize;
    NN_RESULT_DO(util::GetNsaIdToken(&nsaIdTokenSize, resource.nsaIdTokenBuffer.data, sizeof(resource.nsaIdTokenBuffer.data), detailInfo.uid, *this));
    NN_DETAIL_OLSC_TRACE("NsaIdToken: %s\n", resource.nsaIdTokenBuffer);

    // ローカルセーブデータの情報取得と評価
    nn::util::optional<fs::SaveDataCommitId> pCommitId;
    auto pSaveDataInfo = util::FindSaveData(detailInfo.uid, detailInfo.appId);
    if (pSaveDataInfo)
    {
        fs::SaveDataCommitId commitId;
        fs::ScopedAutoAbortDisabler antiAbort;
        NN_RESULT_DO(fs::GetSaveDataCommitId(&commitId, pSaveDataInfo->saveDataSpaceId, pSaveDataInfo->saveDataId));
        pCommitId.emplace(commitId);
    }
    NN_RESULT_DO(EvaluateLocalSaveDataConditionForDownload(pCommitId, detailInfo));

    // サーバの情報取得と評価
    nn::util::optional<SaveDataArchiveInfo> serverSda;
    NN_RESULT_DO(transfer::RequestFixedSaveDataArchiveInfo(
        &serverSda, resource.nsaIdTokenBuffer, detailInfo.appId,
        resource.curlHandle, resource.workBuffer.data(), resource.workBuffer.size(), this));
    SetLatestSaveDataArchive(serverSda);

    NN_RESULT_DO(EvaluateServerSeriesInfoConditionForDownload(serverSda, detailInfo));

    auto contextInitializer = [&](size_t dataSize, uint32_t maxPartitionsCount)
    {
        UpdateContext([&](TransferTaskContext& context) {
            context.totalSize = dataSize;
            context.totalComponentCount = maxPartitionsCount;
        });
    };

    auto contextUpdater = [&](size_t transferredSize, size_t offsetNumber)
    {
        UpdateContext([&](TransferTaskContext& context) {
            context.currentSize += transferredSize;
            context.currentComponentIndex = static_cast<int>(offsetNumber);
        });
    };

    NN_RESULT_DO(transfer::TransferHelper::Download(detailInfo.uid, *serverSda,
        resource.nsaIdTokenBuffer, resource.curlHandle, contextInitializer, contextUpdater, resource.workBuffer.data(), resource.workBuffer.size(), *this));


    NN_RESULT_SUCCESS;
}

bool DownloadTransferTask::IsRetryableResult(Result result) const NN_NOEXCEPT
{
    NN_UNUSED(result);
    // TODO:
    return false;
}

void DownloadTransferTask::Execute() NN_NOEXCEPT
{
    PrintDetailInfo();

    CompleteTask(Download());
}

}}} //namespace nn::olsc::srv

