﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace olsc { namespace srv { namespace database {

namespace {

} // namespace

//------------------------------------------------------------------------------
// Private
//------------------------------------------------------------------------------

const char* TransferTaskContextDatabase::TransferTaskContextStore::GetMetadataFileRelativePath() const NN_NOEXCEPT
{
    return "ttctx_meta";
}

const char* TransferTaskContextDatabase::TransferTaskContextStore::GetEntryFileRelativePath() const NN_NOEXCEPT
{
    return "%sttctx_entry";
}

util::ReadMount TransferTaskContextDatabase::TransferTaskContextStore::AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForRead();
}

util::WriteMount TransferTaskContextDatabase::TransferTaskContextStore::AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForWrite();
}


void TransferTaskContextDatabase::CreateTransferTaskContext(TransferTaskContext* out, TransferTaskId id, int componentFileCount, size_t totalFileSize) NN_NOEXCEPT
{
    out->id = id;
    out->totalSize = totalFileSize;
    out->currentComponentIndex = 0;
    out->totalComponentCount = componentFileCount;
    out->lastResult = nn::ResultSuccess();
    out->currentComponentFile = {};
    out->currentSize = 0;
    out->througput = 0;
}

//------------------------------------------------------------------------------
// Public
//------------------------------------------------------------------------------

Result TransferTaskContextDatabase::GetTransferTaskContext(TransferTaskContext* out, TransferTaskId id) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const TransferTaskContextStore> storeLock(m_Store);

    auto entry = m_Store.FindIf([&id](const TransferTaskContext& entry) {
        return entry.id == id;
    });
    NN_RESULT_THROW_UNLESS(entry, olsc::ResultTransferTaskContextNotFound());

    *out = *entry;
    NN_RESULT_SUCCESS;
}

Result TransferTaskContextDatabase::RemoveTransferTaskContext(TransferTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<TransferTaskContextStore> storeLock(m_Store);

    NN_RESULT_TRY(
        m_Store.RemoveIf([&id](const TransferTaskContext& entry) {
            return entry.id == id;
        }))
        NN_RESULT_CATCH_CONVERT(olsc::ResultDatabaseEntryNotFound, olsc::ResultTransferTaskContextNotFound())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result TransferTaskContextDatabase::CreateTransferTaskContext(TransferTaskId id, int totalComponentCount, size_t totalSize) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<TransferTaskContextStore> storeLock(m_Store);
    NN_ABORT_UNLESS(m_Store.GetCount() < MaxTransferTaskContextCount);

    auto entry = m_Store.FindIf([&id](const TransferTaskContext& entry) {
        return entry.id == id;
    });
    NN_RESULT_THROW_UNLESS(!entry, olsc::ResultTransferTaskContextDuplicated());

    TransferTaskContext context;
    CreateTransferTaskContext(&context, id, totalComponentCount, totalSize);

    NN_RESULT_DO(m_Store.PushBack(context));

    NN_RESULT_SUCCESS;
}

Result TransferTaskContextDatabase::UpdateTransferTaskContext(const TransferTaskContext& context) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<TransferTaskContextStore> storeLock(m_Store);

    auto replaceResult = m_Store.ReplaceIf([&context](const TransferTaskContext& entry) {
        return entry.id == context.id;
    }, context);

    NN_RESULT_TRY(replaceResult)
        NN_RESULT_CATCH(olsc::ResultDatabaseEntryNotFound)
        {
            auto count = m_Store.GetCount();
            if (count >= MaxTransferTaskContextCount)
            {
                TransferTaskContext head;
                NN_RESULT_DO(m_Store.At(&head, 0));
                NN_RESULT_DO(m_Store.Remove(head));
            }
            NN_RESULT_DO(m_Store.PushBack(context));
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::util::optional<TransferTaskContextDatabase::TransferTaskContext> TransferTaskContextDatabase::GetTransferTaskContext(TransferTaskId id) const NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    std::lock_guard<const TransferTaskContextStore> storeLock(m_Store);

    auto entry = m_Store.FindIf([&id](const TransferTaskContext& entry) {
        return entry.id == id;
    });

    return entry;
}

Result TransferTaskContextDatabase::RemoveAllTransferTaskContext() NN_NOEXCEPT
{
    std::lock_guard<const TransferTaskContextStore> storeLock(m_Store);
    return m_Store.Cleanup();
}

}}}} //namespace nn::olsc::srv::database

