﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace olsc { namespace srv { namespace database {
namespace{

    DataInfo GetDataInfoFromSaveDataArchiveInfo(const SaveDataArchiveInfo& sda)
    {
        DataInfo ret;
        ret.id = sda.id;
        ret.appId = sda.applicationId;
        ret.uid = sda.userId;
        ret.status = static_cast<DataStatus>(sda.status);
        ret.size = sda.dataSize;
        ret.updatedTime = sda.savedAt;
        ret.uploadedTime = sda.finishedAt;
        return ret;
    }
}


// -----------------------------------------------------------------------------

const char* SaveDataArchiveInfoCache::SaveDataArchiveInfoStore::GetMetadataFileRelativePath() const NN_NOEXCEPT
{
    return "sdaStore_meta";
}

const char* SaveDataArchiveInfoCache::SaveDataArchiveInfoStore::GetEntryFileRelativePath() const NN_NOEXCEPT
{
    return "sdaStore_entry";
}

util::ReadMount SaveDataArchiveInfoCache::SaveDataArchiveInfoStore::AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireUserSettingsSaveForRead(m_Uid);
}

util::WriteMount SaveDataArchiveInfoCache::SaveDataArchiveInfoStore::AcquireWriteMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireUserSettingsSaveForWrite(m_Uid);
}

// -----------------------------------------------------------------------------

int SaveDataArchiveInfoCache::GetCount() const NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_AppIdList)> keyListLock(m_AppIdList);
    return m_AppIdList.GetCount();
}

nn::util::optional<DataInfo> SaveDataArchiveInfoCache::GetDataInfoByApplicationId(ApplicationId id) const NN_NOEXCEPT
{
    auto sda = GetSaveDataArchiveInfoByApplicationId(id);
    if (!sda)
    {
        return nn::util::nullopt;
    }

    return GetDataInfoFromSaveDataArchiveInfo(*sda);
}

nn::util::optional<DataInfo> SaveDataArchiveInfoCache::GetDataInfoByDataId(DataId id) const NN_NOEXCEPT
{
    auto sda = GetSaveDataArchiveInfoByDataId(id);
    if (!sda)
    {
        return nn::util::nullopt;
    }

    return GetDataInfoFromSaveDataArchiveInfo(*sda);
}

nn::util::optional<SaveDataArchiveInfo> SaveDataArchiveInfoCache::GetSaveDataArchiveInfoByDataId(DataId id) const NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_IdList)> keyListLock(m_IdList);
    std::lock_guard<const decltype(m_Store)> storeLock(m_Store);

    auto found = m_IdList.Find(static_cast<SaveDataArchiveId>(id));

    if (!found)
    {
        return nn::util::nullopt;
    }

    SaveDataArchiveInfo sda;
    NN_ABORT_UNLESS(m_Store.Get(&sda, *found));

    return sda;
}

nn::util::optional<SaveDataArchiveInfo> SaveDataArchiveInfoCache::GetSaveDataArchiveInfoByApplicationId(ApplicationId id) const NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_AppIdList)> keyListLock(m_AppIdList);
    std::lock_guard<const decltype(m_Store)> storeLock(m_Store);

    auto found = m_AppIdList.Find(id);

    if (!found)
    {
        return nn::util::nullopt;
    }

    SaveDataArchiveInfo sda;
    NN_ABORT_UNLESS(m_Store.Get(&sda, *found));

    return sda;
}

Result SaveDataArchiveInfoCache::Add(const SaveDataArchiveInfo& sdaInfo) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_AppIdList)> appIdListLock(m_AppIdList);
    std::lock_guard<decltype(m_IdList)> idListLock(m_IdList);
    std::lock_guard<decltype(m_Store)> storeLock(m_Store);

    NN_RESULT_THROW_UNLESS(m_AppIdList.GetCount() < MaxEntryCount, ResultSaveDataArchiveUnacceptableObjectCount());

    auto found = m_AppIdList.Find(sdaInfo.applicationId);

    if (found)
    {
        SaveDataArchiveInfo oldSda;
        NN_ABORT_UNLESS(m_Store.Get(&oldSda, *found));
        NN_ABORT_UNLESS(m_Store.Replace(*found, sdaInfo));
        m_IdList.Remove(oldSda.id);
        NN_ABORT_UNLESS(m_IdList.Add(sdaInfo.id, *found));
        NN_RESULT_SUCCESS;
    }

    int storeIndex;
    NN_ABORT_UNLESS(m_Store.Put(&storeIndex, sdaInfo));
    NN_ABORT_UNLESS(m_AppIdList.Add(sdaInfo.applicationId, storeIndex));
    NN_ABORT_UNLESS(m_IdList.Add(sdaInfo.id, storeIndex));

    NN_RESULT_SUCCESS;
}


template<typename IndexListType, typename IdType>
Result SaveDataArchiveInfoCache::DeleteImpl(IndexListType& indexList, IdType id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_AppIdList)> appIdListLock(m_AppIdList);
    std::lock_guard<decltype(m_IdList)> idListLock(m_IdList);
    std::lock_guard<decltype(m_Store)> storeLock(m_Store);

    auto found = indexList.Find(id);

    NN_RESULT_THROW_UNLESS(found, olsc::ResultSaveDataArchiveInfoCacheNoLongerExists());

    SaveDataArchiveInfo sda;
    NN_ABORT_UNLESS(m_Store.Get(&sda, *found));
    m_Store.Remove(*found);
    m_AppIdList.Remove(sda.applicationId);
    m_IdList.Remove(sda.id);

    NN_RESULT_SUCCESS;
}

Result SaveDataArchiveInfoCache::Delete(ApplicationId id) NN_NOEXCEPT
{
    return DeleteImpl(m_AppIdList, id);
}
Result SaveDataArchiveInfoCache::Delete(DataId id) NN_NOEXCEPT
{
    return DeleteImpl(m_IdList, id);
}

template <typename T>
int SaveDataArchiveInfoCache::ListImpl(T out[], int maxOutCount, int offset, T (*convert)(SaveDataArchiveInfo&& sda) ) const NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_AppIdList)> appIdListLock(m_AppIdList);
    std::lock_guard<const decltype(m_Store)> storeLock(m_Store);

    NN_SDK_ASSERT(offset >= 0 && maxOutCount >= 0);

    auto begin = offset;
    auto end = std::min(maxOutCount + offset, GetCount());
    for (int i = begin; i < end; ++i)
    {
        auto storeIndex = m_AppIdList[i].value;
        SaveDataArchiveInfo sda;
        NN_ABORT_UNLESS(m_Store.Get(&sda, storeIndex));
        out[i - begin] = convert(std::move(sda));
    }

    return std::max(0, end - begin);
}

int SaveDataArchiveInfoCache::List(SaveDataArchiveInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT
{
    return ListImpl<SaveDataArchiveInfo>(out, maxOutCount, offset, [](SaveDataArchiveInfo&& sda) -> SaveDataArchiveInfo {
        return sda;
    });
}

int SaveDataArchiveInfoCache::ListDataInfo(DataInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT
{
    return ListImpl<DataInfo>(out, maxOutCount, offset, [](SaveDataArchiveInfo&& sda) -> DataInfo {
        return GetDataInfoFromSaveDataArchiveInfo(sda);
    });
}

Result SaveDataArchiveInfoCache::DeleteAll() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_AppIdList)> appIdListLock(m_AppIdList);
    std::lock_guard<decltype(m_IdList)> idListLock(m_IdList);
    std::lock_guard<decltype(m_Store)> storeLock(m_Store);

    m_AppIdList.Cleanup();
    m_IdList.Cleanup();
    m_Store.Cleanup();

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::database

