﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace olsc {

void swap(RemoteStorageController& lhs, RemoteStorageController& rhs) NN_NOEXCEPT
{
    std::swap(lhs.m_Interface, rhs.m_Interface);
    std::swap(lhs.m_User, rhs.m_User);
}

RemoteStorageController::RemoteStorageController() NN_NOEXCEPT
    : m_Interface(nullptr), m_User(account::InvalidUid)
{
}

RemoteStorageController::RemoteStorageController(srv::IRemoteStorageController* interface, const account::Uid& uid) NN_NOEXCEPT
    : m_Interface(interface), m_User(uid)
{
}

RemoteStorageController::RemoteStorageController(RemoteStorageController&& rhs) NN_NOEXCEPT
    : m_Interface(rhs.m_Interface), m_User(rhs.m_User)
{
    rhs.m_Interface = nullptr;
    rhs.m_User = account::InvalidUid;
}

RemoteStorageController& RemoteStorageController::operator=(RemoteStorageController&& rhs) NN_NOEXCEPT
{
    RemoteStorageController t(std::move(rhs));
    swap(*this, t);
    return *this;
}

RemoteStorageController::~RemoteStorageController() NN_NOEXCEPT
{
    if (m_Interface)
    {
        sf::ReleaseSharedObject(m_Interface);
        m_Interface = nullptr;
    }
}

Result RemoteStorageController::RegisterUploadSaveDataTransferTask(TransferTaskId* out, ApplicationId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    return m_Interface->RegisterUploadSaveDataTransferTask(out, m_User, id);
}

Result RemoteStorageController::RegisterDownloadSaveDataTransferTask(TransferTaskId* out, DataId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    return m_Interface->RegisterDownloadSaveDataTransferTask(out, m_User, id);
}

int RemoteStorageController::GetDataInfoCount() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    int ret = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetCount(&ret, m_User));
    return ret;
}

int RemoteStorageController::ListDataInfo(DataInfo outValue[], int count, int offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    int ret = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ListDataInfo(&ret, sf::OutBuffer(reinterpret_cast<char*>(outValue), count * sizeof(DataInfo)), count, m_User, offset));
    return ret;
}

bool RemoteStorageController::GetDataInfo(DataInfo* out, ApplicationId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    bool ret = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetDataInfo(&ret, out, m_User, id));
    return ret;
}

void RemoteStorageController::DeleteDataInfo(DataId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->DeleteDataInfo(m_User, id));
}

Result RemoteStorageController::ClearDataInfoCache() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    return m_Interface->ClearDataInfoCache(m_User);
}

DataNewness RemoteStorageController::GetDataNewness(DataId dataId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    DataNewness out;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetDataNewness(&out, m_User, dataId));
    return out;
}

Result RemoteStorageController::RequestUpdateDataInfoCacheAsync(AsyncRequest* outAsync) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    sf::SharedPointer<srv::IAsyncResult> p;
    NN_RESULT_DO(m_Interface->RequestUpdateDataInfoCacheAsync(&p, m_User));

    AsyncRequest t(p.Detach());
    *outAsync = std::move(t);
    NN_RESULT_SUCCESS;
}

Result RemoteStorageController::RequestUpdateDataInfoCacheAsync(AsyncRequest* outAsync, ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    sf::SharedPointer<srv::IAsyncResult> p;
    NN_RESULT_DO(m_Interface->RequestUpdateDataInfoCacheOfSpecifiedApplicationAsync(&p, m_User, appId));

    AsyncRequest t(p.Detach());
    *outAsync = std::move(t);
    NN_RESULT_SUCCESS;
}

Result RemoteStorageController::RequestDeleteData(AsyncRequest* outAsync, DataId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    sf::SharedPointer<srv::IAsyncResult> p;
    NN_RESULT_DO(m_Interface->RequestDeleteData(&p, m_User, id));

    AsyncRequest t(p.Detach());
    *outAsync = std::move(t);
    NN_RESULT_SUCCESS;
}

}} // namespace nn::olsc

