﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/fs.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ns/ns_Result.private.h>
#include <nn/ns/srv/ns_VersionManagementDatabase.h>

namespace nn { namespace ns { namespace srv {
    namespace {
        const int FormatVersion = 1;
    }

    const char* VersionListDatabaseConfig::DatabaseFileName = "version_list_db";
    const char* VersionListDatabaseConfig::HeaderFileName = "version_list_header";

    const char* RequiredVersionDatabaseConfig::DatabaseFileName = "req_version_db";
    const char* RequiredVersionDatabaseConfig::HeaderFileName = "req_version_header";


    VersionManagementDatabaseHeader VersionManagementDatabaseHeader::Make(int maxEntryCount) NN_NOEXCEPT
    {
        VersionManagementDatabaseHeader header = { FormatVersion,{ 'V', 'M', 'D', 'B' }, 0, 0, maxEntryCount };
        return header;
    }

    bool VersionManagementDatabaseHeader::IsValid() NN_NOEXCEPT
    {
        auto header = VersionManagementDatabaseHeader::Make(this->maxEntryCount);
        return header.formatVersion == this->formatVersion &&
            std::memcmp(header.signature, this->signature, sizeof(header.signature)) == 0;
    }



    VersionManagementDatabaseCommon::Path VersionManagementDatabaseCommon::MakeFilePath(const char* mountName, const char* fileName) NN_NOEXCEPT
    {
        Path path(mountName);
        path.Append(":/");
        path.Append(fileName);
        return path;
    }


    Result VersionManagementDatabaseCommon::RewriteFile(const char* fileName, const void* data, size_t dataSize) NN_NOEXCEPT
    {
        nn::fs::DeleteFile(fileName);
        NN_RESULT_DO(nn::fs::CreateFile(fileName, dataSize));

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, fileName, nn::fs::OpenMode::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

        NN_RESULT_DO(nn::fs::WriteFile(file, 0, data, dataSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result VersionManagementDatabaseCommon::ReadFile(const char* fileName, void* out, size_t readSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, fileName, nn::fs::OpenMode::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        NN_RESULT_DO(nn::fs::ReadFile(file, 0, out, readSize));
        NN_RESULT_SUCCESS;
    }

}}}
