﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_PushNotificationDisptcher.h>
#include <nn/nim/nim_Result.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>

#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>

#include <nn/util/util_LockGuard.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "ns_SystemUpdateUtil.h"
#include "ns_NotificationToken.h"

namespace nn { namespace ns { namespace srv {

    namespace
    {
        Result CreateToken(npns::NotificationToken* out) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(out);

            static NonRecursiveMutex s_Mutex;
            static util::optional<npns::NotificationToken> s_Token;

            NN_UTIL_LOCK_GUARD(s_Mutex);

            if (!s_Token)
            {
                NN_PN_TRACE("Notification token creation.\n");
                nn::ApplicationId id = { NN_NS_PROGRAM_ID_OF_NS_PROCESS };
                nn::account::Uid uid = {};
                npns::NotificationToken token = {};
                NN_RESULT_DO(nn::npns::CreateToken(&token, uid, id)); // ユーザを指定しない通知なので 0 埋めの uid を渡す
                NN_PN_TRACE("Notification token creation done.\n");
                NN_PN_TRACE("NotificationToken = %s\n", token.data);
                s_Token = token;
            }

            *out = *s_Token;
            NN_RESULT_SUCCESS;
        }

        Result RegisterNotificationToken(const npns::NotificationToken& token, os::SystemEvent* cancelEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(cancelEvent);

            NN_PN_TRACE("Notification token registration.\n");
            nn::nim::AsyncResult asyncResult;
            NN_RESULT_TRY(nn::nim::RequestRegisterNotificationToken(&asyncResult, token))
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxTask, ResultOutOfMaxRunningTask())
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxRunningTask, ResultOutOfMaxRunningTask())
            NN_RESULT_END_TRY

            nn::ns::srv::MultiWaitSystemEvent multiWait;
            auto canceled = multiWait.Link(cancelEvent);
            multiWait.Link(&asyncResult.GetEvent());

            auto signaled = multiWait.WaitAny();
            if (signaled == canceled)
            {
                NN_PN_TRACE("Notification token registration is cancelled.\n");
                return ns::ResultCanceled();
            }
            NN_RESULT_DO(asyncResult.Get());

            NN_PN_TRACE("Notification token registration done.\n");
            NN_RESULT_SUCCESS;
        }

        Result RegisterDynamicRightsNotificationToken(const npns::NotificationToken& token, os::SystemEvent* cancelEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(cancelEvent);

            NN_PN_TRACE("Dynamic rights notification token registration.\n");
            nn::nim::AsyncResult asyncResult;
            NN_RESULT_TRY(nn::nim::RequestRegisterDynamicRightsNotificationToken(&asyncResult, token))
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxTask, ResultOutOfMaxRunningTask())
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxRunningTask, ResultOutOfMaxRunningTask())
            NN_RESULT_END_TRY

            nn::ns::srv::MultiWaitSystemEvent multiWait;
            auto canceled = multiWait.Link(cancelEvent);
            multiWait.Link(&asyncResult.GetEvent());

            auto signaled = multiWait.WaitAny();
            if (signaled == canceled)
            {
                NN_PN_TRACE("Dynamic rights notification token registration is cancelled.\n");
                return ns::ResultCanceled();
            }
            NN_RESULT_DO(asyncResult.Get());

            NN_PN_TRACE("Dynamic rights notification token registration done.\n");

            // TODO: ELicense状況確認タスクの登録

            NN_RESULT_SUCCESS;
        }

        Result WaitConnection(nifm::NetworkConnection* connection, nn::os::SystemEvent* cancelEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(connection);
            NN_SDK_ASSERT_NOT_NULL(cancelEvent);

            NN_PN_TRACE("Waiting network connection.\n");

            NN_RESULT_DO(nn::nifm::SetRequestRequirementPreset(connection->GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessPersistent));
            connection->SubmitRequest();

            auto& connectionEvent = connection->GetSystemEvent();

            while (!connection->IsAvailable())
            {
                nn::ns::srv::MultiWaitSystemEvent multiWait;
                auto canceled = multiWait.Link(cancelEvent);
                multiWait.Link(&connectionEvent);

                auto signaled = multiWait.WaitAny();
                if (signaled == canceled)
                {
                    NN_PN_TRACE("Network connection canceled.\n");
                    NN_RESULT_THROW(ns::ResultCanceled());
                }
                connectionEvent.Clear();
            }
            NN_PN_TRACE("Connected.\n");

            NN_RESULT_SUCCESS;
        }

    } // namespace

    Result NotificationTokenCreationTaskBase::Process(nn::os::SystemEvent* cancelEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(cancelEvent);

        nn::npns::NotificationToken token = {}; // npns::CreateToken でも末尾 0 埋めしてくれるようだが念のためここでもクリアしとく
        NN_RESULT_DO(CreateToken(&token));

        // SIGLO-50859
        // TODO: 登録済みのトークンをセーブデータに記録しておき、一致する場合は nim へのリクエストを投げない
        //
        // if(token == latestToken) { NN_RESULT_SUCCESS; }
        // NN_RESULT_DO(UpdateLatestToken(token));

        nifm::NetworkConnection connection(nn::os::EventClearMode_ManualClear);
        NN_RESULT_DO(WaitConnection(&connection, cancelEvent));

        NN_RESULT_DO(RegisterImpl(token, cancelEvent));

        UpdateStatus(Finished);

        NN_RESULT_SUCCESS;
    }

    Result NotificationTokenCreationTask::RegisterImpl(const nn::npns::NotificationToken& token, nn::os::SystemEvent* cancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_DO(RegisterNotificationToken(token, cancelEvent));
        NN_RESULT_SUCCESS;
    }

    Result DynamicRightsNotificationTokenCreationTask::RegisterImpl(const nn::npns::NotificationToken& token, nn::os::SystemEvent* cancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_DO(RegisterDynamicRightsNotificationToken(token, cancelEvent));
        NN_RESULT_SUCCESS;
    }
}}}
