﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/srv/ns_MarkFile.h>
#include <cstring>
#include <nn/os/os_Random.h>

namespace nn { namespace ns { namespace srv {
    namespace
    {
        MarkData GenerateMarkData() NN_NOEXCEPT
        {
            MarkData id;
            os::GenerateRandomBytes(id.data, sizeof(id.data));

            return id;
        }
    }

    void MarkFile::Initialize(const char* directoryPath) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(directoryPath);
        size_t length = std::strlen(directoryPath);
        NN_UNUSED(length);
        // 後で追加されるかもしれない '/' の分も含めて長さ計算をする
        NN_SDK_REQUIRES_NOT_EQUAL(length, 0u);
        NN_SDK_REQUIRES_LESS(length, MaxPathLength - std::strlen(MarkFileName) - 1);

        m_MarkFileDirectory.Assign(directoryPath);
        Path* path = &m_MarkFileDirectory;
        if (m_MarkFileDirectory[m_MarkFileDirectory.GetLength() - 1] != '/')
        {
            path->Append("/");
        }

        m_MarkFilePath.AssignFormat("%s%s", m_MarkFileDirectory.Get(), MarkFileName);
    }

    Result MarkFile::HasMarkFile(bool* outValue) NN_NOEXCEPT
    {
        fs::DirectoryEntryType entry;
        NN_RESULT_TRY(fs::GetEntryType(&entry, m_MarkFilePath))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result MarkFile::GenerateMarkFile(const void* pExternalData, size_t externalDataSize) NN_NOEXCEPT
    {
        MarkData data;
        data = GenerateMarkData();
        NN_RESULT_DO(GenerateMarkFile(&data, pExternalData, externalDataSize));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::GenerateMarkFile(const MarkData* pData, const void* pExternalData, size_t externalDataSize) NN_NOEXCEPT
    {
        GenerateDirectory();

        NN_RESULT_DO(fs::CreateFile(m_MarkFilePath, MarkData::Size + externalDataSize));

        NN_RESULT_DO(WriteMarkData(pData));
        NN_RESULT_DO(WriteExternalData(pExternalData, externalDataSize));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::DeleteMarkFile() NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::DeleteFile(m_MarkFilePath));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::ReadMarkData(MarkData* outValue) const NN_NOEXCEPT
    {
        fs::FileHandle fileHandle;
        NN_RESULT_DO(fs::OpenFile(&fileHandle, m_MarkFilePath, fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(fileHandle);
        };

        NN_RESULT_DO(fs::ReadFile(fileHandle, 0, outValue, MarkData::Size));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::ReadExternalData(void* outBuffer, size_t size) const NN_NOEXCEPT
    {
        fs::FileHandle fileHandle;
        NN_RESULT_DO(fs::OpenFile(&fileHandle, m_MarkFilePath, fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(fileHandle);
        };

        NN_RESULT_DO(fs::ReadFile(fileHandle, MarkData::Size, outBuffer, size));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::WriteMarkData(const MarkData* pData) NN_NOEXCEPT
    {
        fs::FileHandle fileHandle;
        NN_RESULT_DO(fs::OpenFile(&fileHandle, m_MarkFilePath, fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(fileHandle);
        };

        NN_RESULT_DO(fs::WriteFile(fileHandle, 0, pData, MarkData::Size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::WriteExternalData(const void* pData, size_t size) NN_NOEXCEPT
    {
        if (size == 0)
        {
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle fileHandle;
        NN_RESULT_DO(fs::OpenFile(&fileHandle, m_MarkFilePath, fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(fileHandle);
        };

        NN_RESULT_DO(fs::WriteFile(fileHandle, MarkData::Size, pData, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    const char* MarkFile::GetMarkFilePath() const NN_NOEXCEPT
    {
        return m_MarkFilePath;
    }

    Result MarkFile::GenerateDirectoryImpl(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::CreateDirectory(path))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists){}
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result MarkFile::GenerateDirectory() NN_NOEXCEPT
    {
        const char* dirPath = m_MarkFileDirectory;
        for (size_t i = 0; i < m_MarkFileDirectory.GetLength(); i++)
        {
            if (i > 0 && dirPath[i] == '/' && dirPath[i - 1] != ':')
            {
                Path subPath = m_MarkFileDirectory.MakeSubString(0, i);
                NN_RESULT_DO(GenerateDirectoryImpl(subPath));
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result MarkFile::GetExternalDataSize(size_t* outValue) const NN_NOEXCEPT
    {
        fs::FileHandle fileHandle;
        NN_RESULT_DO(fs::OpenFile(&fileHandle, m_MarkFilePath, fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(fileHandle);
        };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, fileHandle));

        // 書き込み時に size_t を指定しているので、 size_t 以下にしかならないはず
        *outValue = static_cast<size_t>(fileSize - MarkData::Size);

        NN_RESULT_SUCCESS;
    }
}}}
