﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_Optional.h>
#include "ns_StringUtil.h"
#include "ns_DebugUtil.h"

namespace nn { namespace ns { namespace srv {
    bool GetSettingsItemValueImpl(void* outValue, size_t size, const char* name, const char* key, bool skipSizeCheck = false) NN_NOEXCEPT
    {
        auto resultSize = settings::fwdbg::GetSettingsItemValue(outValue, size, name, key);
        if (resultSize != size && !skipSizeCheck)
        {
            NN_DETAIL_NS_WARN("[ns] Failed to get firmware debug settings. (name='%s', key='%s')\n", name, key);
            return false;
        }
        return true;
    }

    util::optional<ncm::ContentMetaKey> GetDebugSystemUpdateMeta() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return util::nullopt;
#else
        char debugIdString[64] = {};
        if (!GetSettingsItemValueImpl(debugIdString, sizeof(debugIdString), "systemupdate", "debug_id", true))
        {
            return util::nullopt;
        }

        // 念のため
        debugIdString[63] = '\0';
        auto debugId = NN_NS_STR_TO_ULL(debugIdString, nullptr, 16);
        if (debugId == 0)
        {
            return util::nullopt;
        }

        uint32_t debugVersion;
        settings::fwdbg::GetSettingsItemValue(&debugVersion, sizeof(debugVersion), "systemupdate", "debug_version");

        return ncm::ContentMetaKey::Make(debugId, debugVersion, ncm::ContentMetaType::SystemUpdate);
#endif
    }

    bool IsBgnupEnabled() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return false;
#else
        bool isBgnupEnabled;
        if (!GetSettingsItemValueImpl(&isBgnupEnabled, sizeof(isBgnupEnabled), "systemupdate", "bgnup_enabled"))
        {
            return false;
        }

        return isBgnupEnabled;
#endif
    }

    TimeSpan GetBgnupRetrySpan() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return TimeSpan::FromSeconds(60);
#else
        uint32_t seconds;
        if (!GetSettingsItemValueImpl(&seconds, sizeof(seconds), "systemupdate", "bgnup_retry_seconds"))
        {
            return TimeSpan::FromSeconds(60);
        }

        return TimeSpan::FromSeconds(seconds);
#endif
    }

    bool IsMountSdCardEnabled() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return true;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<bool> s_Enabled;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_Enabled)
        {
            bool enabled;
            if (!GetSettingsItemValueImpl(&enabled, sizeof(enabled), "ns.sdcard", "mount_sdcard"))
            {
                s_Enabled = true;
            }
            else
            {
                s_Enabled = enabled;
            }
        }

        return *s_Enabled;
#endif
    }

    SdCardCidState GetSdCardCidState() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return SdCardCidState::Same;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<SdCardCidState> s_State;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_State)
        {
            int state;
            if (!GetSettingsItemValueImpl(&state, sizeof(state), "ns.sdcard", "compare_sdcard"))
            {
                s_State = SdCardCidState::Real;
            }
            else
            {
                s_State = static_cast<SdCardCidState>(state);
            }
        }

        return *s_State;
#endif
    }

    Result GetMountGameCardResultForDebug() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_RESULT_SUCCESS;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<Result> s_Result;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_Result)
        {
            Bit32 innerValue;
            if (!GetSettingsItemValueImpl(&innerValue, sizeof(innerValue), "ns.gamecard", "mount_gamecard_result_value"))
            {
                // 型が違うので、一旦 emplace する
                s_Result.emplace();
                *s_Result = ResultSuccess();
            }
            else
            {
                s_Result = result::detail::ConstructResult(innerValue);
            }
        }

        return *s_Result;
#endif
    }

    Result GetCleanupSdCardResultForDebug() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_RESULT_SUCCESS;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<Result> s_Result;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_Result)
        {
            Bit32 innerValue;
            if (!GetSettingsItemValueImpl(&innerValue, sizeof(innerValue), "ns.sdcard", "cleanup_sdcard_result_value"))
            {
                // 型が違うので、一旦 emplace する
                s_Result.emplace();
                *s_Result = ResultSuccess();
            }
            else
            {
                s_Result = result::detail::ConstructResult(innerValue);
            }
        }

        return *s_Result;
#endif
    }

    Result GetTryGameCardAccessResultForDebug() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_RESULT_SUCCESS;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<Result> s_Result;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_Result)
        {
            Bit32 innerValue;
            if (!GetSettingsItemValueImpl(&innerValue, sizeof(innerValue), "ns.gamecard", "try_gamecard_access_result_value"))
            {
                // 型が違うので、一旦 emplace する
                s_Result.emplace();
                *s_Result = ResultSuccess();
            }
            else
            {
                s_Result = result::detail::ConstructResult(innerValue);
            }
        }

        return *s_Result;
#endif
    }

    Result GetCleanupAddOnContentWithNoRightsResultForDebug() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_RESULT_SUCCESS;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<Result> s_Result;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_Result)
        {
            Bit32 innerValue;
            if (!GetSettingsItemValueImpl(&innerValue, sizeof(innerValue), "ns.application", "cleanup_add_on_content_with_no_rights_result_value"))
            {
                // 型が違うので、一旦 emplace する
                s_Result.emplace();
                *s_Result = ResultSuccess();
            }
            else
            {
                s_Result = result::detail::ConstructResult(innerValue);
            }
        }

        return *s_Result;
#endif
    }

    util::optional<Bit64> GetDebugSystemUpdateMetaIdForSystemDeliveryInfo() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return util::nullopt;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<Bit64> s_Id;
        static bool s_IsFirst = false;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);

        if (!s_IsFirst)
        {
            s_IsFirst = true;
            char debugIdString[64] = {};
            if (!GetSettingsItemValueImpl(debugIdString, sizeof(debugIdString), "contents_delivery", "system_delivery_info_id", true))
            {
                return util::nullopt;
            }

            // 念のため
            debugIdString[63] = '\0';
            auto id = NN_NS_STR_TO_ULL(debugIdString, nullptr, 16);
            s_Id = (id == 0) ? util::nullopt : util::optional<Bit64>(id);
        }

        return s_Id;
#endif
    }

    bool IsCleanupOrphanCacheStorageEnabledForDebug() NN_NOEXCEPT
    {
        const bool defaultValue = true;
#ifdef NN_BUILD_CONFIG_OS_WIN
        return defaultValue;
#else
        static util::optional<bool> s_IsEnabled;
        if (!s_IsEnabled)
        {
            bool isEnabled;
            if (!GetSettingsItemValueImpl(&isEnabled, sizeof(isEnabled), "ns.application", "cleanup_cache_storage_enabled"))
            {
                s_IsEnabled = defaultValue;
                return defaultValue;
            }
            s_IsEnabled = isEnabled;
        }

        return *s_IsEnabled;
#endif
    }

    void GetApplicationInfoOnBlackListForDebug(ncm::ApplicationId* outId, uint32_t* outVersion) NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        *outId = ncm::ApplicationId::GetInvalidId();
        *outVersion = 0;
        return;
#else
        static NonRecursiveMutex s_Mutex;
        static bool s_IsInitialized = false;
        static ncm::ApplicationId s_Id = ncm::ApplicationId::GetInvalidId();
        static uint32_t s_Version = 0;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_IsInitialized)
        {
            char debugIdString[64] = {};
            if (GetSettingsItemValueImpl(debugIdString, sizeof(debugIdString), "ns.application", "application_id_on_blacklist", true))
            {
                // 念のため
                debugIdString[63] = '\0';
                auto debugId = NN_NS_STR_TO_ULL(debugIdString, nullptr, 16);
                s_Id = { debugId };
            }

            uint32_t version;
            if (GetSettingsItemValueImpl(&version, sizeof(version), "ns.application", "application_version_on_blacklist"))
            {
                s_Version = version;
            }
            s_IsInitialized = true;
        }

        *outId = s_Id;
        *outVersion = s_Version;
        return;
#endif
    }

    util::optional<int> GetLicenseTypeForDebug() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return util::nullopt;
#else
        static NonRecursiveMutex s_Mutex;
        static bool s_IsInitialized = false;
        static util::optional<int> s_LicenseType;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
        if (!s_IsInitialized)
        {
            s_IsInitialized = true;
            int licenseType = 0;
            if (GetSettingsItemValueImpl(&licenseType, sizeof(licenseType), "ns.application", "debug_license_type"))
            {
                if (licenseType != 0)
                {
                    s_LicenseType = licenseType;
                }
            }
        }

        return s_LicenseType;
#endif

    }

    bool IsDebugELicenseImportEnabled() NN_NOEXCEPT
    {
        const bool defaultValue = false;
#ifdef NN_BUILD_CONFIG_OS_WIN
        return defaultValue;
#else
        static util::optional<bool> s_IsEnabled;
        if (!s_IsEnabled)
        {
            bool isEnabled;
            if (!GetSettingsItemValueImpl(&isEnabled, sizeof(isEnabled), "ns.application", "debug_elicense_import"))
            {
                s_IsEnabled = defaultValue;
                return defaultValue;
            }
            s_IsEnabled = isEnabled;
        }

        return *s_IsEnabled;
#endif
    }
}}}
