﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/es/es_RightsApi.h>
#include <nn/es/es_Result.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationVersionManager.h>
#include <nn/ns/srv/ns_BlackListManager.h>
#include <nn/ns/srv/ns_CommitManager.h>
#include <nn/ns/srv/ns_ContentDeliveryManager.h>
#include <nn/ns/srv/ns_GameCardManager.h>
#include <nn/ns/srv/ns_ExFatDriverManager.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>

#include "ns_ApplicationInstallTask.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_AsyncContentDeliveryImpl.h"
#include "ns_DebugUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_ReceiveApplicationTask.h"
#include "ns_SendApplicationTask.h"
#include "ns_SystemUpdateUtil.h"

namespace nn { namespace ns { namespace srv {

namespace {

typedef SystemDeliveryAttribute::Flag<0>    SystemDeliveryAttribute_IncludesExFatDriver;

ApplicationDeliveryAttribute ApplicationDeliveryAttribute_RequestAll = ApplicationDeliveryAttribute_RequestApplication::Mask
                                                                     | ApplicationDeliveryAttribute_RequestPatch::Mask;

typedef ApplicationDeliveryAttribute::Flag<31>    ApplicationDeliveryAttribute_HasApplicationRecord;
typedef ApplicationDeliveryAttribute::Flag<30>    ApplicationDeliveryAttribute_HasPatchRecord;
typedef ApplicationDeliveryAttribute::Flag<29>    ApplicationDeliveryAttribute_HasApplicationEntity;
typedef ApplicationDeliveryAttribute::Flag<28>    ApplicationDeliveryAttribute_HasPatchEntity;

// 再起動後もアプリケーションの状態が変わっているかを検知するためなので、互換を保つ必要がある
struct ApplicationDeliveryInfoToCalculateHash
{
    ncm::ApplicationId id;
    uint32_t version;
    uint32_t requiredVersion;
    uint32_t requiredSystemVersion;
    Bit8 reserved[4];

    bool operator <(const ApplicationDeliveryInfoToCalculateHash& rhs) const NN_NOEXCEPT
    {
        return id.value < rhs.id.value;
    }
};

// RFC 2104 より、長さは SHA256 の 256 bit に合わせる
const Bit8 SystemDeliveryInfoKeyForHmac[] =
{
    0x56, 0x80, 0x12, 0x1b, 0xb6, 0x73, 0xe0, 0x42, 0x89, 0xef, 0x4a, 0xe6, 0xd7, 0xae, 0xf6, 0xf6,
    0xb3, 0xa8, 0xcd, 0x67, 0x76, 0x51, 0x95, 0x73, 0x19, 0xa2, 0xfa, 0x59, 0x8f, 0x32, 0xec, 0xcb,
};

const Bit8 ApplicationDeliveryInfoKeyForHmac[] =
{
    0x1e, 0xa1, 0xfe, 0x6a, 0x41, 0xe2, 0x43, 0x8b, 0x3a, 0x38, 0x23, 0x57, 0xdd, 0x03, 0xaf, 0x62,
    0x7a, 0x44, 0x3e, 0xe0, 0x98, 0xae, 0xc4, 0x45, 0xf9, 0x44, 0x6f, 0x18, 0xae, 0x5f, 0x97, 0xc4,
};

bool IncludesExFatDriver(SystemDeliveryAttribute attributes) NN_NOEXCEPT
{
    return attributes.Test<SystemDeliveryAttribute_IncludesExFatDriver>();
}

bool HasApplicationInstallTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationInstallTask task(id);
    return task.IsValid();
}

bool HasApplicationApplyDeltaTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationApplyDeltaTask task(id);
    return task.IsValid();
}

template <class FuncT>
Result ForEachContentMetaInRecord(ApplicationRecordDatabase* pDb, ncm::ApplicationId id, FuncT func) NN_NOEXCEPT
{
    int offset = 0;
    while (NN_STATIC_CONDITION(true))
    {
        ncm::StorageContentMetaKey keyList[16];
        const int ListCount = static_cast<int>(NN_ARRAY_SIZE(keyList));

        int count;
        NN_RESULT_DO(pDb->ListContentMeta(&count, keyList, ListCount, id, offset));

        for (int i = 0; i < count; i++)
        {
            NN_RESULT_DO(func(keyList[i].key));
        }

        if (count < ListCount)
        {
            break;
        }
        offset += count;
    }
    NN_RESULT_SUCCESS;
}

bool GetSettingsUint32Value(uint32_t* outValue, const char* name, const char* key) NN_NOEXCEPT
{
    const auto TypeSize = sizeof(uint32_t);
    auto resultSize = settings::fwdbg::GetSettingsItemValue(outValue, TypeSize, name, key);
    if (resultSize != TypeSize)
    {
        NN_DETAIL_NS_WARN("[ns] Failed to get firmware debug settings. (name='%s', key='%s')\n", name, key);
        return false;
    }
    return true;
}

util::optional<ncm::ContentMetaKey> GetSystemUpdateMetaForContentDelivery() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    return util::nullopt;
#else
    auto debugId = GetDebugSystemUpdateMetaIdForSystemDeliveryInfo();
    if (debugId)
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));
        // TORIAEZU: 16 もあれば余裕なはず
        const int ListCount = 16;
        ncm::ContentMetaKey keyList[ListCount];
        auto count = db.ListContentMeta(keyList, ListCount, ncm::ContentMetaType::SystemUpdate);
        for (int i = 0; i < count.listed; i++)
        {
            if (keyList[i].id == debugId)
            {
                return keyList[i];
            }
        }
        return util::nullopt;
    }
    else
    {
        return GetSystemUpdateMetaKey();
    }
#endif
}

bool IsContentMetaTypeForApplicationDownloadTask(ncm::ContentMetaType type) NN_NOEXCEPT
{
    return ncm::ContentMetaType::Application == type
        || ncm::ContentMetaType::Patch == type
        || ncm::ContentMetaType::AddOnContent == type;
}

uint32_t GetAcceptableApplicationProtocolVersion() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    return 0;
#else
    static util::optional<uint32_t> s_Version;
    if (!s_Version)
    {
        uint32_t version;
        if (!GetSettingsUint32Value(&version, "contents_delivery", "acceptable_application_delivery_protocol_version"))
        {
            version = 0;
        }
        s_Version = version;
    }
    return *s_Version;
#endif
}

int GetCompareValue(uint32_t lhs, uint32_t rhs) NN_NOEXCEPT
{
    if (lhs > rhs)
    {
        return 1;
    }
    else if (lhs == rhs)
    {
        return 0;
    }
    else
    {
        return -1;
    }
}
}

Result ContentDeliveryManager::Initialize(ApplicationRecordDatabase* pRecordDb, ApplicationVersionManager* pVersionManager, BlackListManager* pBlackListManager, CommitManager* pCommitManager, GameCardManager* pGameCardManager, IntegratedContentManager* pIntegrated, RequestServer* pRequestServer) NN_NOEXCEPT
{
    m_pRecordDb = pRecordDb;
    m_pVersionManager = pVersionManager;
    m_pBlackListManager = pBlackListManager;
    m_pCommitManager = pCommitManager;
    m_pGameCardManager = pGameCardManager;
    m_pIntegrated = pIntegrated;
    m_pRequestServer = pRequestServer;
#ifdef NN_BUILD_CONFIG_OS_WIN
    m_RequiredSystemVersion = 0;
#else
    if (!GetSettingsUint32Value(&m_RequiredSystemVersion, "contents_delivery", "required_system_version_to_deliver_application"))
    {
        m_RequiredSystemVersion = 0;
    }
#endif

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetSystemDeliveryInfo(SystemDeliveryInfo* outValue) NN_NOEXCEPT
{
    SystemDeliveryInfo info = {};
    info.systemDeliveryProtocolVersion = GetSystemDeliveryProtocolVersion();
    info.applicationDeliveryProtocolVersion = GetApplicationDeliveryProtocolVersion();

    auto systemUpdateKey = GetSystemUpdateMetaForContentDelivery();
    NN_RESULT_THROW_UNLESS(systemUpdateKey, ResultSystemUpdateMetaKeyNotFound());
    info.systemUpdateId = systemUpdateKey->id;
    info.systemUpdateVersion = systemUpdateKey->version;

    bool hasExFatDriver;
    NN_RESULT_DO(ExFatDriverManager::IsExFatInstalled(&hasExFatDriver));
    if (hasExFatDriver)
    {
        info.attributes.Set<SystemDeliveryAttribute_IncludesExFatDriver>(true);
    }

    crypto::GenerateHmacSha256Mac(
        &info.mac, sizeof(info.mac),
        &info, sizeof(info) - sizeof(info.mac),
        SystemDeliveryInfoKeyForHmac, sizeof(SystemDeliveryInfoKeyForHmac));

    *outValue = info;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::CompareSystemDeliveryInfo(int* outValue, const SystemDeliveryInfo& lhs, const SystemDeliveryInfo& rhs) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifySystemDeliveryInfo(lhs, true));
    NN_RESULT_DO(VerifySystemDeliveryInfo(rhs, true));

    *outValue = GetCompareValue(lhs.systemUpdateVersion, rhs.systemUpdateVersion);
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::SelectLatestSystemDeliveryInfo(
    int* outValue, const SystemDeliveryInfo list[], int numList, const SystemDeliveryInfo &receiverSystemInfo,
    const ApplicationDeliveryInfo receiverAppInfo[], int numAppList) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifySystemDeliveryInfo(receiverSystemInfo, numAppList == 0));

    int index = -1;
    uint32_t maxSystemUpdateVersion = 0;
    uint32_t requiredSystemVersion = 0;

    for (int i = 0; i < numAppList; i++)
    {
        NN_RESULT_DO(VerifyApplicationDeliveryInfo(receiverAppInfo[i]));
        requiredSystemVersion = std::max(receiverAppInfo[i].requiredSystemVersion, requiredSystemVersion);
    }

    requiredSystemVersion = std::max(requiredSystemVersion, m_RequiredSystemVersion);

    for (int i = 0; i < numList; i++)
    {
        NN_RESULT_DO(VerifySystemDeliveryInfo(list[i]));

        if (
        // INFO:    システム上には SystemUpdate が一つしかない前提
        //          SystemDeliveryInfo::systemUpdateVersion には、必ず RequiredSystemVersion と比較すべき値が入っているとする。
         list[i].systemUpdateVersion >= requiredSystemVersion
         &&  (IncludesExFatDriver(list[i].attributes) || !IncludesExFatDriver(receiverSystemInfo.attributes))
         &&  maxSystemUpdateVersion < list[i].systemUpdateVersion )
        {
            maxSystemUpdateVersion = list[i].systemUpdateVersion;
            index = i;
        }
    }

    *outValue = index;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyDeliveryProtocolVersion(const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyMacOfSystemDeliveryInfo(info));
    // アプリの更新は出来ないけど、システムの更新は出来るというケースを優先するため、システムバージョンを先に検証する
    NN_RESULT_DO(VerifySystemDeliveryProtocolCompatibility(info));
    NN_RESULT_DO(VerifyApplicationDeliveryProtocolCompatibility(info));

    // TODO: SIGLO-70183の暫定対応。5.0 NUP で修正する
    NN_RESULT_THROW_UNLESS(info.systemUpdateVersion >= m_RequiredSystemVersion, ResultSystemUpdateRequiredForLocalContentDelivery());

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetApplicationDeliveryInfo(int* outCount, ApplicationDeliveryInfo* outList, int numList, ncm::ApplicationId id, ApplicationDeliveryAttribute attributes) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(numList > 0, ResultBufferNotEnough());
    NN_RESULT_THROW_UNLESS(!(attributes & ~ApplicationDeliveryAttribute_RequestAll).IsAnyOn(), ResultInvalidContentDeliveryRequest());

    // TORIAEZU: 現在はパッチのみ対応
    NN_RESULT_THROW_UNLESS(!(attributes & ~ApplicationDeliveryAttribute_RequestPatch::Mask).IsAnyOn(), ResultInvalidContentDeliveryRequest());

    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    // TODO: アプリ対応をすることになったら、この条件を外す
    NN_RESULT_THROW_UNLESS(m_pRecordDb->Has(id), ResultApplicationRecordNotFound());

    ApplicationDeliveryInfo info {};
    info.applicationDeliveryProtocolVersion = GetApplicationDeliveryProtocolVersion();
    info.id = id;
    info.attributes = attributes;

    util::optional<ncm::ContentMetaKey> appKey;
    util::optional<ncm::ContentMetaKey> patchKey;

    NN_RESULT_DO(ForEachContentMetaInRecord(m_pRecordDb, id, [&appKey, &patchKey](const ncm::ContentMetaKey& key) -> Result
        {
            if (key.type == ncm::ContentMetaType::Application)
            {
                appKey = key;
            }
            else if (key.type == ncm::ContentMetaType::Patch)
            {
                patchKey = key;
            }
            NN_RESULT_SUCCESS;
        }));

    NN_RESULT_THROW_UNLESS(appKey || patchKey, ResultInvalidContentDeliveryRequest());

    NN_RESULT_DO(m_pVersionManager->GetLaunchRequiredVersion(&info.requiredVersion, id));

    if (patchKey)
    {
        info.version = patchKey->version;
        info.attributes.Set<ApplicationDeliveryAttribute_HasPatchRecord>(true);

        bool hasEntity;
        NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasEntity, *patchKey));
        if (hasEntity)
        {
            info.attributes.Set<ApplicationDeliveryAttribute_HasPatchEntity>(true);
            NN_RESULT_DO(m_pIntegrated->GetRequiredSystemVersion(&info.requiredSystemVersion, *patchKey));
        }
    }
    else if (appKey)
    {
        info.version = appKey->version;

        bool hasEntity;
        NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&hasEntity, *appKey));
        if (hasEntity)
        {
            NN_RESULT_DO(m_pIntegrated->GetRequiredSystemVersion(&info.requiredSystemVersion, *appKey));
        }
    }

    crypto::GenerateHmacSha256Mac(
        &info.mac, sizeof(info.mac),
        &info, sizeof(info) - sizeof(info.mac),
        ApplicationDeliveryInfoKeyForHmac, sizeof(ApplicationDeliveryInfoKeyForHmac));

    outList[0] = info;
    *outCount = 1;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::HasAllContentsToDeliver(bool* outValue, const ApplicationDeliveryInfo appList[], int numList) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(numList == 1, ResultInvalidContentDeliveryRequest());

    // TORIAEZU: 子機などが出てきたらこの判定条件を変える
    for (int i = 0; i < numList; i++)
    {
        NN_RESULT_DO(VerifyApplicationDeliveryInfo(appList[i]));
        if (appList[i].attributes.Test<ApplicationDeliveryAttribute_RequestPatch>() && !appList[i].attributes.Test<ApplicationDeliveryAttribute_HasPatchEntity>())
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::CompareApplicationDeliveryInfo(int* outValue, const ApplicationDeliveryInfo lhsList[], int numLhsList, const ApplicationDeliveryInfo rhsList[], int numRhsList) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(numLhsList == 1 && numRhsList == 1, ResultInvalidContentDeliveryRequest());
    NN_RESULT_DO(VerifyApplicationDeliveryInfo(lhsList[0]));
    NN_RESULT_DO(VerifyApplicationDeliveryInfo(rhsList[0]));

    *outValue = GetCompareValue(lhsList[0].version, rhsList[0].version);
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::CanDeliverApplication(bool* outValue, const ApplicationDeliveryInfo receiverList[], int numReceiverList, const ApplicationDeliveryInfo senderList[], int numSenderList) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(numSenderList == 1, ResultInvalidContentDeliveryRequest());
    const auto& senderInfo = senderList[0];
    // senderInfo の検証
    {
        NN_RESULT_DO(VerifyApplicationDeliveryInfo(senderInfo));

        // 現在はパッチ配信のみ
        NN_RESULT_THROW_UNLESS(senderInfo.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>(), ResultInvalidContentDeliveryRequest());
        NN_RESULT_THROW_UNLESS(!senderInfo.attributes.Test<ApplicationDeliveryAttribute_RequestApplication>(), ResultInvalidContentDeliveryRequest());

        NN_RESULT_THROW_UNLESS(senderInfo.attributes.Test<ApplicationDeliveryAttribute_HasPatchEntity>(), ResultContentEntityNotFoundForContentDelivery());
        NN_RESULT_THROW_UNLESS(!m_pBlackListManager->IsApplicationOnBlackList(senderInfo.id, senderInfo.version), ResultApplicationUpdateRequiredByBlackList());
    }

    if (numReceiverList == 0)
    {
        // 送信者として必要な情報は持っているが、指定された受信者には送れないので、 False を返す
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(numReceiverList == 1, ResultInvalidContentDeliveryRequest());
    const auto& receiverInfo = receiverList[0];

    // receiverInfo の検証
    {
        NN_RESULT_DO(VerifyApplicationDeliveryInfo(receiverInfo));

        // 現在はパッチ配信のみ
        NN_RESULT_THROW_UNLESS(receiverInfo.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>(), ResultInvalidContentDeliveryRequest());
        NN_RESULT_THROW_UNLESS(receiverInfo.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>() == senderInfo.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>(), ResultInvalidContentDeliveryRequest());
        NN_RESULT_THROW_UNLESS(!receiverInfo.attributes.Test<ApplicationDeliveryAttribute_RequestApplication>(), ResultInvalidContentDeliveryRequest());
    }

    if (receiverInfo.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>())
    {
        if (receiverInfo.attributes.Test<ApplicationDeliveryAttribute_HasPatchEntity>())
        {
            // 同じバージョン値を許容する
            if (receiverInfo.version >= senderInfo.version)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            // 同じバージョン値は許容しない
            if (std::max(receiverInfo.version, receiverInfo.requiredVersion) > senderInfo.version)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::ListContentMetaKeyToDeliverApplication(
    int* outCount, ncm::ContentMetaKey outList[], int numList, int offset,
    const ApplicationDeliveryInfo infoList[], int numInfoList) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(numList > 0, ResultBufferNotEnough());
    NN_RESULT_THROW_UNLESS(numInfoList == 1, ResultInvalidContentDeliveryRequest());
    const auto& info = infoList[0];
    NN_RESULT_DO(VerifyApplicationDeliveryInfo(info));

    const bool NeedsApplication = info.attributes.Test<ApplicationDeliveryAttribute_RequestApplication>();
    const bool NeedsPatch = info.attributes.Test<ApplicationDeliveryAttribute_RequestPatch>();

    NN_RESULT_THROW_UNLESS(!NeedsApplication, ResultInvalidContentDeliveryRequest());

    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(info.id), ResultApplyDeltaTaskExists());

    util::optional<ncm::ContentMetaKey> appKey;
    util::optional<ncm::ContentMetaKey> patchKey;
    NN_RESULT_DO(m_pRecordDb->FindLaunchableApplicationAndPatch(&appKey, &patchKey, info.id));

    int count = 0;
    if (NeedsPatch && count >= offset)
    {
        NN_RESULT_THROW_UNLESS(patchKey, ResultContentEntityNotFoundForContentDelivery());
        outList[count++] = *patchKey;
    }

    *outCount = count;

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::NeedsSystemUpdateToDeliverApplication(bool* outValue, const ApplicationDeliveryInfo applicationList[], int numApplicationList, const SystemDeliveryInfo& receiverSystemInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifySystemDeliveryInfo(receiverSystemInfo));
    NN_RESULT_THROW_UNLESS(numApplicationList == 1, ResultInvalidContentDeliveryRequest());
    NN_RESULT_DO(VerifyApplicationDeliveryInfo(applicationList[0]));

    // INFO:    システム上には SystemUpdate が一つしかない前提
    //          SystemDeliveryInfo::systemUpdateVersion には、必ず RequiredSystemVersion と比較すべき値が入っているとする。
    *outValue = applicationList[0].requiredSystemVersion > receiverSystemInfo.systemUpdateVersion || receiverSystemInfo.systemUpdateVersion < m_RequiredSystemVersion;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey keyList[], int numKey) NN_NOEXCEPT
{
    int64_t size = 0;
    for (int i = 0; i < numKey; i++)
    {
        int64_t requiredSize;
        NN_RESULT_DO(m_pIntegrated->EstimateRequiredSize(&requiredSize, keyList[i]));
        size += requiredSize;
    }

    *outValue = size;
    NN_RESULT_SUCCESS;
}

typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;
Result ContentDeliveryManager::RequestReceiveApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id, ncm::StorageId storageId) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationInstallTask(id), ResultDownloadTaskExists());
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    // 現状、記録がないときパッチは受け取れない仕様
    NN_RESULT_THROW_UNLESS(m_pRecordDb->Has(id), ResultInvalidContentDeliveryRequest());

    for (int i = 0; i < static_cast<int>(keyList.GetLength()); i++)
    {
        NN_RESULT_THROW_UNLESS(keyList[i].type == ncm::ContentMetaType::Patch, ResultInvalidContentDeliveryRequest());
    }

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncReceiveApplicationImpl>();
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
    NN_RESULT_DO(emplacedRef.GetImpl().Initialize(id, ipv4, port, keyList.GetData(), static_cast<int>(keyList.GetLength()), storageId));

    NN_RESULT_DO(emplacedRef.GetImpl().Run());

    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::CommitReceiveApplication(ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationInstallTask(id), ResultDownloadTaskExists());
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    NN_RESULT_DO(m_pCommitManager->CommitReceiveTask(id));
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetReceiveApplicationProgress(ReceiveApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationInstallTask(id), ResultDownloadTaskExists());
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    ReceiveApplicationTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultReceiveApplicationTaskNotFound());

    NN_RESULT_DO(task.GetProgress(outValue));
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::RequestSendApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, sf::InArray<ncm::ContentMetaKey> keyList, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncSendApplicationImpl>();
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Initialize(id, ipv4, port));

    for (int i = 0; i < static_cast<int>(keyList.GetLength()); i++)
    {
        NN_RESULT_THROW_UNLESS(keyList[i].type == ncm::ContentMetaType::Patch, ResultInvalidContentDeliveryRequest());

        bool hasRecord;
        bool isInstalled;
        NN_RESULT_DO(m_pRecordDb->HasContentMetaKey(&hasRecord, &isInstalled, keyList[i], id));
        NN_RESULT_THROW_UNLESS(hasRecord, ResultApplicationRecordNotFound());
        NN_UNUSED(isInstalled);


        ncm::StorageId storageId;
        NN_RESULT_DO(m_pIntegrated->GetContentMetaStorage(&storageId, keyList[i]));
        ncm::StorageContentMetaKey storageKey = { keyList[i], storageId };
        // 起動前だと CommonTicket がインポートされていない可能性があるのでインポートしておく
        if (keyList[i].type == ncm::ContentMetaType::Patch && storageId == ncm::StorageId::Card)
        {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
            int offset = 0;

            while (NN_STATIC_CONDITION(true))
            {
                const int ListCount = 16;
                ncm::ContentInfo infoList[ListCount];
                int outCount;
                NN_RESULT_DO(m_pIntegrated->ListContentInfo(&outCount, infoList, ListCount, keyList[i], offset, storageId));

                for (int j = 0; j < outCount; j++)
                {
                    fs::RightsId fsRightsId;
                    uint8_t keyGeneration;
                    NN_RESULT_DO(m_pIntegrated->GetContentRightsId(&fsRightsId, &keyGeneration, keyList[i], infoList[j].type, GetProgramIndex(infoList[j]), storageId));

                    es::RightsIdIncludingKeyId rightsId = es::RightsIdIncludingKeyId::Construct(fsRightsId);
                    if (!rightsId.IsExternalKey())
                    {
                        continue;
                    }

                    es::RightsStatus status;
                    NN_RESULT_DO(es::CheckRightsStatus(&status, 1, &rightsId, 1));
                    if (status.status == es::RightsStatus::NotAvailable)
                    {
                        NN_RESULT_DO(m_pGameCardManager->ImportTicket(fsRightsId));
                    }
                }

                if (outCount != ListCount)
                {
                    break;
                }
            }
#endif
        }

        NN_RESULT_DO(emplacedRef.GetImpl().Add(&storageKey, 1));
    }

    NN_RESULT_DO(emplacedRef.GetImpl().Run());

    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetSendApplicationProgress(SendApplicationProgress* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    SendApplicationTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultSendApplicationTaskNotFound());

    NN_RESULT_DO(task.GetProgress(outValue));

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::ListNotCommittedContentMeta(int* outCount, ncm::ContentMetaKey outList[], int listCount, ncm::ApplicationId id, int offset) NN_NOEXCEPT
{
    auto stopper = m_pRequestServer->Stop();
    NN_RESULT_THROW_UNLESS(!HasApplicationApplyDeltaTask(id), ResultApplyDeltaTaskExists());

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        *outCount = 0;
        NN_RESULT_SUCCESS;
    }

    // NotCommittedContentMeta という名前にふさわしくはないが、
    // タスク作成から Prepare の処理が行われる間に破棄されるとタスクが復旧出来ないので、InstallMeta に書かれた情報を使う。
    // nim::NetworkInstallTask で追加登録されるコンテンツは復旧できないが、仕様とする。
    int count;
    NN_RESULT_DO(task.ListKeyFromInstallMeta(&count, outList, listCount, offset));
    *outCount = count;
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::CreateDownloadTask(const ncm::ContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(listCount > 0, ResultEmptyContentMetaKeyList());

    auto stopper = m_pRequestServer->Stop();

    NN_RESULT_THROW_UNLESS(!HasApplicationInstallTask(id), ResultDownloadTaskExists());

    uint32_t patchVersion = 0;
    int offset = 0;
    while (NN_STATIC_CONDITION(true))
    {
        const int ListCount = 16;
        ncm::StorageContentMetaKey storageKeyList[ListCount];
        int outCount;
        NN_RESULT_DO(m_pRecordDb->ListInstalledContentMeta(&outCount, storageKeyList, ListCount, id, offset));

        bool found = false;
        for (int i = 0; i < outCount; i++)
        {
            if (storageKeyList[i].key.type == ncm::ContentMetaType::Patch)
            {
                bool has;
                // GameCard には無いことが確定しているので、
                // 単純に ContentMetaKey が IntegratedManager に存在しているかどうかだけを調べればよい
                NN_RESULT_DO(m_pIntegrated->HasContentMetaKey(&has, storageKeyList[i].key));
                if (has)
                {
                    patchVersion = storageKeyList[i].key.version;
                }

                // Record 内にはパッチが重複していないので、ループを抜けてもよい
                found = true;
                break;
            }
        }
        offset += outCount;

        if (found || outCount != ListCount)
        {
            break;
        }
    }

    nim::NetworkInstallTaskId taskId;
    bool created = false;
    for (int i = 0; i < listCount; i++)
    {
        const auto& key = keyList[i];
        NN_RESULT_THROW_UNLESS(IsContentMetaTypeForApplicationDownloadTask(key.type), ResultInvalidContentMetaKey());
        if (key.type == ncm::ContentMetaType::Patch && key.version <= patchVersion)
        {
            continue;
        }
        ncm::StorageId installedStorage;
        bool found = true;
        NN_RESULT_TRY(m_pIntegrated->GetInstalledContentMetaStorage(&installedStorage, key));
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                found = false;
            }
        NN_RESULT_END_TRY;
        NN_UNUSED(installedStorage);

        if (found)
        {
            continue;
        }

        if (!created)
        {
            NN_RESULT_DO(nim::CreateNetworkInstallTask(&taskId, id, &keyList[i], 1, ncm::StorageId::Any));
            created = true;
        }
        else
        {
            NN_RESULT_DO(nim::AddNetworkInstallTaskContentMeta(&keyList[i], 1, taskId));
        }
    }

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetSystemUpdateMeta(ncm::ContentMetaKey* outValue, const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyMacOfSystemDeliveryInfo(info));
    NN_RESULT_DO(VerifySystemDeliveryProtocolCompatibility(info));
    auto systemUpdateMeta = GetSystemUpdateMetaForContentDelivery();
    NN_RESULT_THROW_UNLESS(systemUpdateMeta, ResultSystemUpdateMetaKeyNotFound());
    NN_RESULT_THROW_UNLESS(info.systemUpdateId == systemUpdateMeta->id, ResultInvalidSystemDeliveryInfo());
    *outValue = ncm::ContentMetaKey::Make(info.systemUpdateId, info.systemUpdateVersion, ncm::ContentMetaType::SystemUpdate);
    NN_RESULT_SUCCESS;
}

uint32_t ContentDeliveryManager::GetSystemDeliveryProtocolVersion() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
        return 0;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<uint32_t> s_Version;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);

        if (!s_Version)
        {
            uint32_t version;
            if (GetSettingsUint32Value(&version, "contents_delivery", "system_delivery_protocol_version"))
            {
                s_Version = version;
            }
            else
            {
                s_Version = 0;
            }
        }

        return *s_Version;
#endif
}

uint32_t ContentDeliveryManager::GetApplicationDeliveryProtocolVersion() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
        return 0;
#else
        static NonRecursiveMutex s_Mutex;
        static util::optional<uint32_t> s_Version;

        std::lock_guard<NonRecursiveMutex> guard(s_Mutex);

        if (!s_Version)
        {
            uint32_t version;
            if (GetSettingsUint32Value(&version, "contents_delivery", "application_delivery_protocol_version"))
            {
                s_Version = version;
            }
            else
            {
                s_Version = 0;
            }
        }

        return *s_Version;
#endif
}

Result ContentDeliveryManager::VerifySystemDeliveryInfo(const SystemDeliveryInfo& info, bool skipApplicationCompatilityCheck) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifySystemDeliveryProtocolCompatibility(info));
    NN_RESULT_DO(VerifyMacOfSystemDeliveryInfo(info));
    auto systemUpdateKey = GetSystemUpdateMetaForContentDelivery();
    NN_RESULT_THROW_UNLESS(systemUpdateKey, ResultSystemUpdateMetaKeyNotFound());
    NN_RESULT_THROW_UNLESS(info.systemUpdateId == systemUpdateKey->id, ResultInvalidSystemDeliveryInfo());

    if (!skipApplicationCompatilityCheck)
    {
        NN_RESULT_DO(VerifyApplicationDeliveryProtocolCompatibility(info));
    }

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyApplicationDeliveryInfo(const ApplicationDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyApplicationDeliveryProtocolCompatibility(info));
    NN_RESULT_DO(VerifyMacOfApplicationDeliveryInfo(info));

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifySystemDeliveryProtocolCompatibility(const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(info.systemDeliveryProtocolVersion <= GetSystemDeliveryProtocolVersion(), ResultUnknownSystemDeliveryProtocolVersion());
    NN_RESULT_THROW_UNLESS(info.systemDeliveryProtocolVersion == GetSystemDeliveryProtocolVersion(), ResultOldSystemDeliveryProtocolVersion());

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyApplicationDeliveryProtocolCompatibility(const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyApplicationDeliveryProtocolCompatibilityImpl(info.applicationDeliveryProtocolVersion));
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyApplicationDeliveryProtocolCompatibility(const ApplicationDeliveryInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyApplicationDeliveryProtocolCompatibilityImpl(info.applicationDeliveryProtocolVersion));
    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyApplicationDeliveryProtocolCompatibilityImpl(uint32_t protocolVersion) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(protocolVersion <= GetApplicationDeliveryProtocolVersion(), ResultUnknownApplicationDeliveryProtocolVersion());
    NN_RESULT_THROW_UNLESS(protocolVersion >= GetAcceptableApplicationProtocolVersion(), ResultOldApplicationDeliveryProtocolVersion());

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyMacOfSystemDeliveryInfo(const SystemDeliveryInfo &info) NN_NOEXCEPT
{
    Bit8 mac[sizeof(info.mac)];

    crypto::GenerateHmacSha256Mac(
        &mac, sizeof(mac),
        &info, sizeof(info) - sizeof(info.mac),
        SystemDeliveryInfoKeyForHmac, sizeof(SystemDeliveryInfoKeyForHmac));
    NN_RESULT_THROW_UNLESS(std::memcmp(mac, info.mac, sizeof(mac)) == 0, ResultInvalidSystemDeliveryInfo());

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::VerifyMacOfApplicationDeliveryInfo(const ApplicationDeliveryInfo &info) NN_NOEXCEPT
{
    Bit8 mac[sizeof(info.mac)];

    crypto::GenerateHmacSha256Mac(
        &mac, sizeof(mac),
        &info, sizeof(info) - sizeof(info.mac),
        ApplicationDeliveryInfoKeyForHmac, sizeof(ApplicationDeliveryInfoKeyForHmac));
    NN_RESULT_THROW_UNLESS(std::memcmp(mac, info.mac, sizeof(mac)) == 0, ResultInvalidApplicationDeliveryInfo());

    NN_RESULT_SUCCESS;
}

Result ContentDeliveryManager::GetApplicationDeliveryInfoHash(ApplicationDeliveryInfoHash* outValue, const ApplicationDeliveryInfo infoList[], int numList) NN_NOEXCEPT
{
    crypto::Sha256Generator generator;
    generator.Initialize();

    std::unique_ptr<ApplicationDeliveryInfoToCalculateHash[]> dataList(new ApplicationDeliveryInfoToCalculateHash[numList]);
    NN_RESULT_THROW_UNLESS(dataList, ResultBufferNotEnough());

    for (int i = 0; i < numList; i++)
    {
        const auto& info = infoList[i];
        NN_RESULT_DO(VerifyApplicationDeliveryInfo(info));
        ApplicationDeliveryInfoToCalculateHash data = { info.id, info.version, info.requiredVersion, info.requiredSystemVersion };
        dataList[i] = data;
    }

    const auto begin = dataList.get();
    const auto end = dataList.get() + numList;
    std::sort(begin, end);

    std::for_each(begin, end, [&generator](const ApplicationDeliveryInfoToCalculateHash& data) NN_NOEXCEPT
    {
        generator.Update(&data, sizeof(data));
    });
    generator.GetHash(&outValue->data, sizeof(outValue->data));

    NN_RESULT_SUCCESS;
}

}}}
