﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_Api.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_CommitManager.h>

#include <nn/ovln/format/ovln_DownloadMessage.h>

#include "ns_ApplicationInstallTask.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_ReceiveApplicationTask.h"
#include "ns_Config.h"
#include "ns_LogUtil.h"
#include "ns_ShellUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{

bool IncludesMainApplication(const ncm::StorageContentMetaKey list[], int count, ncm::ApplicationId id) NN_NOEXCEPT
{
    return std::any_of(list, list + count, [&id](const ncm::StorageContentMetaKey& key) NN_NOEXCEPT -> bool
                       {
                           return key.key.type == ncm::ContentMetaType::Application && key.key.id == id.value;
                       });
}

void ConvertFullInstalledContentMeta(ncm::StorageContentMetaKey* list, int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; ++i)
    {
        list[i].key.installType = ncm::ContentInstallType::Full;
    }
}

ovln::format::DownloadedDataType GetDownloadedDataType(const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    switch (key.type)
    {
    case ncm::ContentMetaType::Application:
        return ovln::format::DownloadedDataType::Application;
    case ncm::ContentMetaType::Patch:
        return ovln::format::DownloadedDataType::Patch;
    case ncm::ContentMetaType::AddOnContent:
        return ovln::format::DownloadedDataType::AddOnContent;
    default:
        return ovln::format::DownloadedDataType::Unknown;
    }
}

bool IsApplicationPrepurchased(ncm::ApplicationId id) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return es::CountTicket(id.value) <= 0 && es::CountPrepurchaseRecord(id.value) > 0;
#else
    NN_UNUSED(id);
    return false;
#endif
}

} // namespace

NonRecursiveMutex CommitManager::s_KeyMutex;
ncm::StorageContentMetaKey CommitManager::s_KeyList[MaxContentMetaCountPerApplication];

void CommitManager::Initialize(ApplicationRecordDatabase* recordDb, ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, SystemReportManager* systemReportManager, ovln::SenderForOverlayType* sender) NN_NOEXCEPT
{
    m_RecordDatabase = recordDb;
    m_EntityManager = entityManager;
    m_IntegratedManager = integrated;
    m_SystemReportManager = systemReportManager;
    m_Sender = sender;

    bool forceAutoCommit = false;
    bool forceDisableAutoCommit = false;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    {
        auto readSize = nn::settings::fwdbg::GetSettingsItemValue(&forceAutoCommit, sizeof(forceAutoCommit), "contents_delivery", "enable_auto_commit_forcibly");
        if (readSize != sizeof(forceAutoCommit))
        {
            forceAutoCommit = false;
        }
    }
    {
        auto readSize = nn::settings::fwdbg::GetSettingsItemValue(&forceDisableAutoCommit, sizeof(forceDisableAutoCommit), "contents_delivery", "disable_auto_commit_forcibly");
        if (readSize != sizeof(forceDisableAutoCommit))
        {
            forceDisableAutoCommit = false;
        }
    }
#endif
    m_IsAutoCommitEnabledForcibly = forceAutoCommit;
    m_IsAutoCommitDisabledForcibly = forceDisableAutoCommit;
}

Result CommitManager::IsDownloadTaskCommittable(bool* outValue, ncm::ApplicationId id, bool checkRequiredSystemVersion) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        *outValue = false;
    }
    else
    {
        bool isCommittable;
        NN_RESULT_DO(task.IsCommittable(&isCommittable, checkRequiredSystemVersion));
        *outValue = (!IsRunningApplication(id) && isCommittable);
    }
    NN_RESULT_SUCCESS;
}

Result CommitManager::IsApplyDeltaTaskCommittable(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    ApplicationApplyDeltaTask task(id);
    if (!task.IsValid())
    {
        *outValue = false;
    }
    else
    {
        bool isCommittable;
        NN_RESULT_DO(task.IsCommittable(&isCommittable));
        *outValue = isCommittable;
    }
    NN_RESULT_SUCCESS;
}

Result CommitManager::CommitDownloadTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    bool isCommittable;
    NN_RESULT_DO(IsDownloadTaskCommittable(&isCommittable, id, true));

    // ロックしているわけではないため、上位でコミット可能と判断されていたとしても
    // タイミング次第ではここではコミット済みということもある
    // そのため、isCommittable でなくてもエラーとはしない
    if (isCommittable)
    {
        NN_RESULT_DO(CommitDownloadTaskImpl(id));
    }
    NN_RESULT_SUCCESS;
}
Result CommitManager::CommitApplyDeltaTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    bool isCommittable;
    NN_RESULT_DO(IsApplyDeltaTaskCommittable(&isCommittable, id));

    // ロックしているわけではないため、上位でコミット可能と判断されていたとしても
    // タイミング次第ではここではコミット済みということもある
    // そのため、isCommittable でなくてもエラーとはしない
    if (isCommittable)
    {
        NN_RESULT_DO(CommitApplyDeltaTaskImpl(id));
    }
    NN_RESULT_SUCCESS;
}

Result CommitManager::CommitReceiveTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    // ローカルコンテンツ配信の場合、自動的にコミットされるということは無いため、
    // この API を呼んだ時にコミットできない場合はプログラミングエラーとしてエラーを返す
    {
        ReceiveApplicationTask task(id);
        NN_RESULT_THROW_UNLESS(task.IsValid(), ResultReceiveApplicationTaskNotFound());

        bool isDownloaded;
        NN_RESULT_DO(task.IsDownloaded(&isDownloaded));
        NN_RESULT_THROW_UNLESS(isDownloaded, ResultApplicationContentNotDownloaded());

        NN_RESULT_THROW_UNLESS(!IsRunningApplication(id), ResultApplicationIsRunning());
    }

    NN_RESULT_DO(CommitReceiveTaskImpl(id));
    NN_RESULT_SUCCESS;
}

Result CommitManager::NotifyDownloadTaskCompleted(nim::NetworkInstallTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    nim::NetworkInstallTaskInfo info;
    NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, id));
    NN_NS_TRACE_RESULT_IF_FAILED(ReportThroughput(info.applicationId), "[CommitManager] Report throughput failed 0x%016llx\n", info.applicationId);

    NN_NS_TRACE_RESULT_IF_FAILED(SendDownloadCompleteOverlayNotification(info.applicationId), "[CommitManager] Send overlay notification failed 0x%016llx\n", info.applicationId);

    bool isCommittable;
    NN_RESULT_DO(IsDownloadTaskCommittable(&isCommittable, info.applicationId, true));
    if (IsAutoCommitEnabled() && isCommittable)
    {
        NN_DETAIL_NS_TRACE("[CommitManager] Do commit download task automatically\n");
        NN_RESULT_DO(CommitDownloadTaskImpl(info.applicationId));
    }
    else if (IsAutoCommitEnabled() && IsDynamicCommitRequested(info.applicationId))
    {
        bool isDynamicCommittable;
        NN_RESULT_DO(IsCurrentApplicationDownloadTaskDynamicallyCommittable(&isDynamicCommittable));
        if (isDynamicCommittable)
        {
            NN_DETAIL_NS_TRACE("[CommitManager] Do dynamic-commit download task automatically\n");
            NN_RESULT_DO(CommitCurrentApplicationDownloadTaskDynamically());
            m_DynamicCommitEvent.Signal();
        }
    }

    NN_RESULT_SUCCESS;
}
Result CommitManager::NotifyApplyDeltaTaskCompleted(nim::ApplyDeltaTaskId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    nim::ApplyDeltaTaskInfo info;
    NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, id));

    bool isCommittable;
    NN_RESULT_DO(IsApplyDeltaTaskCommittable(&isCommittable, info.applicationId));
    if (IsAutoCommitEnabled() && isCommittable)
    {
        NN_DETAIL_NS_TRACE("[CommitManager] Do commit apply-delta task automatically\n");
        NN_RESULT_DO(CommitApplyDeltaTaskImpl(info.applicationId));
    }
    NN_RESULT_SUCCESS;
}

Result CommitManager::IsCurrentApplicationDownloadTaskDynamicallyCommittable(bool* outValue) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    arp::ApplicationLaunchProperty property;
    NN_RESULT_DO(GetCurrentApplicationLaunchProperty(&property));

    DynamicCommitStatus status;
    NN_RESULT_TRY(CheckDynamicCommitStatus(&status, property))
        NN_RESULT_CATCH_ALL
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    *outValue = (status == DynamicCommitStatus::FullCommittable || status == DynamicCommitStatus::PartialCommittable);

    NN_RESULT_SUCCESS;
}

Result CommitManager::CommitCurrentApplicationDownloadTaskDynamically() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    arp::ApplicationLaunchProperty property;
    NN_RESULT_DO(GetCurrentApplicationLaunchProperty(&property));

    // dynamic commit では現在のところ AoC のみコミットする。
    // そのため、IsDownloadTaskCommittable (strictly == true) のチェックは今のところしなくて良い
    // 将来 RequiredSystemVersion に関係するものを Dynamic Commit する場合、
    // その判定をここで行う必要がある
    DynamicCommitStatus status;
    NN_RESULT_DO(CheckDynamicCommitStatus(&status, property));
    switch(status)
    {
    case DynamicCommitStatus::FullCommittable:
        NN_DETAIL_NS_TRACE("[CommitManager] DynamicCommit: Do full commit\n");
        NN_RESULT_DO(CommitDownloadTaskImpl(property.id));
        break;
    case DynamicCommitStatus::PartialCommittable:
        NN_DETAIL_NS_TRACE("[CommitManager] DynamicCommit: Do partial commit\n");
        NN_RESULT_DO(CommitAddOnContentOnlyImpl(property.id));
        break;
    default:
        break;
    }

    NN_RESULT_SUCCESS;
}

void CommitManager::EnableAutoCommit() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_EnabledFlagMutex);
    m_IsAutoCommitEnabled = true;
}
void CommitManager::DisableAutoCommit() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_EnabledFlagMutex);
    m_IsAutoCommitEnabled = false;
}

Result CommitManager::TriggerDynamicCommitEvent() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CommitMutex);

    arp::ApplicationLaunchProperty property;
    NN_RESULT_DO(GetCurrentApplicationLaunchProperty(&property));
    NN_RESULT_THROW_UNLESS(IsDynamicCommitRequested(property.id), ResultRuntimeInstallNotAllowed());

    m_DynamicCommitEvent.Signal();
    NN_RESULT_SUCCESS;
}

std::unique_lock<os::Mutex> CommitManager::GetCommitStopper() NN_NOEXCEPT
{
    return std::unique_lock<os::Mutex>(m_CommitMutex);
}

Result CommitManager::CommitDownloadTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationInstallTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultDownloadTaskNotFound());

    NN_RESULT_TRY(task.Commit())
        NN_RESULT_CATCH_ALL
        {
            NN_DETAIL_NS_TRACE("[CommitManager] Failed to commit application install task %016llx, %08x\n", id.value, (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_SUCCESS; // Commit 失敗はタスクに永続化されるため、ここでは Success を返す
        }
    NN_RESULT_END_TRY;
    {
        std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList
        NN_UTIL_SCOPE_EXIT
        {
            NN_NS_TRACE_RESULT_IF_FAILED(task.Destroy(), "[CommitManager] failed to destroy\n");
        };

        int count;
        NN_RESULT_DO(task.ListKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));

        // パッチ間差分は記録しないので、フィルタリングする
        auto end = std::remove_if(s_KeyList, s_KeyList + count,
                                  [](const ncm::StorageContentMetaKey& key) NN_NOEXCEPT -> bool
                                  {
                                      return key.key.type == ncm::ContentMetaType::Patch &&
                                      key.key.installType == ncm::ContentInstallType::FragmentOnly;
                                  });
        count = static_cast<int>(end - s_KeyList);

        NN_RESULT_DO(m_RecordDatabase->UpdateKey(id, ApplicationEvent::DownloadTaskCommitted, s_KeyList, count));

        if (IncludesMainApplication(s_KeyList, count, id))
        {
            NN_NS_TRACE_RESULT_IF_FAILED(m_RecordDatabase->EnableAutoUpdate(id), "[CommitManager] Failed to enable auto update 0x%016llx\n", id);
            m_SystemReportManager->ReportApplicationInstall(ApplicationInstallReportType::Installed, id);
        }
    }
    NN_NS_TRACE_RESULT_IF_FAILED(m_EntityManager->CleanupUnrecordedApplicationEntity(id), "[CommitManager] Failed to cleanup unrecorded application entity\n");
    m_SystemReportManager->ReportSdCardFreeSpace();

    NN_RESULT_SUCCESS;
}
Result CommitManager::CommitApplyDeltaTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationApplyDeltaTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultApplyDeltaTaskNotFound());

    NN_RESULT_TRY(task.Commit())
        NN_RESULT_CATCH_ALL
        {
            // commit の失敗は task に永続化されるので、ここでは success を返す
            NN_DETAIL_NS_TRACE("[CommitManager] Failed to commit apply delta task 0x%016llx %08x\n", id.value, (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;
    {
        NN_UTIL_SCOPE_EXIT
        {
            NN_NS_TRACE_RESULT_IF_FAILED(task.Destroy(), "[CommitManager] failed to destroy 0x%016llx\n", id);
        };

        std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList
        int count;
        NN_RESULT_DO(task.ListKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));
        ConvertFullInstalledContentMeta(s_KeyList, count);
        NN_RESULT_DO(m_RecordDatabase->UpdateKey(id, ApplicationEvent::ApplyDeltaTaskCommitted, s_KeyList, count));
    }

    m_SystemReportManager->ReportSdCardFreeSpace();
    NN_RESULT_SUCCESS;
}
Result CommitManager::CommitAddOnContentOnlyImpl(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationInstallTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultDownloadTaskNotFound());

    // 追加コンテンツのみを列挙
    std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList
    int count;
    NN_RESULT_DO(task.ListNotCommittedKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));

    auto end = std::remove_if(s_KeyList, s_KeyList + count,
                              [](const ncm::StorageContentMetaKey& key) NN_NOEXCEPT -> bool
                              {
                                  return key.key.type != ncm::ContentMetaType::AddOnContent;
                              });
    count = static_cast<int>(end - s_KeyList);

    NN_RESULT_TRY(task.CommitPartially(s_KeyList, count))
        NN_RESULT_CATCH_ALL
        {
            NN_DETAIL_NS_TRACE("[CommitManager] Failed to partial-commit application install task %016llx, %08x\n", id.value, (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_SUCCESS; // Commit 失敗はタスクに永続化されるため、ここでは Success を返す
        }
    NN_RESULT_END_TRY;

    // 指定した keylist の一部がコミットできなかった場合はエラーが返る。
    // ここにきているということは、すべてコミットできた場合のみである。
    // そのため、keylist はすべてコミットされているため、UpdateKey しても良い。
    NN_RESULT_DO(m_RecordDatabase->UpdateKey(id, ApplicationEvent::DownloadTaskCommittedPartially, s_KeyList, count));
    m_SystemReportManager->ReportSdCardFreeSpace();

    NN_RESULT_SUCCESS;
}

Result CommitManager::CommitReceiveTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT
{
    ReceiveApplicationTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultReceiveApplicationTaskNotFound());

    NN_RESULT_DO(task.Commit());

    {
        std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList

        int count;
        NN_RESULT_DO(task.ListKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));

        NN_RESULT_DO(m_RecordDatabase->UpdateKey(id, ApplicationEvent::ReceiveTaskCommitted, s_KeyList, count));

        if (IncludesMainApplication(s_KeyList, count, id))
        {
            NN_NS_TRACE_RESULT_IF_FAILED(m_RecordDatabase->EnableAutoUpdate(id), "[CommitManager] Failed to enable auto update 0x%016llx\n", id);
            m_SystemReportManager->ReportApplicationInstall(ApplicationInstallReportType::Installed, id);
        }
        NN_NS_TRACE_RESULT_IF_FAILED(m_EntityManager->CleanupUnrecordedApplicationEntity(id), "[CommitManager] Failed to cleanup unrecorded application entity\n");
        m_SystemReportManager->ReportSdCardFreeSpace();
    }

    NN_RESULT_SUCCESS;
}

Result CommitManager::ReportThroughput(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        // 本当はタスクが見つからないエラーを返す
        NN_RESULT_SUCCESS;
    }

    int64_t throughput;
    NN_RESULT_DO(task.GetThroughput(&throughput));

    int count = 0;
    int offset = 0;
    ncm::InstallTaskOccupiedSize list[16];
    constexpr int ListCount = static_cast<int>(NN_ARRAY_SIZE(list));
    do
    {
        NN_RESULT_DO(task.ListOccupiedSize(&count, list, ListCount, offset));
        for (int i = 0; i < count; i++)
        {
            const auto& entry = list[i];
            m_SystemReportManager->ReportApplicationInstallPerformance(id, entry.key, entry.storageId, entry.size, throughput);
        }
        offset += count;
    } while (count == ListCount);

    NN_RESULT_SUCCESS;
}

Result CommitManager::SendDownloadCompleteOverlayNotification(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationInstallTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultDownloadTaskNotFound());

    ovln::format::DownloadCompleteData data = { id, 0 };

    std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList

    int count;
    NN_RESULT_DO(task.ListKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));

    for (int i = 0; i < count; ++i)
    {
        auto type = GetDownloadedDataType(s_KeyList[i].key);
        if (type != ovln::format::DownloadedDataType::Unknown)
        {
            data.downloadedDataTypes |= static_cast<Bit32>(type);
        }
    }

    if (IsApplicationPrepurchased(id))
    {
        data.downloadedApplicationAttributes |= static_cast<Bit8>(ovln::format::DownloadedApplicationAttributes::Prepurchase);
    }

    ovln::Message message;
    message.tag = ovln::format::DownloadCompleteDataTag;
    message.dataSize = sizeof(data);
    std::memcpy(&message.data, &data, sizeof(data));

    ovln::Send(m_Sender, message);

    NN_RESULT_SUCCESS;
}

Result CommitManager::CheckDynamicCommitStatus(DynamicCommitStatus* outValue, const arp::ApplicationLaunchProperty& property) NN_NOEXCEPT
{
    ApplicationInstallTask task(property.id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultDownloadTaskNotFound());

    bool isDownloaded;
    NN_RESULT_DO(task.IsDownloaded(&isDownloaded));
    NN_RESULT_THROW_UNLESS(isDownloaded, ResultNotCommittable());

    std::lock_guard<NonRecursiveMutex> guard(s_KeyMutex); // for s_KeyList
    int count;
    NN_RESULT_DO(task.ListNotCommittedKey(&count, s_KeyList, MaxContentMetaCountPerApplication, 0));

    bool requiredReboot = false;
    bool hasAddOnContent = false;
    bool hasContentExceptForAddOnContent = false;

    uint32_t version;
    NN_RESULT_DO(task.FindMaxRequiredApplicationVersion(&version));
    if (property.version < version)
    {
        requiredReboot = true;
    }
    for (int i = 0; i < count; ++i)
    {
        if (s_KeyList[i].key.type == ncm::ContentMetaType::AddOnContent)
        {
            hasAddOnContent = true;
            ncm::ContentMetaKey key;
            auto result = m_IntegratedManager->GetLatest(&key, s_KeyList[i].key.id);
            if (result.IsSuccess())
            {
                requiredReboot = true;
            }
            else if (ncm::ResultContentMetaNotFound::Includes(result))
            {
            }
            else
            {
                NN_RESULT_DO(result);
            }
        }
        else
        {
            hasContentExceptForAddOnContent = true;
        }
    }

    if (hasAddOnContent)
    {
        if (requiredReboot)
        {
            *outValue = DynamicCommitStatus::RebootApplicationRequired;
        }
        else if (hasContentExceptForAddOnContent)
        {
            *outValue = DynamicCommitStatus::PartialCommittable;
        }
        else
        {
            *outValue = DynamicCommitStatus::FullCommittable;
        }
    }
    else
    {
        *outValue = DynamicCommitStatus::NoTarget;
    }

    NN_RESULT_SUCCESS;
}

bool CommitManager::IsDynamicCommitRequested(ncm::ApplicationId id) NN_NOEXCEPT
{
    return m_RegisteredId == id && m_RegisteredRuntimeInstallPolicy == RuntimeAddOnContentInstall::AllowAppend;
}

bool CommitManager::IsAutoCommitEnabled() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_EnabledFlagMutex);
    return (m_IsAutoCommitEnabled || m_IsAutoCommitEnabledForcibly) && !m_IsAutoCommitDisabledForcibly;
}

}}}
