﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <utility>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_Api.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/es/es_Api.h>
#include <nn/es/es_ELicenseApi.h>
#include <nn/es/es_ELicenseApiForSystem.h>
#include <nn/es/debug/es_ELicenseArchiveJsonUtil.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_DynamicRightsSystemApi.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_RightsEnvironment.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_ScopedTransaction.h>
#include <nn/util/util_Span.h>
#include "ns_AccountUtil.h"
#include "ns_AsyncDynamicRightsImpl.h"
#include "ns_DebugUtil.h"
#include "ns_ForEachUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_RightsUtil.h"
#include "ns_ThreadAllocator.h"

namespace nn { namespace ns { namespace srv {
namespace {
    constexpr int ThreadCount = 1;
    constexpr size_t StackSize = 16 * 1024;
    constexpr int MaxRightsIdCount = 2002; // app + patch + aoc * 2000

    struct LicenseType
    {
        ApplicationRightsOnServerFlag applicationRightsOnServerFlag;
        nim::ELicenseType eLicenseType;
    };

    const LicenseType LicenseTypeList[] =
    {
        { ApplicationRightsOnServerFlag_DeviceLinkedPermanentLicense::Mask, nim::ELicenseType::DeviceLinkedPermanent },
        { ApplicationRightsOnServerFlag_PermanentLicense::Mask, nim::ELicenseType::Permanent },
        { ApplicationRightsOnServerFlag_AccountRestrictivePermanentLicense::Mask, nim::ELicenseType::AccountRestrictivePermanent },
        { ApplicationRightsOnServerFlag_TemporaryLicense::Mask, nim::ELicenseType::Temporary },
    };

    nim::ELicenseType GetHighPriorityELicenseType(nim::ELicenseType lhs, nim::ELicenseType rhs) NN_NOEXCEPT
    {
        auto lhsValue = static_cast<int>(lhs);
        auto rhsValue = static_cast<int>(rhs);
        return static_cast<nim::ELicenseType>(std::min(lhsValue, rhsValue));
    }

    bool IsAvailableELicense(const nim::AvailableELicense& eLicense) NN_NOEXCEPT
    {
        switch (eLicense.licenseStatus)
        {
        case nim::ELicenseStatus::Assignable:
            return true;
        default:
            return false;
        }
    }

    nim::AvailableELicense GetHighPriorityAvailableELicense(const nim::AvailableELicense& lhs, const nim::AvailableELicense& rhs) NN_NOEXCEPT
    {
        const auto& highPriorityELicense = GetHighPriorityELicenseType(lhs.licenseType, rhs.licenseType) == lhs.licenseType ? lhs : rhs;
        const auto& lowPriorityELicense = &highPriorityELicense == &lhs ? rhs : lhs;

        // SIGLO-85038
        // 6 NUP 時点では、 DeviceLinkedPermanent と Temporary の2つの権利しかなく、
        // 優先度は DeviceLinkedPermanent > Temporary
        // 共に権利取得失敗した場合は Temporary の失敗理由を優先させる
        return IsAvailableELicense(highPriorityELicense) ? highPriorityELicense : lowPriorityELicense;
    }

    // 基本一つずつしか作られないはずなので、オーバーフローは気にしない
    AvailableELicenseHolderHandle CreateAvailableELicenseHolderHandle() NN_NOEXCEPT
    {
        static std::atomic<Bit8> s_HandleValue { 0 };
        auto handleValue = ++s_HandleValue;
        while (handleValue == 0)
        {
            handleValue = ++s_HandleValue;
        }
        return { handleValue };
    }

    class AvailableELicenseHolder
    {
        NN_DISALLOW_COPY(AvailableELicenseHolder);
        NN_DISALLOW_MOVE(AvailableELicenseHolder);

    public:
        AvailableELicenseHolder() = default;
        class Writer
        {
        NN_DISALLOW_COPY(Writer);
        public:
            Writer() NN_NOEXCEPT : m_pHolder(), m_Handle(InvalidAvailableELicenseHolderHandle) {}
            Writer(AvailableELicenseHolder* pHolder, AvailableELicenseHolderHandle handle) NN_NOEXCEPT : m_pHolder(pHolder), m_Handle(handle) {}
            Writer(Writer&& rvalue) NN_NOEXCEPT : m_pHolder(rvalue.m_pHolder), m_Handle(rvalue.m_Handle)
            {
                rvalue.m_pHolder = nullptr;
                rvalue.m_Handle = InvalidAvailableELicenseHolderHandle;
            }
            Writer& operator=(Writer&& rvalue) NN_NOEXCEPT
            {
                Writer(std::move(rvalue)).swap(*this);
                return *this;
            }

            void swap(Writer& other) NN_NOEXCEPT
            {
                std::swap(m_pHolder, other.m_pHolder);
                std::swap(m_Handle, other.m_Handle);
            }

            void Add(const nim::AvailableELicense& license) NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_Handle == m_pHolder->m_Handle);
                NN_ABORT_UNLESS(m_pHolder->m_Count < MaxRightsIdCount);
                NN_ABORT_UNLESS(license.licenseType != nim::ELicenseType::Unknown);

                auto endIterator = m_pHolder->m_RightsIdList.begin() + m_pHolder->m_Count;
                auto oldELicense = std::find(m_pHolder->m_RightsIdList.begin(), endIterator, license.rightsId);
                if (oldELicense != endIterator)
                {
                    auto index = std::distance(m_pHolder->m_RightsIdList.begin(), oldELicense);
                    m_pHolder->m_ELicenseTypeList[index] = GetHighPriorityELicenseType(m_pHolder->m_ELicenseTypeList[index], license.licenseType);
                }
                else
                {
                    m_pHolder->m_RightsIdList[m_pHolder->m_Count] = license.rightsId;
                    m_pHolder->m_ELicenseTypeList[m_pHolder->m_Count] = license.licenseType;
                    m_pHolder->m_Count++;
                }
            }

            AvailableELicenseHolderHandle GetHandle() const NN_NOEXCEPT
            {
                return m_Handle;
            }

        private:
            AvailableELicenseHolder* m_pHolder;
            AvailableELicenseHolderHandle m_Handle;
        };

        Writer OpenWriter() NN_NOEXCEPT
        {
            m_Handle = CreateAvailableELicenseHolderHandle();
            m_Count = 0;
            return Writer(this, m_Handle);
        }

        class Reader
        {
        public:
            Reader() NN_NOEXCEPT : m_pHolder(), m_Handle(InvalidAvailableELicenseHolderHandle) {}
            Reader(AvailableELicenseHolder* pHolder, AvailableELicenseHolderHandle handle) NN_NOEXCEPT : m_pHolder(pHolder), m_Handle(handle) {}
            Reader(Reader&& rvalue) NN_NOEXCEPT : m_pHolder(rvalue.m_pHolder), m_Handle(rvalue.m_Handle)
            {
                rvalue.m_pHolder = nullptr;
                rvalue.m_Handle = InvalidAvailableELicenseHolderHandle;
            }
            Reader& operator=(Reader&& rvalue) NN_NOEXCEPT
            {
                Reader(std::move(rvalue)).swap(*this);
                return *this;
            }

            void swap(Reader& other) NN_NOEXCEPT
            {
                std::swap(m_pHolder, other.m_pHolder);
                std::swap(m_Handle, other.m_Handle);
            }

            int CountRightsIdByLicenseType(nim::ELicenseType type) const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);
                return static_cast<int>(std::count(m_pHolder->m_ELicenseTypeList.begin(), m_pHolder->m_ELicenseTypeList.begin() + m_pHolder->m_Count, type));
            }

            int GetByLicenseType(es::RightsId* outList, int listCount, nim::ELicenseType type) const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);

                int count = 0;
                for (int i = 0; i < m_pHolder->m_Count; i++)
                {
                    if (m_pHolder->m_ELicenseTypeList[i] == type)
                    {
                        outList[count++] = m_pHolder->m_RightsIdList[i];
                        if (count >= listCount)
                        {
                            break;
                        }
                    }
                }

                return count;
            }

            int GetCount() const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);
                return m_pHolder->m_Count;
            }

            bool Has(es::RightsId rightsId) const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);
                return std::find(m_pHolder->m_RightsIdList.begin(), m_pHolder->m_RightsIdList.begin() + m_pHolder->m_Count, rightsId) != m_pHolder->m_RightsIdList.end();
            }

            util::Span<const es::RightsId> GetRightsIdSpan() const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);
                return util::MakeSpan(m_pHolder->m_RightsIdList.data(), m_pHolder->m_Count);
            }

            nim::ELicenseType GetLicenseType(es::RightsId rightsId) const NN_NOEXCEPT
            {
                NN_ABORT_UNLESS(m_pHolder->m_Handle == m_Handle);
                auto endIterator = m_pHolder->m_RightsIdList.begin() + m_pHolder->m_Count;
                auto found = std::find(m_pHolder->m_RightsIdList.begin(), endIterator, rightsId);
                NN_ABORT_UNLESS(found != endIterator);
                auto index = std::distance(m_pHolder->m_RightsIdList.begin(), found);
                return m_pHolder->m_ELicenseTypeList[index];
            }

        private:
            const AvailableELicenseHolder* m_pHolder;
            AvailableELicenseHolderHandle m_Handle;
        };

        bool CanOpenReader(AvailableELicenseHolderHandle handle) const NN_NOEXCEPT
        {
            return m_Handle == handle;
        }

        Reader OpenReader(AvailableELicenseHolderHandle handle) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_Handle == handle);
            return Reader(this, m_Handle);
        }

    private:
        // TORIAEZU: 本来権利保有者の naId も保持しておくべきだが、今はライセンスタイプで判別出来るので、省略している
        std::array<es::RightsId, MaxRightsIdCount>  m_RightsIdList;
        std::array<nim::ELicenseType, MaxRightsIdCount> m_ELicenseTypeList;
        int m_Count {};
        AvailableELicenseHolderHandle m_Handle {};
    };

    AvailableELicenseHolder g_AvailableELicenseHolder;
    NonRecursiveMutex g_AvailableELicenseHolderMutex;

    os::ThreadType g_ThreadList[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK char g_Stack[StackSize * ThreadCount];

    ThreadAllocator g_ThreadAllocator(g_ThreadList, ThreadCount, os::InvalidThreadPriority, g_Stack, sizeof(g_Stack), StackSize);

    ApplicationRightsOnServerFlag GetAvailabilityFlag(nim::ELicenseStatus licenseStatus) NN_NOEXCEPT
    {
        ApplicationRightsOnServerFlag flags {};
        switch(licenseStatus)
        {
        case nim::ELicenseStatus::Assignable:
            flags |= ApplicationRightsOnServerFlag_HasAvailableRights::Mask;
            break;
        default:
            flags |= ApplicationRightsOnServerFlag_HasUnavailableRights::Mask;
            break;
        }
        return flags;
    }

    ApplicationRightsOnServerFlag GetLicenseTypeFlag(nim::ELicenseType licenseType) NN_NOEXCEPT
    {
        ApplicationRightsOnServerFlag flags {};
        switch(licenseType)
        {
        case nim::ELicenseType::DeviceLinkedPermanent:
            {
                flags |= ApplicationRightsOnServerFlag_DeviceLinkedPermanentLicense::Mask;
                flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            }
            break;
        case nim::ELicenseType::Permanent:
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            flags |= ApplicationRightsOnServerFlag_PermanentLicense::Mask;
            break;
        case nim::ELicenseType::AccountRestrictivePermanent:
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            flags |= ApplicationRightsOnServerFlag_IsAccountRestrictedRights::Mask;
            flags |= ApplicationRightsOnServerFlag_AccountRestrictivePermanentLicense::Mask;
            break;
        case nim::ELicenseType::Temporary:
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            flags |= ApplicationRightsOnServerFlag_IsAccountRestrictedRights::Mask;
            flags |= ApplicationRightsOnServerFlag_TemporaryLicense::Mask;
            break;
        case nim::ELicenseType::Unknown:
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        return flags;
    }

    ApplicationRightsUnavailableFlag GetApplicationRightsUnavailableFlag(nim::ELicenseStatus status) NN_NOEXCEPT
    {
        ApplicationRightsUnavailableFlag flags {};
        // TODO: 要本体更新エラーと未配信エラーに対応する
        switch(status)
        {
        case nim::ELicenseStatus::Assignable:
            break;
        case nim::ELicenseStatus::Unknown:
            // TORIAEZU: 権利なしと同じにする
            flags |= ApplicationRightsUnavailableFlag_NoRights::Mask;
            break;
        case nim::ELicenseStatus::NotAssignableSinceLimitExceeded:
            flags |= ApplicationRightsUnavailableFlag_AssignableRightsLimitExceeded::Mask;
            break;
        case nim::ELicenseStatus::NotAssignableSinceNoRights:
            flags |= ApplicationRightsUnavailableFlag_NoRights::Mask;
            break;
        case nim::ELicenseStatus::NotAssignableSinceNotDeviceLinked:
            flags |= ApplicationRightsUnavailableFlag_NoRights::Mask;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        return flags;
    }

    enum class DebugLicenseType : int
    {
        None,
        DeviceLinkedPermanent,
        Permanent,
        AccountRestrictivePermanent,
        Temporary,
    };

    util::optional<ApplicationRightsOnServerFlag> GetDebugApplicationRightsOnServerFlag() NN_NOEXCEPT
    {
        auto licenseType = GetLicenseTypeForDebug();
        if (!licenseType)
        {
            return util::nullopt;
        }

        ApplicationRightsOnServerFlag flags {};
        switch(static_cast<DebugLicenseType>(*licenseType))
        {
        case DebugLicenseType::DeviceLinkedPermanent:
            flags |= ApplicationRightsOnServerFlag_DeviceLinkedPermanentLicense::Mask;
            flags |= ApplicationRightsOnServerFlag_HasAvailableRights::Mask;
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            break;
        case DebugLicenseType::Permanent:
            flags |= ApplicationRightsOnServerFlag_PermanentLicense::Mask;
            flags |= ApplicationRightsOnServerFlag_HasAvailableRights::Mask;
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            break;
        case DebugLicenseType::AccountRestrictivePermanent:
            flags |= ApplicationRightsOnServerFlag_AccountRestrictivePermanentLicense::Mask;
            flags |= ApplicationRightsOnServerFlag_HasAvailableRights::Mask;
            flags |= ApplicationRightsOnServerFlag_IsAccountRestrictedRights::Mask;
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            break;
        case DebugLicenseType::Temporary:
            flags |= ApplicationRightsOnServerFlag_TemporaryLicense::Mask;
            flags |= ApplicationRightsOnServerFlag_HasAvailableRights::Mask;
            flags |= ApplicationRightsOnServerFlag_IsAccountRestrictedRights::Mask;
            flags |= ApplicationRightsOnServerFlag_RecommendAssignRights::Mask;
            break;
        default:
            return util::nullopt;
        }
        return flags;
    }

    template <class FuncT>
    Result ForEachExternalKeyContentCandidate(
        ApplicationRecordDatabase* pDb, const IntegratedContentManager* pIntegrated,
        ncm::ApplicationId appId, FuncT func) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(pDb->Open(&list, appId));

        for (auto& storageKey : list.GetSpan())
        {
            const auto& key = storageKey.key;

            bool hasKey;
            NN_RESULT_DO(pIntegrated->HasContentMetaKey(&hasKey, key));
            if(!hasKey)
            {
                continue;
            }

            // 外部鍵にならないものを除外
            // アプリケーション記録のストレージ情報は信頼ならないが、
            // ゲームカードの記録だけは都度設定しているので信頼できる
            const auto storageId = storageKey.storageId;
            if (storageId == ncm::StorageId::GameCard && key.type != ncm::ContentMetaType::Patch)
            {
                continue;
            }

            bool isEnd = false;
            NN_RESULT_DO(func(&isEnd, key));
            if(isEnd)
            {
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_SUCCESS;
    }

    template<class AsyncObject, class LockObject, class FuncT>
    Result CreateAsyncObject(util::optional<AsyncObject>* pAsyncObject, LockObject* pLockObject, FuncT func) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(*pLockObject);
        util::ScopedTransaction transaction;
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            *pAsyncObject = util::nullopt;
        };

        pAsyncObject->emplace();

        NN_RESULT_DO(func(&(pAsyncObject->value())));

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    Result RequestQueryAvailableELicensesImpl(
        nim::AsyncAvailableELicenses* pAsync,
        const es::RightsId* pList,
        int listCount,
        const account::Uid& uid) NN_NOEXCEPT
    {
        if (uid == account::InvalidUid)
        {
            NN_RESULT_DO(nim::RequestQueryAvailableELicenses(pAsync, pList, listCount));
        }
        else
        {
            NN_RESULT_DO(nim::RequestQueryAvailableELicenses(pAsync, uid, pList, listCount));
        }
        NN_RESULT_SUCCESS;
    }

    template <class CancelMutex>
    Result RequestAssignELicensesImpl(util::optional<nim::AsyncAssignedELicenses>* pOptionalAsync, CancelMutex* pCancel, AvailableELicenseHolderHandle handle, const account::Uid& uid, nim::ELicenseType type) NN_NOEXCEPT
    {
        auto reader = g_AvailableELicenseHolder.OpenReader(handle);
        auto countRightsId = reader.CountRightsIdByLicenseType(type);
        if (countRightsId == 0)
        {
            NN_RESULT_SUCCESS;
        }

        auto rightsIdList = std::make_unique<es::RightsId[]>(countRightsId);
        NN_ABORT_UNLESS(rightsIdList);
        auto count = reader.GetByLicenseType(rightsIdList.get(), countRightsId, type);

        NN_RESULT_DO(CreateAsyncObject(pOptionalAsync, pCancel, [&](nim::AsyncAssignedELicenses* pAsync) NN_NOEXCEPT -> Result
        {
            if (uid == account::InvalidUid || type == nim::ELicenseType::DeviceLinkedPermanent)
            {
                NN_RESULT_DO(nim::RequestAssignELicenses(pAsync, rightsIdList.get(), count, type));
            }
            else
            {
                NN_RESULT_DO(nim::RequestAssignELicenses(pAsync, uid, rightsIdList.get(), count, type));
            }

            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

    template <class AddFunc>
    Result SetUpAvailableELicenseHolderImpl(nim::AsyncAvailableELicenses* pAsync, AddFunc addFunc) NN_NOEXCEPT
    {
        auto listFunc = [&pAsync](int* outCount, nim::AvailableELicense* outList, int listCount, int offset) NN_NOEXCEPT -> Result
        {
            NN_RESULT_TRY(pAsync->Get(outCount, outList, listCount, offset))
                NN_RESULT_CATCH_CONVERT(nim::ResultSystemUpdateRequired, ResultSystemUpdateRequired())
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        };

        util::optional<nim::AvailableELicense> currentELicense;
        NN_RESULT_DO(ForEachFunc<nim::AvailableELicense>(listFunc, [&](bool* outIsEnd, const nim::AvailableELicense& eLicense) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;

            if (eLicense.licenseType == nim::ELicenseType::Unknown)
            {
                // スキップ
                NN_RESULT_SUCCESS;
            }

            if (currentELicense)
            {
                if (currentELicense->rightsId == eLicense.rightsId)
                {
                    currentELicense = GetHighPriorityAvailableELicense(*currentELicense, eLicense);
                }
                else
                {
                    addFunc(*currentELicense);
                    currentELicense = eLicense;
                }
            }
            else
            {
                currentELicense = eLicense;
            }

            NN_RESULT_SUCCESS;
        }));

        if (currentELicense)
        {
            addFunc(*currentELicense);
        }

        NN_RESULT_SUCCESS;
    }

    Result NeedsSyncTicket(bool* outValue, const es::ELicenseId* pELicenseIdList, int listCount) NN_NOEXCEPT
    {
    #if defined(NN_BUILD_CONFIG_OS_HORIZON)
        int offset = 0;
        bool needsSyncTicket = false;
        while (NN_STATIC_CONDITION(true))
        {
            constexpr int InfoListCount = 8; // es::ELicenseInfoWrapper が 72 byte もあるので、 8 個に抑えておく
            es::ELicenseInfoWrapper infoList[InfoListCount];
            auto queryCount = std::min(listCount - offset, InfoListCount);
            auto count = es::ListELicenseInfo(infoList, InfoListCount, &pELicenseIdList[offset], queryCount);
            needsSyncTicket = std::any_of(infoList, infoList + count, [](const es::ELicenseInfoWrapper& info) NN_NOEXCEPT { return !info.HasTicket(); });
            if (needsSyncTicket || count < InfoListCount)
            {
                break;
            }
            offset += count;
        }

        *outValue = needsSyncTicket;
    #else
        NN_UNUSED(pELicenseIdList);
        NN_UNUSED(listCount);
        *outValue = false;
    #endif

        NN_RESULT_SUCCESS;
    }

    template <class CancelMutex>
    Result SyncELicensesImpl(util::optional<nim::AsyncResult>* pOptionalAsync, CancelMutex* pCancel, const account::NintendoAccountId& naId) NN_NOEXCEPT
    {
        NN_RESULT_DO(CreateAsyncObject(pOptionalAsync, pCancel, [&naId](nim::AsyncResult* pAsync) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(nim::RequestSyncELicenses(pAsync, naId));
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_DO((*pOptionalAsync)->Get());

        // タスクを破棄しないと次のタスクが実行できない
        {
            NN_UTIL_LOCK_GUARD(*pCancel);
            *pOptionalAsync = util::nullopt;
        }

        NN_RESULT_SUCCESS;
    }

    template <class CancelMutex>
    Result SyncTicketImpl(util::optional<ec::system::AsyncProgressResult>* pOptionalAsync, CancelMutex* pCancel, const AssignedELicenseInfo* pInfoList, int listCount) NN_NOEXCEPT
    {
        bool needsSyncTicket = false;
        for (auto& info : util::MakeSpan(pInfoList, listCount))
        {
            const auto& list = info.deviceLinkedELicense;
            NN_RESULT_DO(NeedsSyncTicket(&needsSyncTicket, list.eLicenseIds.get(), list.eLicenseCount));
            if (needsSyncTicket)
            {
                break;
            }
        }
        if (needsSyncTicket)
        {
            NN_RESULT_DO(CreateAsyncObject(pOptionalAsync, pCancel, [](ec::system::AsyncProgressResult* pAsync) NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(ec::system::RequestSyncTicketForSystem(pAsync));
                NN_RESULT_SUCCESS;
            }));

            NN_RESULT_DO((*pOptionalAsync)->Get());

            // タスクを破棄しないと次のタスクが実行できない
            {
                NN_UTIL_LOCK_GUARD(*pCancel);
                *pOptionalAsync = util::nullopt;
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result ExcludeELicenseIdsWithTicket(util::Span<es::ELicenseId>* outValue, es::ELicenseId* pList, int listCount)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // メモリ効率のため、以降では pList を直接操作している
        int offset = 0;
        auto uniqueList = util::MakeSpan(pList, std::unique(pList, pList + listCount));
        while (NN_STATIC_CONDITION(true))
        {
            constexpr int InfoListCount = 32;
            es::ELicenseInfoWrapper infoList[InfoListCount];
            auto queryCount = std::min(static_cast<int>(uniqueList.size()) - offset, InfoListCount);
            if (queryCount < 1)
            {
                break;
            }
            auto count = es::ListELicenseInfo(infoList, InfoListCount, &uniqueList[offset], queryCount);
            // ELicense 同期後なので、情報が得られない ELicense があった場合はエラー
            NN_RESULT_THROW_UNLESS(count == queryCount, ResultDynamicRightsDownloadFailed());
            for (auto& info : util::MakeSpan(infoList, count))
            {
                if (info.HasTicket())
                {
                    auto iterator = std::find(uniqueList.begin(), uniqueList.end(), info.info.eLicenseId);
                    if (iterator != uniqueList.end())
                    {
                        *iterator = es::InvalidELicenseId;
                    }
                }
            }
            if (count < InfoListCount)
            {
                break;
            }
            offset += count;
        }

        *outValue = util::MakeSpan(uniqueList.begin(), std::remove(uniqueList.begin(), uniqueList.end(), es::InvalidELicenseId));
#else
        NN_UNUSED(pList);
        NN_UNUSED(listCount);
        *outValue = util::MakeSpan(pList, 0);
#endif
        NN_RESULT_SUCCESS;
    }

    template <class CancelMutex>
    Result DownloadDynamicETicketIfNecessaryImpl(
        util::optional<nim::AsyncResult>* pOptionalAsync, CancelMutex* pCancel,
        const AssignedELicenseInfo* pInfoList, int listCount,
        const account::Uid& uid)
    {
        if (listCount == 0 || uid == account::InvalidUid)
        {
            NN_RESULT_SUCCESS;
        }

        // 6.0 NUP では利用者 == 所有者の権利しか落とせないとしておく

        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        auto found = std::find_if(pInfoList, pInfoList + listCount, [&naId](const AssignedELicenseInfo& info) NN_NOEXCEPT
        {
            return info.ownerId == naId && info.dynamicRightsELicense.eLicenseCount > 0;
        });

        if (found == pInfoList + listCount)
        {
            NN_RESULT_SUCCESS;
        }

        auto& list = found->dynamicRightsELicense;

        util::Span<es::ELicenseId> filteredList;
        NN_RESULT_DO(ExcludeELicenseIdsWithTicket(&filteredList, list.eLicenseIds.get(), list.eLicenseCount));

        if (filteredList.size() == 0)
        {
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(CreateAsyncObject(pOptionalAsync, pCancel, [&](nim::AsyncResult* pAsync) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(nim::RequestDownloadETickets(pAsync, uid, filteredList.begin(), static_cast<int>(filteredList.size())));
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_DO((*pOptionalAsync)->Get());

        // タスクを破棄しないと次のタスクが実行できない
        {
            NN_UTIL_LOCK_GUARD(*pCancel);
            *pOptionalAsync = util::nullopt;
        }

        NN_RESULT_SUCCESS;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    util::optional<es::debug::ELicenseInfoForDebug> MakeELicenseInfoForDebug(
        const ApplicationRightsOnServerFlag& flags,
        const es::RightsIdIncludingKeyId rightsIdIncludingKeyId,
        const account::NintendoAccountId naId) NN_NOEXCEPT
    {
        util::optional<es::debug::ELicenseInfoForDebug> elicense;

        // TORIAEZU: 機器認証のチケットがある場合を許容するので問答無用でインポートする
        if (flags.Test<ApplicationRightsOnServerFlag_DeviceLinkedPermanentLicense>())
        {
            es::debug::ELicenseInfoForDebug tmp;
            if (es::debug::MakeDeviceLinkedPermanentELicenseFromTicketDBInfo(&tmp, rightsIdIncludingKeyId.GetRightsId(), naId))
            {
                return tmp;
            }
        }
        else if (flags.Test<ApplicationRightsOnServerFlag_TemporaryLicense>())
        {
            es::debug::ELicenseInfoForDebug tmp;
            if (es::debug::MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(&tmp, rightsIdIncludingKeyId.GetRightsId(), naId))
            {
                return tmp;
            }
        }

        return util::nullopt;
    }
#endif

    Result InitializeAssignedELicenseInfo(
        int* outCount,
        AssignedELicenseInfo* pOutInfoList,
        int listCount,
        nim::AsyncAssignedELicenses* pAsync,
        AvailableELicenseHolderHandle handle) NN_NOEXCEPT
    {
        int assignedCount;
        NN_RESULT_DO(pAsync->GetSize(&assignedCount));

        auto listFunc = [&](int* outValue, nim::AssignedELicense* outList, int count, int offset) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(pAsync->Get(outValue, outList, count, offset));
            NN_RESULT_SUCCESS;
        };

        auto reader = g_AvailableELicenseHolder.OpenReader(handle);
        int ownerCount = 0;
        NN_RESULT_DO(ForEachFunc<nim::AssignedELicense>(listFunc, [&](bool* outIsEnd, const nim::AssignedELicense& assignedELicense) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;

            if ((assignedELicense.licenseType == nim::ELicenseType::Unknown) ||
                !reader.Has(assignedELicense.rightsId))
            {
                // スキップ
                NN_RESULT_SUCCESS;
            }
            auto found = std::find_if(pOutInfoList, pOutInfoList + ownerCount, [&assignedELicense](const AssignedELicenseInfo& info) NN_NOEXCEPT
            {
                return info.ownerId == assignedELicense.naId;
            });
            if (found != pOutInfoList + ownerCount)
            {
                if (assignedELicense.licenseType == nim::ELicenseType::DeviceLinkedPermanent)
                {
                    found->deviceLinkedELicense.eLicenseCount++;
                }
                else
                {
                    found->dynamicRightsELicense.eLicenseCount++;
                }
            }
            else
            {
                NN_ABORT_UNLESS(ownerCount < listCount);
                int eLicenseCount = assignedELicense.licenseType != nim::ELicenseType::DeviceLinkedPermanent ? 1 : 0;
                int eLicenseOfDeviceLinkedPermanentCount = eLicenseCount == 0 ? 1 : 0;
                pOutInfoList[ownerCount++] = { assignedELicense.naId, { nullptr, eLicenseCount }, { nullptr, eLicenseOfDeviceLinkedPermanentCount } };
            }
            NN_RESULT_SUCCESS;
        }));

        for (auto& info : util::MakeSpan(pOutInfoList, ownerCount))
        {
            if (info.dynamicRightsELicense.eLicenseCount > 0)
            {
                info.dynamicRightsELicense.eLicenseIds = std::make_unique<es::ELicenseId[]>(info.dynamicRightsELicense.eLicenseCount);
                NN_ABORT_UNLESS(info.dynamicRightsELicense.eLicenseIds);
            }
            if (info.deviceLinkedELicense.eLicenseCount > 0)
            {
                info.deviceLinkedELicense.eLicenseIds = std::make_unique<es::ELicenseId[]>(info.deviceLinkedELicense.eLicenseCount);
                NN_ABORT_UNLESS(info.deviceLinkedELicense.eLicenseIds);
            }
        }

        *outCount = ownerCount;
        NN_RESULT_SUCCESS;
    }

    Result CopyELicenseIds(
        AssignedELicenseInfo* pOutInfoList,
        int ownerCount,
        nim::AsyncAssignedELicenses* pAsync,
        AvailableELicenseHolderHandle handle) NN_NOEXCEPT
    {
        auto listFunc = [&](int* outValue, nim::AssignedELicense* outList, int count, int offset) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(pAsync->Get(outValue, outList, count, offset));
            NN_RESULT_SUCCESS;
        };

        auto reader = g_AvailableELicenseHolder.OpenReader(handle);
        int copiedDeviceLinkedELicenseCount = 0;
        int copiedDynamicRightsELicenseCount = 0;
        NN_RESULT_DO(ForEachFunc<nim::AssignedELicense>(listFunc, [&](bool* outIsEnd, const nim::AssignedELicense& assignedELicense) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;

            if ((assignedELicense.licenseType == nim::ELicenseType::Unknown) ||
                !reader.Has(assignedELicense.rightsId))
            {
                // スキップ
                NN_RESULT_SUCCESS;
            }
            auto found = std::find_if(pOutInfoList, pOutInfoList + ownerCount, [&assignedELicense](const AssignedELicenseInfo& info) NN_NOEXCEPT
            {
                return info.ownerId == assignedELicense.naId;
            });
            // 必ず見つかるはず
            NN_ABORT_UNLESS(found != pOutInfoList + ownerCount);
            if (assignedELicense.licenseType == nim::ELicenseType::DeviceLinkedPermanent)
            {
                NN_ABORT_UNLESS(copiedDeviceLinkedELicenseCount < found->deviceLinkedELicense.eLicenseCount);
                found->deviceLinkedELicense.eLicenseIds[copiedDeviceLinkedELicenseCount++] = assignedELicense.eLicenseId;
            }
            else
            {
                NN_ABORT_UNLESS(copiedDynamicRightsELicenseCount < found->dynamicRightsELicense.eLicenseCount);
                found->dynamicRightsELicense.eLicenseIds[copiedDynamicRightsELicenseCount++] = assignedELicense.eLicenseId;
            }
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

    Result GetAssignedELicenseInfoList(
        int* outCount,
        AssignedELicenseInfo* pOutInfoList,
        int listCount,
        nim::AsyncAssignedELicenses* pAsync,
        AvailableELicenseHolderHandle handle) NN_NOEXCEPT
    {
        NN_RESULT_DO(InitializeAssignedELicenseInfo(
            outCount, pOutInfoList, listCount, pAsync, handle));
        NN_RESULT_DO(CopyELicenseIds(pOutInfoList, *outCount, pAsync, handle));

        NN_RESULT_SUCCESS;
    }

    void MergeELicenseList(AssignedELicenseInfo::ELicenseList* pMain, AssignedELicenseInfo::ELicenseList* pTarget) NN_NOEXCEPT
    {
        if (pTarget->eLicenseCount > 0)
        {
            if (pMain->eLicenseCount > 0)
            {
                auto count = pMain->eLicenseCount + pTarget->eLicenseCount;
                auto buffer = std::make_unique<es::ELicenseId[]>(count);
                NN_ABORT_UNLESS(buffer);
                std::copy(pMain->eLicenseIds.get(), pMain->eLicenseIds.get() + pMain->eLicenseCount, buffer.get());
                std::copy(pTarget->eLicenseIds.get(), pTarget->eLicenseIds.get() + pTarget->eLicenseCount, buffer.get() + pMain->eLicenseCount);
                pMain->eLicenseIds = std::move(buffer);
                pMain->eLicenseCount = count;
                pTarget->eLicenseIds = nullptr;
            }
            else
            {
                pMain->eLicenseIds = std::move(pTarget->eLicenseIds);
                pMain->eLicenseCount = pTarget->eLicenseCount;
            }
        }
    }

    int MergeAssignedELicenseInfoList(std::array<AssignedELicenseInfo, account::UserCountMax>* pMainInfoList, int mainCount, std::array<AssignedELicenseInfo, account::UserCountMax>* pTargetInfoList, int targetCount) NN_NOEXCEPT
    {
        int additionalCount = 0;
        for (auto& info : util::MakeSpan(pTargetInfoList->data(), targetCount))
        {
            auto found = std::find_if(pMainInfoList->begin(), pMainInfoList->begin() + mainCount, [&info](const AssignedELicenseInfo& mainInfo) NN_NOEXCEPT
            {
                return mainInfo.ownerId == info.ownerId;
            });

            if (found != pMainInfoList->begin() + mainCount)
            {
                MergeELicenseList(&found->dynamicRightsELicense, &info.dynamicRightsELicense);
                MergeELicenseList(&found->deviceLinkedELicense, &info.deviceLinkedELicense);
            }
            else
            {
                (*pMainInfoList)[mainCount + additionalCount] =
                {
                    info.ownerId,
                    { std::move(info.dynamicRightsELicense.eLicenseIds), info.dynamicRightsELicense.eLicenseCount },
                    { std::move(info.deviceLinkedELicense.eLicenseIds), info.deviceLinkedELicense.eLicenseCount },
                };
                additionalCount++;
            }
        }
        return mainCount + additionalCount;
    }
}

    //--------------------------------------------------------------------------
    //  QueryApplicationContentRightsManager
    //--------------------------------------------------------------------------
    QueryApplicationContentRightsManager::QueryApplicationContentRightsManager(
        ApplicationRecordDatabase* pDb,
        IntegratedContentManager* pIntegrated,
        ncm::ApplicationId appId,
        ApplicationRightsQueryFlags flags) NN_NOEXCEPT
        : m_pRecordDb(pDb), m_pIntegrated(pIntegrated), m_AppId(appId), m_Flags(flags), m_AppRightsId(), m_QueryAppRights(false), m_QueryAocRights(false)
    {
    }

    Result QueryApplicationContentRightsManager::Initialize() NN_NOEXCEPT
    {
        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;

        NN_RESULT_DO(m_pRecordDb->FindLaunchableApplicationAndPatch(&appKey, &patchKey, m_AppId));
        if (appKey)
        {
            int count;
            NN_RESULT_DO(srv::GetRightsIdList(&count, &m_AppRightsId, 1, *appKey, m_pIntegrated));
        }
        NN_RESULT_SUCCESS;
    }

    Result QueryApplicationContentRightsManager::GetRightsIdList(int* outCount, es::RightsId* outList, int listCount) NN_NOEXCEPT
    {
        int count = 0;
        NN_RESULT_DO(ForEachExternalKeyContentCandidate(m_pRecordDb, m_pIntegrated, m_AppId, [&, this](bool* outIsEnd, const ncm::ContentMetaKey& key) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;
            NN_RESULT_THROW_UNLESS(count < listCount, ResultBufferNotEnough());
            switch(key.type)
            {
            case ncm::ContentMetaType::Application:
                {
                    int tmpCount;
                    NN_RESULT_DO(srv::GetRightsIdList(&tmpCount, &outList[count], listCount - count, key, m_pIntegrated));
                    m_QueryAppRights = tmpCount > 0;
                    count += tmpCount;
                }
                break;
            case ncm::ContentMetaType::Patch:
                // Patch は CommonKey なので問い合わせない
                break;
            case ncm::ContentMetaType::AddOnContent:
                {
                    int tmpCount;
                    NN_RESULT_DO(GetRightsIdListLightly(&tmpCount, &outList[count], listCount - count, key, m_pIntegrated));
                    m_QueryAocRights = tmpCount > 0;
                    count += tmpCount;
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            NN_RESULT_SUCCESS;
        }));
        *outCount = count;
        NN_RESULT_SUCCESS;
    }

    ApplicationContentType QueryApplicationContentRightsManager::GetApplicationContentType(const es::RightsId& rightsId) const NN_NOEXCEPT
    {
        return (rightsId == m_AppRightsId) ? ApplicationContentType::Application : ApplicationContentType::AddOnContent;
    }

    bool QueryApplicationContentRightsManager::IsQueried(ApplicationContentType type) const NN_NOEXCEPT
    {
        switch (type)
        {
        case ApplicationContentType::Application:
            return m_QueryAppRights;
        case ApplicationContentType::AddOnContent:
            return m_QueryAocRights;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    bool QueryApplicationContentRightsManager::NeedsRights(const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        bool needsApp = m_Flags.Test<ApplicationRightsQueryFlag_Application>();
        bool needsAoc = m_Flags.Test<ApplicationRightsQueryFlag_AddOnContent>();

        if (needsApp && key.type == ncm::ContentMetaType::Application)
        {
            return true;
        }
        if (needsAoc && key.type == ncm::ContentMetaType::AddOnContent)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    //--------------------------------------------------------------------------
    //  AsyncApplicationRightsOnServerImpl
    //--------------------------------------------------------------------------

    AsyncApplicationRightsOnServerImpl::AsyncApplicationRightsOnServerImpl(
        ApplicationRecordDatabase* pDb,
        IntegratedContentManager* pIntegrated,
        ncm::ApplicationId id,
        const account::Uid& uid,
        ApplicationRightsQueryFlags queryFlags,
        RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
        : m_QueryRightsManager(pDb, pIntegrated, id, queryFlags), m_AppId(id), m_Uid(uid), m_QueryFlags(queryFlags), m_Stopper(std::move(stopper))
    {
        if (m_QueryFlags.Test<ApplicationRightsQueryFlag_Application>())
        {
            m_ApplicationRightsList[m_Count++] = { m_AppId, m_Uid, {}, ApplicationContentType::Application };
        }
        if (m_QueryFlags.Test<ApplicationRightsQueryFlag_AddOnContent>())
        {
            m_ApplicationRightsList[m_Count++] = { m_AppId, m_Uid, {}, ApplicationContentType::AddOnContent };
        }
    }

    AsyncApplicationRightsOnServerImpl::~AsyncApplicationRightsOnServerImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            g_ThreadAllocator.Free(*m_ThreadInfo);
        }
    }

    Result AsyncApplicationRightsOnServerImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(g_ThreadAllocator.Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncApplicationRightsOnServerTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                g_ThreadAllocator.Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncApplicationRightsOnServerImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncApplicationRightsOnServerTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    void AsyncApplicationRightsOnServerImpl::CancelImpl() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);

        if (m_Async)
        {
            m_Async->Cancel();
        }

        m_IsCanceled = true;
    }

    size_t AsyncApplicationRightsOnServerImpl::GetSizeImpl() NN_NOEXCEPT
    {
        return sizeof(ApplicationRightsOnServer) * m_Count;
    }

    Result AsyncApplicationRightsOnServerImpl::GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Result);

        const auto size = GetSizeImpl();
        NN_RESULT_THROW_UNLESS(size <= buffer.GetSize(), ResultBufferNotEnough());

        std::memcpy(buffer.GetPointerUnsafe(), m_ApplicationRightsList.data(), size);

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationRightsOnServerImpl::Execute() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_AvailableELicenseHolderMutex);

        NN_RESULT_DO(m_QueryRightsManager.Initialize());

        // NA が連携されていないアカウントだった場合、機器認証のみを問い合わせるようにする
        m_Uid = IsAvailableNaIdToken(m_Uid) ? m_Uid : account::InvalidUid;

        auto debugFlags = GetDebugApplicationRightsOnServerFlag();
        if (debugFlags)
        {
            AddDebugFlags(*debugFlags);
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(QueryAvailableELicenses());

        m_Async->Wait();

        AvailableELicenseHolderHandle handle;
        NN_RESULT_DO(SetUpAvailableELicenseHolder(&handle));

        NN_RESULT_DO(AddLicenseTypeFlags(handle));

        NN_RESULT_SUCCESS;
    }

    void AsyncApplicationRightsOnServerImpl::AddDebugFlags(ApplicationRightsOnServerFlag flags) NN_NOEXCEPT
    {
        for (auto& rights : util::MakeSpan(m_ApplicationRightsList.data(), m_Count))
        {
            rights._flags = flags;
            rights._handle = InvalidAvailableELicenseHolderHandle.value;
        }
    }

    Result AsyncApplicationRightsOnServerImpl::QueryAvailableELicenses() NN_NOEXCEPT
    {
        auto rightsIdList = std::make_unique<es::RightsId[]>(MaxRightsIdCount);
        NN_RESULT_THROW_UNLESS(rightsIdList, ResultBufferNotEnough());
        int count;
        NN_RESULT_DO(m_QueryRightsManager.GetRightsIdList(&count, rightsIdList.get(), MaxRightsIdCount));

        std::sort(rightsIdList.get(), rightsIdList.get() + count);
        NN_RESULT_DO(CreateAsyncObject(&m_Async, &m_CancelMutex, [&rightsIdList, &count, this](nim::AsyncAvailableELicenses* pAsync) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(RequestQueryAvailableELicensesImpl(pAsync, rightsIdList.get(), count, m_Uid));
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationRightsOnServerImpl::SetUpAvailableELicenseHolder(AvailableELicenseHolderHandle* outValue) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Async->TryWait());
        NN_UTIL_LOCK_GUARD(m_CancelMutex);
        auto writer = g_AvailableELicenseHolder.OpenWriter();
        auto handle = writer.GetHandle();
        NN_RESULT_TRY(SetUpAvailableELicenseHolderImpl(&(*m_Async), [&, this](const nim::AvailableELicense& eLicense) NN_NOEXCEPT
        {
            ApplicationContentType rightsType = m_QueryRightsManager.GetApplicationContentType(eLicense.rightsId);
            if (eLicense.licenseType != nim::ELicenseType::Unknown &&
                rightsType != ApplicationContentType::Unknown)
            {
                auto rights = std::find_if(m_ApplicationRightsList.begin(), m_ApplicationRightsList.end(), [&rightsType](const ApplicationRightsOnServer& ref) NN_NOEXCEPT
                {
                    return ref.type == rightsType;
                });

                if (rights != m_ApplicationRightsList.end())
                {
                    auto flags = GetAvailabilityFlag(eLicense.licenseStatus);
                    rights->_flags |= flags;
                    rights->reason._flags |= GetApplicationRightsUnavailableFlag(eLicense.licenseStatus);
                    rights->_handle = handle.value;

                    if (flags.Test<ApplicationRightsOnServerFlag_HasAvailableRights>())
                    {
                        writer.Add(eLicense);
                    }
                }
            }
        }))
            NN_RESULT_CATCH(ResultSystemUpdateRequired)
            {
                for (auto& rights : util::MakeSpan(m_ApplicationRightsList.data(), m_Count))
                {
                    rights._flags = ApplicationRightsOnServerFlag_HasUnavailableRights::Mask;
                    rights.reason._flags = ApplicationRightsUnavailableFlag_SystemUpdateRequired::Mask;
                }
            }
        NN_RESULT_END_TRY

        *outValue = handle;
        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationRightsOnServerImpl::AddLicenseTypeFlags(const AvailableELicenseHolderHandle& handle) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(g_AvailableELicenseHolder.CanOpenReader(handle), ResultUnexpectedUsage());
        auto reader = g_AvailableELicenseHolder.OpenReader(handle);
        for (auto rightsId : reader.GetRightsIdSpan())
        {
            ApplicationContentType rightsType = m_QueryRightsManager.GetApplicationContentType(rightsId);
            // すでにフィルタ済みのはず
            NN_ABORT_UNLESS(rightsType != ApplicationContentType::Unknown);

            auto rights = std::find_if(m_ApplicationRightsList.begin(), m_ApplicationRightsList.end(), [&rightsType](const ApplicationRightsOnServer& ref) NN_NOEXCEPT
            {
                return ref.type == rightsType;
            });
            if (rights != m_ApplicationRightsList.end())
            {
                auto licenseType = reader.GetLicenseType(rightsId);
                rights->_flags |= GetLicenseTypeFlag(licenseType);
            }
        }

        for (auto& rights : util::MakeSpan(m_ApplicationRightsList.data(), m_Count))
        {
            if (!rights._flags.IsAnyOn() && m_QueryRightsManager.IsQueried(rights.type))
            {
                rights._flags |= ApplicationRightsOnServerFlag_HasUnavailableRights::Mask;
                rights.reason._flags |= ApplicationRightsUnavailableFlag_NoRights::Mask;
            }
        }

        NN_RESULT_SUCCESS;
    }

    bool AsyncApplicationRightsOnServerImpl::IsCanceled() const NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);
        return m_IsCanceled;
    }

    //--------------------------------------------------------------------------
    //  AsyncAssignRightsImpl
    //--------------------------------------------------------------------------

    Result AsyncAssignRightsImpl::Initialize(
        ApplicationRecordDatabase* pRecord,
        IntegratedContentManager* pIntegrated,
        const ApplicationRightsOnServer list[],
        int numList,
        RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(numList > 0, ResultInvalidListCount());
        NN_RESULT_THROW_UNLESS(numList <= static_cast<int>(m_ApplicationRightsList.size()), ResultBufferNotEnough());

        m_pRecord = pRecord;
        m_pIntegrated = pIntegrated;
        const auto& base = list[0];
        NN_RESULT_THROW_UNLESS(std::all_of(list, list + numList, [&base](const ApplicationRightsOnServer& rights) NN_NOEXCEPT
        {
            return (base.id == rights.id) && (base.uid == rights.uid);
        }), ResultNotImplemented());
        m_AppId = base.id;
        m_Uid = base.uid;
        m_NumList = numList;
        std::copy(list, list + numList, m_ApplicationRightsList.data());
        m_Stopper = std::move(stopper);
        NN_RESULT_SUCCESS;
    }

    AsyncAssignRightsImpl::~AsyncAssignRightsImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            g_ThreadAllocator.Free(*m_ThreadInfo);
        }
    }

    Result AsyncAssignRightsImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(g_ThreadAllocator.Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncAssignRightsTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                g_ThreadAllocator.Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncAssignRightsImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncAssignRightsTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    void AsyncAssignRightsImpl::CancelImpl() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);

        if (m_AssignAsync)
        {
            m_AssignAsync->Cancel();
        }
        if (m_ResultAsync)
        {
            m_ResultAsync->Cancel();
        }
        if (m_EcAsync)
        {
            m_EcAsync->Cancel();
        }

        m_IsCanceled = true;
    }

    Result AsyncAssignRightsImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    Result AsyncAssignRightsImpl::Execute() NN_NOEXCEPT
    {
        if (IsDebugELicenseImportEnabled())
        {
            NN_RESULT_DO(ImportELicensesByDebugUtility());
        }
        else
        {
            NN_RESULT_DO(ImportELicensesFromServer());
        }
        NN_RESULT_SUCCESS;
    }

    Result AsyncAssignRightsImpl::ImportELicensesFromServer() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_AvailableELicenseHolderMutex);

        std::array<AssignedELicenseInfo, account::UserCountMax> infoList = { {} };
        int infoCount = 0;
        for (const auto& licenseType : LicenseTypeList)
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());
            auto found = std::find_if(m_ApplicationRightsList.begin(), m_ApplicationRightsList.end(), [&licenseType](const ApplicationRightsOnServer& rights) NN_NOEXCEPT
            {
                const auto& flag = licenseType.applicationRightsOnServerFlag;
                return (rights._flags & flag) == flag;
            });
            if (found == m_ApplicationRightsList.end())
            {
                continue;
            }

            AvailableELicenseHolderHandle handle { found->_handle };
            NN_RESULT_DO(RequestAssignELicensesImpl(&m_AssignAsync, &m_CancelMutex, handle, m_Uid, licenseType.eLicenseType));

            m_AssignAsync->Wait();

            {
                NN_UTIL_LOCK_GUARD(m_CancelMutex);
                std::array<AssignedELicenseInfo, account::UserCountMax> tmpList = { {} };
                int tmpCount;
                NN_RESULT_DO(GetAssignedELicenseInfoList(&tmpCount, tmpList.data(), static_cast<int>(tmpList.size()), &(*m_AssignAsync), handle));
                infoCount = MergeAssignedELicenseInfoList(&infoList, infoCount, &tmpList, tmpCount);

                // 破棄しないと、次のタスクが実行できない
                m_AssignAsync = util::nullopt;
            }
        }

        if (infoCount <= 0)
        {
            NN_RESULT_SUCCESS;
        }

        for (auto& info : util::MakeSpan(infoList.data(), infoCount))
        {
            NN_RESULT_DO(SyncELicensesImpl(&m_ResultAsync, &m_CancelMutex, info.ownerId));
        }

        NN_RESULT_DO(SyncTicketImpl(&m_EcAsync, &m_CancelMutex, infoList.data(), infoCount));

        NN_RESULT_DO(DownloadDynamicETicketIfNecessaryImpl(&m_ResultAsync, &m_CancelMutex, infoList.data(), infoCount, m_Uid));

        NN_RESULT_SUCCESS;
    }

    Result AsyncAssignRightsImpl::ImportELicensesByDebugUtility() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, m_Uid));
        es::ELicenseImportContext context;
        NN_RESULT_DO(es::BeginImportELicenseArchive(&context, naId));
        auto archiveInfo = es::debug::MakeELicenseArchiveInfo(context.GetChallenge().value, naId);
        constexpr size_t MaxElicenseCountForDebug = 16;
        std::array<es::debug::ELicenseInfoForDebug, MaxElicenseCountForDebug> eLicenseInfoList {};
        int elicenseCount = 0;

        for (const auto& licenseType : LicenseTypeList)
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());
            auto found = std::any_of(m_ApplicationRightsList.begin(), m_ApplicationRightsList.end(), [&licenseType](const ApplicationRightsOnServer& rights) NN_NOEXCEPT
            {
                const auto& flag = licenseType.applicationRightsOnServerFlag;
                return (rights._flags & flag) == flag;
            });
            if (!found)
            {
                continue;
            }

            es::RightsIdIncludingKeyId checkedRightsId {};
            NN_RESULT_DO(ForEachRightsId([&naId, &licenseType, &eLicenseInfoList, &elicenseCount, &checkedRightsId](bool* outIsEnd, const es::RightsIdIncludingKeyId& rightsIdIncludingKeyId) NN_NOEXCEPT -> Result
            {
                *outIsEnd = false;

                // 内部鍵であれば、対応不要
                if (!rightsIdIncludingKeyId.IsExternalKey())
                {
                    NN_RESULT_SUCCESS;
                }
                // 最適化：同じ ContentMeta に紐づくコンテンツは RightsId が同じなので、1個前のと同じかどうかで重複を省く
                if (checkedRightsId == rightsIdIncludingKeyId)
                {
                    NN_RESULT_SUCCESS;
                }
                checkedRightsId = rightsIdIncludingKeyId;

                auto elicense = MakeELicenseInfoForDebug(licenseType.applicationRightsOnServerFlag, rightsIdIncludingKeyId, naId);
                if (elicense)
                {
                    NN_ABORT_UNLESS(elicenseCount < eLicenseInfoList.size());
                    eLicenseInfoList[elicenseCount++] = *elicense;
                    if (elicenseCount >= eLicenseInfoList.size())
                    {
                        *outIsEnd = true;
                    }
                }
                NN_RESULT_SUCCESS;
            }));
        }

        {
            constexpr size_t JsonBufferSize = 16 * 1024; // 16 KB
            auto jsonBuffer = std::make_unique<char[]>(JsonBufferSize);
            NN_RESULT_THROW_UNLESS(jsonBuffer, ResultBufferNotEnough());

            size_t jsonSize = es::debug::BuildELicenseArchiveJsonString(jsonBuffer.get(), JsonBufferSize, archiveInfo, eLicenseInfoList.data(), elicenseCount);
            NN_RESULT_THROW_UNLESS(jsonSize > 0, ResultNotImplemented());

            NN_RESULT_DO(es::ImportELicenseArchive(context, jsonBuffer.get(), jsonSize));
            es::ELicenseArchiveId elicenseId;
            NN_RESULT_DO(es::EndImportELicenseArchiveForDebug(&elicenseId, context));
        }

        {
            // DeviceLinked とそれ以外で分割する
            auto endIterator = eLicenseInfoList.begin() + elicenseCount;
            auto deviceLinkedInfoBegin = std::remove_if(eLicenseInfoList.begin(), endIterator, [](const es::debug::ELicenseInfoForDebug& info) NN_NOEXCEPT
            {
                return info.scope != es::ELicenseScope::Device;
            });

            int temporaryRightsCount = static_cast<int>(std::distance(eLicenseInfoList.begin(), deviceLinkedInfoBegin));
            int deviceLinkedRightsCount = static_cast<int>(std::distance(deviceLinkedInfoBegin, endIterator));
            NN_ABORT_UNLESS(elicenseCount == temporaryRightsCount + deviceLinkedRightsCount);

            std::unique_ptr<es::ELicenseId[]> temporaryList;
            if (temporaryRightsCount)
            {
                temporaryList = std::make_unique<es::ELicenseId[]>(temporaryRightsCount);
                NN_ABORT_UNLESS(temporaryList);
            }
            std::unique_ptr<es::ELicenseId[]> deviceLinkedPermanentList;
            if (deviceLinkedRightsCount > 0)
            {
                deviceLinkedPermanentList = std::make_unique<es::ELicenseId[]>(deviceLinkedRightsCount);
            }
            AssignedELicenseInfo info = { naId, { std::move(temporaryList), temporaryRightsCount }, { std::move(deviceLinkedPermanentList), deviceLinkedRightsCount } };

            // DeviceLinked
            {
                int copiedCount = 0;
                auto& list = info.deviceLinkedELicense;
                for(auto debugInfo : util::MakeSpan(deviceLinkedInfoBegin, endIterator))
                {
                    NN_ABORT_UNLESS(copiedCount < list.eLicenseCount);
                    list.eLicenseIds[copiedCount++] = debugInfo.id;
                }
                NN_RESULT_DO(SyncTicketImpl(&m_EcAsync, &m_CancelMutex, &info, 1));
            }

            // Temporary
            {
                int copiedCount = 0;
                auto& list = info.dynamicRightsELicense;
                for(auto& debugInfo : util::MakeSpan(eLicenseInfoList.begin(), deviceLinkedInfoBegin))
                {
                    NN_ABORT_UNLESS(copiedCount < list.eLicenseCount);
                    list.eLicenseIds[copiedCount++] = debugInfo.id;
                }
                NN_RESULT_DO(DownloadDynamicETicketIfNecessaryImpl(&m_ResultAsync, &m_CancelMutex, &info, 1, m_Uid));
            }
        }

#endif // ~defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_RESULT_SUCCESS;
    }

    bool AsyncAssignRightsImpl::IsCanceled() const NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);
        return m_IsCanceled;
    }

    template <class FuncT>
    Result AsyncAssignRightsImpl::ForEachRightsId(FuncT func) NN_NOEXCEPT
    {
        NN_RESULT_DO(ForEachExternalKeyContentCandidate(m_pRecord, m_pIntegrated, m_AppId, [&, this](bool* isEnd, const ncm::ContentMetaKey& key) NN_NOEXCEPT -> Result
        {
            *isEnd = false;

            auto listFunc = [&key, this](int* outCount, es::RightsIdIncludingKeyId* outList, int listCount, int offset) NN_NOEXCEPT -> Result
            {
                return GetRightsIdIncludingKeyIdList(outCount, outList, listCount, key, m_pIntegrated, offset);
            };

            NN_RESULT_DO(ForEachFunc<es::RightsIdIncludingKeyId>(listFunc, [&](bool* outIsEnd, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(func(isEnd, rightsId));
                *outIsEnd = *isEnd;
                NN_RESULT_SUCCESS;
            }));

            NN_RESULT_SUCCESS;
        }));
        NN_RESULT_SUCCESS;
    }

    //--------------------------------------------------------------------------
    //  AsyncAssignRightsToResumeImpl
    //--------------------------------------------------------------------------
    AsyncAssignRightsToResumeImpl::AsyncAssignRightsToResumeImpl(
        RightsEnvironmentManager *pRightsEnvironmentManager,
        RightsEnvironmentHandle handle,
        const account::Uid& uid,
        RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
        : m_pRightsEnvironmentManager(pRightsEnvironmentManager), m_Handle(handle), m_Uid(uid), m_Stopper(std::move(stopper))
    {
    }

    AsyncAssignRightsToResumeImpl::~AsyncAssignRightsToResumeImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            g_ThreadAllocator.Free(*m_ThreadInfo);
        }
    }

    Result AsyncAssignRightsToResumeImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(g_ThreadAllocator.Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncAssignRightsTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                g_ThreadAllocator.Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncAssignRightsToResumeImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncAssignRightsTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;
        NN_RESULT_SUCCESS;
    }

    void AsyncAssignRightsToResumeImpl::CancelImpl() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);

        if (m_AsyncAvailableELicenses)
        {
            m_AsyncAvailableELicenses->Cancel();
        }
        if (m_AsyncAssignedELicenses)
        {
            m_AsyncAssignedELicenses->Cancel();
        }
        if (m_AsyncSyncELicense)
        {
            m_AsyncSyncELicense->Cancel();
        }
        if (m_AsyncSyncTicket)
        {
            m_AsyncSyncTicket->Cancel();
        }
        if (m_AsyncDownloadETicket)
        {
            m_AsyncDownloadETicket->Cancel();
        }

        m_IsCanceled = true;
    }

    Result AsyncAssignRightsToResumeImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    bool AsyncAssignRightsToResumeImpl::IsCanceled() const NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_CancelMutex);
        return m_IsCanceled;
    }

    Result AsyncAssignRightsToResumeImpl::Execute() NN_NOEXCEPT
    {
        // NA が連携されていないアカウントだった場合、機器認証のみを問い合わせるようにする
        m_Uid = IsAvailableNaIdToken(m_Uid) ? m_Uid : account::InvalidUid;

        NN_UTIL_LOCK_GUARD(g_AvailableELicenseHolderMutex);
        AvailableELicenseHolderHandle handle;
        NN_RESULT_DO(QueryAvailableELicenses(&handle));
        NN_RESULT_DO(AssignRights(handle));

        NN_RESULT_SUCCESS;
    }

    Result AsyncAssignRightsToResumeImpl::QueryAvailableELicenses(AvailableELicenseHolderHandle* outHandle) NN_NOEXCEPT
    {
        {
            auto rightsIdList = std::make_unique<es::RightsId>(MaxRightsIdCount);
            NN_ABORT_UNLESS(rightsIdList);

            auto count = m_pRightsEnvironmentManager->ListActivatedRights(rightsIdList.get(), MaxRightsIdCount, m_Handle);

            NN_RESULT_DO(CreateAsyncObject(&m_AsyncAvailableELicenses, &m_CancelMutex, [&rightsIdList, &count, this](nim::AsyncAvailableELicenses* pAsync) NN_NOEXCEPT -> Result
            {
                NN_RESULT_DO(RequestQueryAvailableELicensesImpl(pAsync, rightsIdList.get(), count, m_Uid));
                NN_RESULT_SUCCESS;
            }));
        }

        m_AsyncAvailableELicenses->Wait();

        NN_UTIL_LOCK_GUARD(m_CancelMutex);
        auto writer = g_AvailableELicenseHolder.OpenWriter();
        NN_RESULT_DO(SetUpAvailableELicenseHolderImpl(&(*m_AsyncAvailableELicenses), [&writer](const nim::AvailableELicense& eLicense) NN_NOEXCEPT
        {
            writer.Add(eLicense);
        }));

        *outHandle = writer.GetHandle();
        NN_RESULT_SUCCESS;
    }

    Result AsyncAssignRightsToResumeImpl::AssignRights(AvailableELicenseHolderHandle handle) NN_NOEXCEPT
    {
        std::array<AssignedELicenseInfo, account::UserCountMax> infoList;
        int infoCount = 0;
        for (const auto& licenseType : LicenseTypeList)
        {
            NN_RESULT_DO(RequestAssignELicensesImpl(&m_AsyncAssignedELicenses, &m_CancelMutex, handle, m_Uid, licenseType.eLicenseType));

            if (!m_AsyncAssignedELicenses)
            {
                continue;
            }

            m_AsyncAssignedELicenses->Wait();

            {
                NN_UTIL_LOCK_GUARD(m_CancelMutex);
                std::array<AssignedELicenseInfo, account::UserCountMax> tmpList;
                int tmpCount;
                NN_RESULT_DO(GetAssignedELicenseInfoList(&tmpCount, tmpList.data(), static_cast<int>(tmpList.size()), &(*m_AsyncAssignedELicenses), handle));
                infoCount = MergeAssignedELicenseInfoList(&infoList, infoCount, &tmpList, tmpCount);

                // 破棄しないと、次のタスクが実行できない
                m_AsyncAssignedELicenses = util::nullopt;
            }
        }

        if (infoCount <= 0)
        {
            NN_RESULT_SUCCESS;
        }

        for (auto& info : util::MakeSpan(infoList.data(), infoCount))
        {
            NN_RESULT_DO(SyncELicensesImpl(&m_AsyncSyncELicense, &m_CancelMutex, info.ownerId));
        }

        NN_RESULT_DO(SyncTicketImpl(&m_AsyncSyncTicket, &m_CancelMutex, infoList.data(), infoCount));

        NN_RESULT_DO(DownloadDynamicETicketIfNecessaryImpl(&m_AsyncDownloadETicket, &m_CancelMutex, infoList.data(), infoCount, m_Uid));

        NN_RESULT_SUCCESS;
    }
}}} // ~nn::ns::srv
