﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationVerificationManager.h>
#include <nn/util/util_Optional.h>

#include "ns_AsyncImpl.h"
#include "ns_AsyncVerifyContentsImpl.h"
#include "ns_Config.h"

namespace nn { namespace ns { namespace srv {
namespace {
    typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

    Result FindContentImpl(util::optional<ncm::ContentId>* outValue, const ncm::ContentMetaKey& key, ncm::ContentType contentType, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        ncm::ContentId contentId;
        bool found = true;

        uint8_t programIndex;
        NN_RESULT_DO(pContentManager->GetMinimumProgramIndex(&programIndex, key));

        NN_RESULT_TRY(pContentManager->GetContentIdByStorageId(&contentId, key, contentType, programIndex, storageId))
            NN_RESULT_CATCH(ncm::ResultContentNotFound) { found = false; }
        NN_RESULT_END_TRY

        if (found)
        {
            *outValue = contentId;
        }
        else
        {
            *outValue = util::nullopt;
        }

        NN_RESULT_SUCCESS;
    }


    Result FindContent(util::optional<ncm::ContentId>* outValue, const ncm::ContentMetaKey& key, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        util::optional<ncm::ContentId> contentId;
        switch (key.type)
        {
        case ncm::ContentMetaType::Application:
        case ncm::ContentMetaType::Patch:
            NN_RESULT_DO(FindContentImpl(&contentId, key, ncm::ContentType::Program, pContentManager, storageId));
            break;
        case ncm::ContentMetaType::AddOnContent:
            NN_RESULT_DO(FindContentImpl(&contentId, key, ncm::ContentType::Data, pContentManager, storageId));
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        *outValue = contentId;
        NN_RESULT_SUCCESS;
    }

    Result SelectCorruptContent(util::optional<ncm::ContentId>* outValue, const ncm::ContentMetaKey& key, IntegratedContentManager* pContentManager, ncm::StorageId storageId)
    {
        bool hasContent;
        NN_RESULT_DO(pContentManager->HasContentMetaKey(&hasContent, key, storageId));
        if (hasContent)
        {
            NN_RESULT_DO(FindContent(outValue, key, pContentManager, storageId));
        }
        else
        {
            *outValue = util::nullopt;
        }

        NN_RESULT_SUCCESS;
    }

    Result WriteZero(ncm::ContentId contentId, int64_t begin, int64_t end, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        const int64_t bufferSize = 0x100;
        Bit8 buffer[bufferSize] = {};
        int64_t writeArea = end - begin;

        // 全部壊すのは時間がかかるので３箇所だけ壊す
        auto offset = begin;
        auto writeSize = static_cast<size_t>(writeArea < bufferSize ? end - offset : bufferSize);
        NN_RESULT_DO(pContentManager->WriteContentIdFile(contentId, offset, buffer, writeSize, storageId));

        if (writeArea > bufferSize)
        {
            offset = (end - begin) / 2 + begin;
            writeSize = static_cast<size_t>(offset + bufferSize > end ? end - offset : bufferSize);
            NN_RESULT_DO(pContentManager->WriteContentIdFile(contentId, offset, buffer, writeSize, storageId));

            offset = end - bufferSize;
            writeSize = static_cast<size_t>(offset + bufferSize > end ? end - offset : bufferSize);
            NN_RESULT_DO(pContentManager->WriteContentIdFile(contentId, offset, buffer, writeSize, storageId));
        }

        NN_RESULT_SUCCESS;
    }

    Result CorruptNcaHeader(ncm::ContentId contentId, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        const int64_t ncaHeaderSize = 0x400;
        NN_RESULT_DO(WriteZero(contentId, 0, ncaHeaderSize, pContentManager, storageId));

        NN_RESULT_SUCCESS;
    }

    Result CorruptProgramCode(ncm::ContentId contentId, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(pContentManager->GetContentIdFileSize(&size, contentId, storageId));

        // nnSdk.nso がない場合は、nnrtld.nso とrocrt.o を合わせて約 25 KB なので、そのぐらい壊しておく
        // nnSdk.nso が ある場合、nnSdk.nso が Release ビルドで 3MB 弱なので、 2.5 MB ぐらい破壊しておく
        // 上記条件に満たさない場合には、破壊するコンテンツがなかったとする
        const int64_t nnSdkNsoSize = 0x280000;
        const int64_t nnrtldSize = 0x6400;
        NN_RESULT_THROW_UNLESS(size >= nnrtldSize, ResultApplicationContentNotFound());
        int64_t offset = size - ((size >= nnSdkNsoSize) ? nnSdkNsoSize : nnrtldSize);

        NN_RESULT_DO(WriteZero(contentId, offset, size, pContentManager, storageId));

        NN_RESULT_SUCCESS;
    }

    Result CorruptSdCardEncryption() NN_NOEXCEPT
    {
        fs::EncryptionSeed seed = {};
        NN_RESULT_DO(fs::SetSdCardEncryptionSeed(seed));
        NN_RESULT_SUCCESS;
    }

    Result CorruptDataImpl(bool* doCorrupt, const ncm::ContentMetaKey& key, CorruptContentFlag flags, IntegratedContentManager* pContentManager, ncm::StorageId storageId)
    {
        util::optional<ncm::ContentId> contentId;
        NN_RESULT_DO(SelectCorruptContent(&contentId, key, pContentManager, storageId));
        if (contentId)
        {
            if ((key.type == ncm::ContentMetaType::Application || key.type == ncm::ContentMetaType::Patch) && flags.Test<CorruptContentFlag_Code>())
            {
                bool isSuccess = true;
                NN_RESULT_TRY(CorruptProgramCode(*contentId, pContentManager, storageId))
                    NN_RESULT_CATCH(ResultApplicationContentNotFound) { isSuccess = false; }
                NN_RESULT_END_TRY;
                *doCorrupt = isSuccess;
            }
            else
            {
                NN_RESULT_DO(CorruptNcaHeader(*contentId, pContentManager, storageId));
                *doCorrupt = true;
            }
        }
        else
        {
            *doCorrupt = false;
        }

        NN_RESULT_SUCCESS;
    }

    Result CorruptData(bool* outValue, ncm::ContentMetaType metaType, ncm::ApplicationId id, CorruptContentFlag flags, IntegratedContentManager* pContentManager, ncm::StorageId storageId) NN_NOEXCEPT
    {
        ncm::ContentMetaKey key;
        int count = pContentManager->ListInstalledApplicationContent(&key, 1, id, metaType);
        NN_RESULT_THROW_UNLESS(count == 1, ResultApplicationContentNotFound());

        ncm::StorageId registeredStorages[IntegratedContentManager::MaxStorageCount];
        auto numStorage = pContentManager->GetRegisteredStorages(registeredStorages, IntegratedContentManager::MaxStorageCount);
        *outValue = false;
        for (auto i = 0; i < numStorage; i++)
        {
            if (registeredStorages[i] != ncm::StorageId::Card &&
                (storageId == ncm::StorageId::Any || registeredStorages[i] == storageId))
            {
                if (registeredStorages[i] == ncm::StorageId::SdCard && flags.Test<CorruptContentFlag_SdCardEncryption>())
                {
                    NN_RESULT_DO(CorruptSdCardEncryption());
                    *outValue = true;
                }
                else
                {
                    bool doCorrupt;
                    NN_RESULT_DO(CorruptDataImpl(&doCorrupt, key, flags, pContentManager, registeredStorages[i]));
                    *outValue = doCorrupt ? doCorrupt : *outValue;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }
}

Result ApplicationVerificationManager::Initialize(IntegratedContentManager* pContentManager, ApplicationRecordDatabase* pRecordDb, RequestServer* pRequestServer, TicketManager* pTicketManager) NN_NOEXCEPT
{
    m_pContentManager = pContentManager;
    m_pRecordDb = pRecordDb;
    m_pRequestServer = pRequestServer;
    m_pTicketManager = pTicketManager;

    NN_RESULT_SUCCESS;
}

Result ApplicationVerificationManager::RequestVerifyApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IProgressAsyncResult>> outAsync, ncm::ApplicationId appId, Bit32 innerFlagsValue, sf::NativeHandle transferHandle, uint64_t transferSize) NN_NOEXCEPT
{
    VerifyContentFlag flags;
    flags._storage[0] = innerFlagsValue;

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IProgressAsyncResult, AsyncVerifyApplicationImpl>();
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Initialize(appId, flags, m_pRecordDb, m_pContentManager, m_pRequestServer->Stop(), std::move(transferHandle), static_cast<size_t>(transferSize)));

    NN_RESULT_DO(emplacedRef.GetImpl().Run());

    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result ApplicationVerificationManager::CorruptApplicationForDebug(ncm::ApplicationId id, Bit32 innerFlagsValue, ncm::StorageId storageId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(id != ncm::ApplicationId::GetInvalidId(), ResultApplicationRecordNotFound());

    CorruptContentFlag flags;
    flags._storage[0] = innerFlagsValue;
    bool doCorrupt = false;

    NN_RESULT_THROW_UNLESS((flags & CorruptContentFlag_All).IsAnyOn(), ResultApplicationContentNotFound());

    if (flags.Test<CorruptContentFlag_Application>())
    {
        bool isSuccess = true;
        NN_RESULT_TRY(CorruptData(&isSuccess, ncm::ContentMetaType::Application, id, flags, m_pContentManager, storageId))
            NN_RESULT_CATCH(ResultApplicationContentNotFound) { isSuccess = false; }
        NN_RESULT_END_TRY
        doCorrupt = isSuccess ? true : doCorrupt;
    }
    if (flags.Test<CorruptContentFlag_Patch>())
    {
        bool isSuccess = true;
        NN_RESULT_TRY(CorruptData(&isSuccess, ncm::ContentMetaType::Patch, id, flags, m_pContentManager, storageId))
            NN_RESULT_CATCH(ResultApplicationContentNotFound) { isSuccess = false; }
        NN_RESULT_END_TRY
        doCorrupt = isSuccess ? true : doCorrupt;
    }
    if (flags.Test<CorruptContentFlag_AddOnContent>())
    {
        bool isSuccess = true;
        NN_RESULT_TRY(CorruptData(&isSuccess, ncm::ContentMetaType::AddOnContent, id, flags, m_pContentManager, storageId))
            NN_RESULT_CATCH(ResultApplicationContentNotFound) { isSuccess = false; }
        NN_RESULT_END_TRY
        doCorrupt = isSuccess ? true : doCorrupt;
    }
    NN_RESULT_THROW_UNLESS(doCorrupt, ResultApplicationContentNotFound());

    NN_RESULT_SUCCESS;
}

Result ApplicationVerificationManager::CorruptContentForDebug(const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
{
    ncm::StorageId registeredStorages[IntegratedContentManager::MaxStorageCount];
    auto numStorage = m_pContentManager->GetRegisteredStorages(registeredStorages, IntegratedContentManager::MaxStorageCount);
    bool hasCorrupted = false;
    for (auto i = 0; i < numStorage; i++)
    {
        const auto registeredStorageId = registeredStorages[i];
        if (ncm::IsInstallableStorage(registeredStorageId) &&
            (storageId == ncm::StorageId::Any || registeredStorageId == storageId))
        {
            bool hasKey;
            NN_RESULT_DO(m_pContentManager->HasContentMetaKey(&hasKey, key, registeredStorageId));
            if (!hasKey)
            {
                continue;
            }
            int offset = 0;
            while (NN_STATIC_CONDITION(true))
            {
                const int InfoCount = 16;
                ncm::ContentInfo infoList[InfoCount];
                int count;
                NN_RESULT_DO(m_pContentManager->ListContentInfo(&count, infoList, InfoCount, key, offset, registeredStorageId));
                for (int j = 0; j < count; j++)
                {
                    NN_RESULT_DO(CorruptNcaHeader(infoList[j].GetId(), m_pContentManager, registeredStorageId));
                    hasCorrupted = true;
                }
                if (count < InfoCount)
                {
                    break;
                }
                offset += count;
            }
        }
    }
    NN_RESULT_THROW_UNLESS(hasCorrupted, ResultContentMetaNotFound());
    NN_RESULT_SUCCESS;
}

Result ApplicationVerificationManager::RequestVerifyAddOnContentsRights(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IProgressAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IProgressAsyncResult, AsyncVerifyAddOnContentsImpl>();
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Initialize(id, m_pContentManager, m_pRecordDb, m_pTicketManager));

    NN_RESULT_DO(emplacedRef.GetImpl().Run());

    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

}}}
