﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/fs.h>
#include <nn/fs/fs_Result.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationContentMetaStatusUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Span.h>
#include "ns_SystemSaveDataUtil.h"
#include "ns_Config.h"

namespace nn { namespace ns { namespace srv {

    namespace {
        typedef kvdb::BoundedString<32> Path;

        const char* MountName = "apprecdb";

        Path MakeApplicationRecordIndexPath() NN_NOEXCEPT
        {
            Path path(MountName);
            return path.Append(":/index");
        }

        Path MakeKvsDirectoryPath() NN_NOEXCEPT
        {
            Path path(MountName);
            return path.Append(":/kvs");
        }

        Result GetLaunchableContentMetaKey(util::optional<ncm::ContentMetaKey>* outValue, util::optional<ncm::ContentMetaKey> key, IntegratedContentManager* pIntegrated, ncm::StorageId storageId) NN_NOEXCEPT
        {
            bool hasKey = false;
            if (key)
            {
                NN_RESULT_DO(pIntegrated->HasContentMetaKey(&hasKey, *key, storageId));
            }

            // util::optional の制約で三項演算子で書けない
            if (hasKey)
            {
                *outValue = key;
            }
            else
            {
                *outValue = util::nullopt;
            }

            NN_RESULT_SUCCESS;
        }

        util::optional<ncm::ContentMetaKey> DecideContentMetaKeyImpl(
            util::optional<ncm::ContentMetaKey> appKey,
            util::optional<ncm::ContentMetaKey> patchKey) NN_NOEXCEPT
        {
            if (!patchKey)
            {
                return appKey;
            }
            else if (appKey->version < patchKey->version)
            {
                return patchKey;
            }
            else
            {
                return appKey;
            }
        }

        util::optional<ncm::ContentMetaKey> DecideControlContentMetaKey(
            util::optional<ncm::ContentMetaKey> appKey,
            util::optional<ncm::ContentMetaKey> patchKey) NN_NOEXCEPT
        {
            if (!appKey)
            {
                return patchKey;
            }
            return DecideContentMetaKeyImpl(appKey, patchKey);
        }

        Result DecideLaunchableMainContentMetaKey(
            util::optional<ncm::ContentMetaKey>* outValue,
            util::optional<ncm::ContentMetaKey> appKey,
            util::optional<ncm::ContentMetaKey> patchKey,
            IntegratedContentManager* pIntegrated,
            ncm::StorageId appStorageId,
            ncm::StorageId patchStorageId) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(appKey, ResultApplicationContentNotFound());

            util::optional<ncm::ContentMetaKey> launchableAppKey;
            NN_RESULT_DO(GetLaunchableContentMetaKey(&launchableAppKey, appKey, pIntegrated, appStorageId));
            NN_RESULT_THROW_UNLESS(launchableAppKey, ResultApplicationContentNotFound());

            util::optional<ncm::ContentMetaKey> launchablePatchKey;
            NN_RESULT_DO(GetLaunchableContentMetaKey(&launchablePatchKey, patchKey, pIntegrated, patchStorageId));

            *outValue = DecideMainContentMetaKey(launchableAppKey, launchablePatchKey);
            NN_RESULT_SUCCESS;
        }

        Result DecideReadableControlContentMetaKey(
            util::optional<ncm::ContentMetaKey>* outValue,
            util::optional<ncm::ContentMetaKey> appKey,
            util::optional<ncm::ContentMetaKey> patchKey,
            IntegratedContentManager* pIntegrated) NN_NOEXCEPT
        {
            util::optional<ncm::ContentMetaKey> readableAppKey;
            NN_RESULT_DO(GetLaunchableContentMetaKey(&readableAppKey, appKey, pIntegrated, ncm::StorageId::Any));
            util::optional<ncm::ContentMetaKey> readablePatchKey;
            NN_RESULT_DO(GetLaunchableContentMetaKey(&readablePatchKey, patchKey, pIntegrated, ncm::StorageId::Any));

            *outValue = DecideControlContentMetaKey(readableAppKey, readablePatchKey);

            NN_RESULT_SUCCESS;
        }

        Result FindApplicationAndPatch(
            util::optional<ncm::ContentMetaKey>* outApplication,
            util::optional<ncm::ContentMetaKey>* outPatch,
            const ApplicationRecordAccessor& list) NN_NOEXCEPT
        {
            *outApplication = util::nullopt;
            *outPatch = util::nullopt;
            for (int i = 0; i < list.Count(); i++)
            {
                const auto& key = list[i].key;
                switch (key.type)
                {
                case ncm::ContentMetaType::Application:
                {
                    *outApplication = key;
                }
                break;
                case ncm::ContentMetaType::Patch:
                {
                    *outPatch = key;
                }
                break;
                default: break;
                }
            }

            NN_RESULT_SUCCESS;
        }

        Result MakeAutoUpdateInfo(AutoUpdateInfo* outValue, ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, ncm::ApplicationId id) NN_NOEXCEPT
        {
            util::optional<ncm::ContentMetaKey> mainMeta;
            NN_RESULT_DO(recordDb->FindLaunchableMain(&mainMeta, id));
            NN_RESULT_THROW_UNLESS(mainMeta, ResultMainApplicationNotFound());

            ncm::PatchId patchId;
            switch (mainMeta->type)
            {
            case ncm::ContentMetaType::Application:
                {
                    NN_RESULT_DO(integrated->GetPatchId(&patchId, *mainMeta));
                }
                break;
            case ncm::ContentMetaType::Patch:
                {
                    patchId.value = mainMeta->id;
                }
                break;
            default: NN_RESULT_THROW(ResultUnexpectedMainContent());
            }

            *outValue = { patchId, mainMeta->version };
            NN_RESULT_SUCCESS;
        }

        bool operator==(const ncm::StorageContentMetaKey& l, const ncm::ContentMetaKey& r) NN_NOEXCEPT
        {
            return l.key == r;
        }

        template <class RawType, class FlagType>
        void SetFlag(RawType* pRefFlags, FlagType flag, bool isOn) NN_NOEXCEPT
        {
            const RawType rawFlag = static_cast<RawType>(flag);
            if (isOn)
            {
                *pRefFlags |= rawFlag;
            }
            else
            {
                *pRefFlags &= ~rawFlag;
            }
        }

        template <class RawType>
        bool AreSet(RawType flags, RawType target) NN_NOEXCEPT
        {
            return (flags & target) == target;
        }

        template <class RawType, class FlagType>
        bool IsSet(RawType flags, FlagType flag) NN_NOEXCEPT
        {
            return AreSet<RawType>(flags, static_cast<RawType>(flag));
        }

        Result ConvertTerminateResult(Result result) NN_NOEXCEPT
        {
            // 想定された Result だけを返す
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(fs::ResultSdCardAccessFailed) { NN_RESULT_RETHROW; }
                NN_RESULT_CATCH(fs::ResultGameCardAccessFailed) { NN_RESULT_RETHROW; }
                NN_RESULT_CATCH(fs::ResultMmcAccessFailed) { NN_RESULT_RETHROW; }
                NN_RESULT_CATCH(fs::ResultDataCorrupted) { NN_RESULT_RETHROW; }
                NN_RESULT_CATCH(ns::ResultApplicationLogoCorrupted) { NN_RESULT_THROW(fs::ResultDataCorrupted()); }
                NN_RESULT_CATCH_ALL { NN_RESULT_SUCCESS; }
            NN_RESULT_END_TRY
            NN_RESULT_SUCCESS;
        }
    }

    namespace detail
    {
        Result ApplicationRecordIndex::Push(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT
        {
            if (Update(id, event).IsSuccess())
            {
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_THROW_UNLESS(count < static_cast<int>(sizeof(list) / sizeof(ApplicationRecord)), ResultOutOfMaxApplicationRecord());

            std::memmove(&list[1], &list[0], sizeof(list[0]) * count);
            list[0] = {};
            list[0].id = id;
            list[0].lastEvent = event;
            list[0].lastUpdated = GenerateEventCount();
            count++;

            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::Update(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));
            auto previous = list[index];

            std::memmove(&list[1], &list[0], sizeof(list[0]) * index);
            list[0] = previous;
            list[0].lastEvent = event;
            list[0].lastUpdated = GenerateEventCount();

            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::UpdateWithoutReorder(ncm::ApplicationId id, ApplicationEvent event) NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));
            list[index].lastEvent = event;
            list[index].lastUpdated = GenerateEventCount();

            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::Delete(ncm::ApplicationId id) NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));

            std::memmove(&list[index], &list[index + 1], sizeof(list[0]) * (count - (index + 1)));
            count--;

            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::Get(ApplicationRecord* outValue, ncm::ApplicationId id) const NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));

            *outValue = list[index];
            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::SetAttribute(ncm::ApplicationId id, ApplicationRecordAttribute attribute, bool isOn) NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));

            SetApplicationRecordAttribute(&list[index], attribute, isOn);
            NN_RESULT_SUCCESS;
        }

        Result ApplicationRecordIndex::GetAttribute(bool* outValue, ncm::ApplicationId id, ApplicationRecordAttribute attribute) const NN_NOEXCEPT
        {
            int index;
            NN_RESULT_DO(FindIndex(&index, id));

            *outValue = GetApplicationRecordAttribute(list[index], attribute);
            NN_RESULT_SUCCESS;
        }

        bool ApplicationRecordIndex::Has(ncm::ApplicationId id) const NN_NOEXCEPT
        {
            int index;
            return FindIndex(&index, id).IsSuccess();
        }

        Result ApplicationRecordIndex::FindIndex(int* outValue, ncm::ApplicationId id) const NN_NOEXCEPT
        {
            for (int i = 0; i < count; i++)
            {
                if (list[i].id.value == id.value)
                {
                    *outValue = i;
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_THROW(ResultApplicationRecordNotFound());
        }

        Result ApplicationRecordData::Push(const ncm::StorageContentMetaKey& key, bool allowOverwrite) NN_NOEXCEPT
        {
            for (int i = 0; i < entryCount; i++)
            {
                if (list[i] == key)
                {
                    NN_RESULT_SUCCESS;
                }

                if (list[i].key.id == key.key.id)
                {
                    if (allowOverwrite)
                    {
                        list[i] = key;
                    }
                    NN_RESULT_SUCCESS;
                }
            }
            NN_RESULT_THROW_UNLESS(entryCount < static_cast<int>(sizeof(list) / sizeof(ncm::ContentMetaKey)), ResultOutOfMaxContentMetaKey());

            list[entryCount] = key;
            entryCount++;

            NN_RESULT_SUCCESS;
        }

        bool ApplicationRecordData::HasAll(const ncm::ContentMetaKey keyList[], int count) const NN_NOEXCEPT
        {
            for (int i = 0; i < count; i++)
            {
                if (!Has(keyList[i]))
                {
                    return false;
                }
            }

            return true;
        }

        bool ApplicationRecordData::Has(const ncm::ContentMetaKey& key) const NN_NOEXCEPT
        {
            for (int i = 0; i < entryCount; i++)
            {
                if (list[i].key == key)
                {
                    return true;
                }
            }

            return false;
        }

        util::optional<ncm::StorageContentMetaKey> ApplicationRecordData::GetById(Bit64 id) const NN_NOEXCEPT
        {
            for (int i = 0; i < entryCount; i++)
            {
                if (list[i].key.id == id)
                {
                    return list[i];
                }
            }

            return util::nullopt;
        }

        template <class T>
        void ApplicationRecordData::Delete(const T& key) NN_NOEXCEPT
        {
            // 後ろから消していく
            for (int i = entryCount - 1; i >= 0; i--)
            {
                if (list[i] == key)
                {
                    if (i < entryCount - 1)
                    {
                        std::memmove(&list[i], &list[i + 1], sizeof(list[0]) * (entryCount - (i + 1)));
                    }
                    entryCount--;
                }
            }
        }
    }

    Result ApplicationRecordDatabase::Initialize(fs::SystemSaveDataId systemSaveDataId, IntegratedContentManager* pIntegrated) NN_NOEXCEPT
    {
        m_pIntegrated = pIntegrated;

        fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(MountName, systemSaveDataId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(systemSaveDataId, ApplicationRecordSaveDataSize, ApplicationRecordSaveDataJournalSize, ApplicationRecordSaveDataFlag));
                NN_RESULT_DO(fs::MountSystemSaveData(MountName, systemSaveDataId));
            }
        NN_RESULT_END_TRY

        SystemSaveDataSignature signature("ARDB", 0);

        bool hasValidSignature;
        NN_RESULT_DO(signature.IsValidOn(&hasValidSignature, MountName));
        if (!hasValidSignature)
        {
            NN_DETAIL_NS_TRACE("[ApplicationRecordDatabase] No valid signature. Create.\n");
            // TODO: セーブデータの作り直し

            NN_RESULT_DO(CreateIndex());
            NN_RESULT_DO(CreateKvs());
            NN_RESULT_DO(signature.WriteTo(MountName));
            NN_RESULT_DO(Commit());
        }

        NN_RESULT_DO(LoadIndex());
        NN_RESULT_DO(LoadKvs());

        NN_RESULT_SUCCESS;
    }

    void ApplicationRecordDatabase::Finalize() NN_NOEXCEPT
    {
        fs::Unmount(MountName);
    }

    Result ApplicationRecordDatabase::Push(ncm::ApplicationId id, ApplicationEvent event, const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite, bool skipNotify) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(m_Index.Has(id) ? Open(&accessor, id) : CreateAndOpen(&accessor, id));
        NN_RESULT_DO(accessor.Push(keyList, count, allowOverwrite));
        NN_RESULT_DO(WriteData(accessor.GetData(), id));

        NN_RESULT_DO(m_Index.Push(id, event));
        NN_RESULT_DO(FlushIndex());

        NN_RESULT_DO(Commit());

        if (!skipNotify)
        {
            m_SystemEvent.Signal();
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::UpdateEvent(ncm::ApplicationId id, ApplicationEvent event, bool doReorder, bool skipNotify) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        if (doReorder)
        {
            NN_RESULT_DO(m_Index.Update(id, event));
        }
        else
        {
            NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, event));
        }
        NN_RESULT_DO(FlushIndex());

        NN_RESULT_DO(Commit());

        if (!skipNotify)
        {
            m_SystemEvent.Signal();
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::UpdateKey(ncm::ApplicationId id, ApplicationEvent event, const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));
        NN_RESULT_DO(accessor.Push(keyList, count, allowOverwrite));
        NN_RESULT_DO(WriteData(accessor.GetData(), id));

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, event));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::Delete(ncm::ApplicationId id, bool skipNotify) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_THROW_UNLESS(m_Index.Has(id), ResultApplicationRecordNotFound());

        NN_RESULT_DO(m_Kvs.Delete(&id, sizeof(id)));
        NN_RESULT_DO(m_Index.Delete(id));

        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(Commit());

        if (!skipNotify)
        {
            m_SystemEvent.Signal();
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::Get(ApplicationRecord* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        return m_Index.Get(outValue, id);
    }

    Result ApplicationRecordDatabase::FindMain(util::optional<ncm::ContentMetaKey>* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(Open(&list, id));

        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;
        NN_RESULT_DO(FindApplicationAndPatch(&appKey, &patchKey, list));

        *outValue = DecideMainContentMetaKey(appKey, patchKey);
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::FindLaunchableMain(
            util::optional<ncm::ContentMetaKey>* outValue,
            ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(Open(&list, id));

        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;
        NN_RESULT_DO(FindApplicationAndPatch(&appKey, &patchKey, list));

        NN_RESULT_DO(DecideLaunchableMainContentMetaKey(outValue, appKey, patchKey, m_pIntegrated, ncm::StorageId::Any, ncm::StorageId::Any));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::FindControl(util::optional<ncm::ContentMetaKey>* outCache, util::optional<ncm::ContentMetaKey>* outStorage, ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(Open(&list, id));

        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;
        NN_RESULT_DO(FindApplicationAndPatch(&appKey, &patchKey, list));

        *outCache = DecideControlContentMetaKey(appKey, patchKey);
        NN_RESULT_DO(DecideReadableControlContentMetaKey(outStorage, appKey, patchKey, m_pIntegrated));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::FindLaunchableApplicationAndPatch(
        util::optional<ncm::ContentMetaKey>* outApplication,
        util::optional<ncm::ContentMetaKey>* outPatch,
        ncm::ApplicationId id,
        ncm::StorageId appStorageId,
        ncm::StorageId patchStorageId) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(Open(&list, id));

        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;
        NN_RESULT_DO(FindApplicationAndPatch(&appKey, &patchKey, list));

        NN_RESULT_DO(GetLaunchableContentMetaKey(outApplication, appKey, m_pIntegrated, appStorageId));
        NN_RESULT_DO(GetLaunchableContentMetaKey(outPatch, patchKey, m_pIntegrated, patchStorageId));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::EnableAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::AttributeUpdated));
        NN_RESULT_DO(m_Index.SetAttribute(id, ApplicationRecordAttribute::AutoDeleteDisabled, false));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::DisableAutoDelete(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::AttributeUpdated));
        NN_RESULT_DO(m_Index.SetAttribute(id, ApplicationRecordAttribute::AutoDeleteDisabled, true));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::IsAutoDeleteDisabled(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        return m_Index.GetAttribute(outValue, id, ApplicationRecordAttribute::AutoDeleteDisabled);
    }

    Result ApplicationRecordDatabase::SetApplicationTerminateResult(ncm::ApplicationId id, Result result) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.SetTerminateResult(result);
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::GetApplicationTerminateResult(Result* outValue, ncm::ApplicationId id, bool needsRaw) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(ReadData(&m_Data, id));

        auto rawResult = m_Data.property.GetTerminateResult();
        *outValue = needsRaw ? rawResult : ConvertTerminateResult(rawResult);
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::ClearApplicationTerminateResult(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.SetTerminateResult(ResultSuccess());
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::EnableAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT
    {
        AutoUpdateInfo info;
        NN_RESULT_DO(MakeAutoUpdateInfo(&info, this, m_pIntegrated, id));
        NN_RESULT_DO(EnableAutoUpdate(id, info));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::EnableAutoUpdate(ncm::ApplicationId id, const AutoUpdateInfo& info) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::AttributeUpdated));
        NN_RESULT_DO(m_Index.SetAttribute(id, ApplicationRecordAttribute::AutoUpdateEnabled, true));
        NN_RESULT_DO(FlushIndex());

        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.autoUpdateInfo = info;
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::IsAutoUpdateEnabled(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        return m_Index.GetAttribute(outValue, id, ApplicationRecordAttribute::AutoUpdateEnabled);
    }

    Result ApplicationRecordDatabase::GetAutoUpdateInfo(AutoUpdateInfo* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(ReadData(&m_Data, id));

        *outValue = m_Data.property.autoUpdateInfo;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::DisableAutoUpdate(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::AttributeUpdated));
        NN_RESULT_DO(m_Index.SetAttribute(id, ApplicationRecordAttribute::AutoUpdateEnabled, false));
        NN_RESULT_DO(FlushIndex());

        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.autoUpdateInfo = {};
        NN_RESULT_DO(WriteData(m_Data, id));

        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    int ApplicationRecordDatabase::List(ApplicationRecord outList[], int count, int offset) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        if (offset < 0)
        {
            return 0;
        }

        auto readCount = std::min(count, m_Index.count - offset);
        for (int i = 0; i < readCount; i++)
        {
            outList[i] = m_Index.list[offset + i];
        }

        return readCount;
    }

    int ApplicationRecordDatabase::ListSpecificAttribute(ApplicationRecord outList[], int count, ApplicationRecordAttribute attribute) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        int readCount = 0;
        for (int i = 0; i < m_Index.count && readCount < count; i++)
        {
            if (GetApplicationRecordAttribute(m_Index.list[i], attribute))
            {
                outList[readCount] = m_Index.list[i];
                readCount++;
            }
        }

        return readCount;
    }

    bool ApplicationRecordDatabase::Has(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        return m_Index.Has(id);
    }
    Result ApplicationRecordDatabase::HasContentMetaKey(bool* outHasRecord, bool* outInstalled, ncm::ContentMetaKey key, ncm::ApplicationId id) NN_NOEXCEPT
    {
        // implemented in ns_ApplicationContentMetaStatusUtil.h
        NN_RESULT_DO(CheckRecordAndDatabaseImpl(outHasRecord, outInstalled, key, id, *this, *m_pIntegrated));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::RequestApplicationUpdate(ncm::ApplicationId id, Result result) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.requestUpdateInfo.needsUpdate = true;
        m_Data.property.requestUpdateInfo.SetReasonResult(result);
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::IsApplicationUpdateRequested(util::optional<Result>* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(ReadData(&m_Data, id));
        *outValue = m_Data.property.requestUpdateInfo.needsUpdate
            ? util::optional<Result>(m_Data.property.requestUpdateInfo.GetReasonResult())
            : util::nullopt;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::WithdrawApplicationUpdateRequest(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.requestUpdateInfo.needsUpdate = false;
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::RecommendCleanupAddOnContentsWithNoRights(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.needsCleanupAoc = true;
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::WithdrawCleanupAddOnContentsWithNoRightsRecommendation(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        m_Data.property.needsCleanupAoc = false;
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    int64_t ApplicationRecordDatabase::GenerateCount() NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        return m_Index.GenerateEventCount();
    }

    Result ApplicationRecordDatabase::CreateAndOpen(ApplicationRecordAccessor* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        NN_RESULT_THROW_UNLESS(!m_IsDataOccupied, ResultApplicationRecordDataOccupied());

        m_Data.Clear();

        outValue->Initialize(id, &m_Data, &m_DataView, m_GameCardDb ? &(*m_GameCardDb) : nullptr, this);
        m_IsDataOccupied = true;
        m_Mutex.Lock();

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::CountContentMeta(int* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));

        *outValue = accessor.Count();
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::ListContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultNotImplemented());

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));

        auto readCount = std::min(count, accessor.Count() - offset);
        for (int i = 0; i < readCount; i++)
        {
            outList[i] = accessor[offset + i];
        }

        *outCount = readCount;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::ListInstalledContentMeta(int* outCount, ncm::StorageContentMetaKey outList[], int count, ncm::ApplicationId id, int offset) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultNotImplemented());

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));

        auto& installedData = accessor.GetData();
        auto readCount = std::min(count, installedData.entryCount - offset);
        for (int i = 0; i < readCount; i++)
        {
            outList[i] = installedData.list[offset + i];
        }

        *outCount = readCount;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::Open(ApplicationRecordAccessor* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        NN_RESULT_THROW_UNLESS(!m_IsDataOccupied, ResultApplicationRecordDataOccupied());

        NN_RESULT_THROW_UNLESS(m_Index.Has(id), ResultApplicationRecordNotFound());
        size_t size;
        NN_RESULT_DO(m_Kvs.Get(&size, &m_Data, sizeof(m_Data), &id, sizeof(id)));

        outValue->Initialize(id, &m_Data, &m_DataView, m_GameCardDb ? &(*m_GameCardDb) : nullptr, this);
        m_IsDataOccupied = true;
        m_Mutex.Lock();

        NN_RESULT_SUCCESS;
    }

    void ApplicationRecordDatabase::Close(ApplicationRecordAccessor*) NN_NOEXCEPT
    {
        m_IsDataOccupied = false;
        m_Mutex.Unlock();
    }

    Result ApplicationRecordDatabase::GetProperty(ApplicationRecordProperty* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(ReadData(&m_Data, id));

        *outValue = m_Data.property;
        NN_RESULT_SUCCESS;
    }

    void ApplicationRecordDatabase::RegisterGameCardDatabase(ncm::ContentMetaDatabase&& db) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        m_GameCardDb = std::move(db);
    }

    void ApplicationRecordDatabase::UnregisterGameCardDatabase() NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);
        m_GameCardDb = util::nullopt;
    }

    Result ApplicationRecordDatabase::CleanupRedundant() NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        // インデックスの反対から削除していけばインデックスの再読み出しが不要な前提
        for (int i = m_Index.count; i > 0; i--)
        {
            auto appId = m_Index.list[i - 1].id;
            int count;
            NN_RESULT_DO(CountContentMeta(&count, appId));
            if (count == 0)
            {
                NN_RESULT_DO(Delete(appId));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::CreateIndex() NN_NOEXCEPT
    {
        fs::DeleteFile(MakeApplicationRecordIndexPath());
        NN_RESULT_DO(fs::CreateFile(MakeApplicationRecordIndexPath(), sizeof(m_Index)));

        m_Index.Clear();
        NN_RESULT_DO(FlushIndex());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::LoadIndex() NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeApplicationRecordIndexPath(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        return fs::ReadFile(file, 0, &m_Index, sizeof(m_Index));
    }

    Result ApplicationRecordDatabase::FlushIndex() NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeApplicationRecordIndexPath(), fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        return fs::WriteFile(file, 0, &m_Index, sizeof(m_Index), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush));
    }

    Result ApplicationRecordDatabase::CreateKvs() NN_NOEXCEPT
    {
        fs::DeleteDirectoryRecursively(MakeKvsDirectoryPath());
        return fs::CreateDirectory(MakeKvsDirectoryPath());
    }

    Result ApplicationRecordDatabase::LoadKvs() NN_NOEXCEPT
    {
        return m_Kvs.Initialize(MakeKvsDirectoryPath(), m_KvsCacheBuffer, sizeof(m_KvsCacheBuffer), 256);
    }

    Result ApplicationRecordDatabase::ReadData(detail::ApplicationRecordData* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        size_t size;
        return m_Kvs.Get(&size, outValue, sizeof(*outValue), &id, sizeof(id));
    }

    Result ApplicationRecordDatabase::WriteData(const detail::ApplicationRecordData& data, ncm::ApplicationId id) NN_NOEXCEPT
    {
        return m_Kvs.Put(&id, sizeof(id), &data, data.CalculateSerializeSize());
    }

    Result ApplicationRecordDatabase::Commit() NN_NOEXCEPT
    {
        return fs::CommitSaveData(MountName);
    }

    Result ApplicationRecordDatabase::DeleteKey(ncm::ApplicationId id, const ncm::StorageContentMetaKey& key) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));
        accessor.Delete(key);
        NN_RESULT_DO(WriteData(accessor.GetData(), id));
        NN_RESULT_DO(Commit());

        accessor.RefreshView();
        // TORIAEZU: m_SystemEvent のシグナルは不要とする

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::DeleteKey(ncm::ApplicationId id, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(Open(&accessor, id));
        accessor.Delete(key);
        NN_RESULT_DO(WriteData(accessor.GetData(), id));
        NN_RESULT_DO(Commit());

        accessor.RefreshView();
        // TORIAEZU: m_SystemEvent のシグナルは不要とする

        NN_RESULT_SUCCESS;
    }

    void ApplicationRecordAccessor::Initialize(
        ncm::ApplicationId id,
        detail::ApplicationRecordData* data,
        detail::ApplicationRecordData* dataView,
        ncm::ContentMetaDatabase* gameCardDb,
        ApplicationRecordDatabase* db) NN_NOEXCEPT
    {
        m_Id = id;
        m_Data = data;
        m_DataView = dataView;
        m_GameCardDb = gameCardDb;
        m_Db = db;

        RefreshView();
    }

    void ApplicationRecordAccessor::RefreshView() NN_NOEXCEPT
    {
        *m_DataView = *m_Data;
        if (m_GameCardDb)
        {
            static ncm::ContentMetaKey s_List[2048];
            auto count = m_GameCardDb->ListContentMeta(s_List, 2048, ncm::ContentMetaType::Unknown, m_Id);
            for (int i = 0; i < count.listed; i++)
            {
                // 一時記録のエントリはバージョンの高いエントリを上書きしない
                auto existsKey = m_Data->GetById(s_List[i].id);
                if (existsKey && existsKey->key.version > s_List[i].version)
                {
                    continue;
                }

                ncm::StorageContentMetaKey key = { s_List[i], ncm::StorageId::Card };
                m_DataView->Push(key, true);
            }
        }
    }

    Result ApplicationRecordAccessor::Push(const ncm::StorageContentMetaKey keyList[], int count, bool allowOverwrite) NN_NOEXCEPT
    {
        for (int i = 0; i < count; i++)
        {
            auto& key = keyList[i];

            if (key.storageId == ncm::StorageId::Card)
            {
                // ゲームカードの本体以外のエントリは実記録に追加しない
                if (key.key.type != ncm::ContentMetaType::Application)
                {
                    continue;
                }

                // ゲームカードのエントリはインストールされたエントリを上書きしない
                auto installed = m_Data->GetById(key.key.id);
                if (installed && installed->storageId != ncm::StorageId::Card)
                {
                    continue;
                }
            }
            NN_RESULT_DO(m_Data->Push(key, allowOverwrite));
        }
        RefreshView();

        NN_RESULT_SUCCESS;
    }

    void ApplicationRecordAccessor::Delete(const ncm::StorageContentMetaKey& key) NN_NOEXCEPT
    {
        m_Data->Delete(key);
        RefreshView();
    }

    void ApplicationRecordAccessor::Delete(const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        m_Data->Delete(key);
        RefreshView();
    }

    util::optional<ncm::ContentMetaKey> DecideMainContentMetaKey(
        util::optional<ncm::ContentMetaKey> appKey,
        util::optional<ncm::ContentMetaKey> patchKey) NN_NOEXCEPT
    {
        if (!appKey)
        {
            return util::nullopt;
        }
        return DecideContentMetaKeyImpl(appKey, patchKey);
    }

    Result ApplicationRecordDatabase::SetPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        SetFlag(&m_Data.property.preInstallFlag, ApplicationRecordProperty::PreInstallFlag::IsPreInstalledApplication, true);
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::ClearPreInstalledApplicationFlag(ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
        NN_RESULT_DO(FlushIndex());
        NN_RESULT_DO(ReadData(&m_Data, id));
        SetFlag(&m_Data.property.preInstallFlag, ApplicationRecordProperty::PreInstallFlag::IsPreInstalledApplication, false);
        NN_RESULT_DO(WriteData(m_Data, id));
        NN_RESULT_DO(Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::ClearAllPreInstalledApplicationFlag() NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        bool needsCommit = false;
        for (auto& record : util::MakeSpan(m_Index.list, m_Index.list + m_Index.count))
        {
            const auto& id = record.id;
            NN_RESULT_DO(ReadData(&m_Data, id));
            if (IsSet(m_Data.property.preInstallFlag, ApplicationRecordProperty::PreInstallFlag::IsPreInstalledApplication))
            {
                needsCommit = true;
                NN_RESULT_DO(m_Index.UpdateWithoutReorder(id, ApplicationEvent::PropertyUpdated));
                SetFlag(&m_Data.property.preInstallFlag, ApplicationRecordProperty::PreInstallFlag::IsPreInstalledApplication, false);
                NN_RESULT_DO(WriteData(m_Data, id));
            }
        }
        if (needsCommit)
        {
            NN_RESULT_DO(FlushIndex());
            NN_RESULT_DO(Commit());
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationRecordDatabase::IsPreInstalledApplication(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<RecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(ReadData(&m_Data, id));
        *outValue = IsSet(m_Data.property.preInstallFlag, ApplicationRecordProperty::PreInstallFlag::IsPreInstalledApplication);

        NN_RESULT_SUCCESS;
    }


}}}
