﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_ApplicationLaunchPropertyManager.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ns { namespace srv {
namespace {
    constexpr ApplicationLaunchInfo InvalidApplicationLaunchInfo = {};
}

ApplicationLaunchPropertyManager::ApplicationLaunchPropertyManager() NN_NOEXCEPT
    : m_SequenceId(1)
{
    Clear();
}

Result ApplicationLaunchPropertyManager::Update(const ApplicationLaunchInfo& info, uint8_t programIndex, os::ProcessId processId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(info.IsValid(), ResultInvalidApplicationLaunchInfo());
    std::lock_guard<NonRecursiveMutex> lock(m_Mutex);
    // Acquire を呼んでこない場合もあるため、とりあえず、上書き可能にするためにコメントアウト
    //NN_RESULT_THROW_UNLESS(!m_CanAcquire, ResultApplicationIsRunning());
    m_Info = info;
    m_Info._infoId = m_SequenceId++;
    m_ProgramIndex = programIndex;
    m_ProcessId = processId;
    m_CanAcquire = true;

    NN_RESULT_SUCCESS;
}

Result ApplicationLaunchPropertyManager::Acquire(ApplicationLaunchInfo* outInfo, os::ProcessId processId) NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> lock(m_Mutex);
    NN_RESULT_THROW_UNLESS(m_CanAcquire, ResultApplicationLaunchInfoNotFound());
    NN_RESULT_THROW_UNLESS(m_ProcessId == processId, ResultApplicationLaunchInfoNotFound());
    *outInfo = m_Info;
    m_CanAcquire = false;
    NN_RESULT_SUCCESS;
}

Bit64 ApplicationLaunchPropertyManager::GenerateApplicationLaunchInfoId() NN_NOEXCEPT
{
    return m_SequenceId++;
}

Result ApplicationLaunchPropertyManager::GetProgramIndex(uint8_t* outValue, const ApplicationLaunchInfo& info) const NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(info._infoId == m_Info._infoId, ResultApplicationLaunchInfoNotFound());
    *outValue = m_ProgramIndex;

    NN_RESULT_SUCCESS;
}

void ApplicationLaunchPropertyManager::Clear() NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> lock(m_Mutex);

    m_Info = InvalidApplicationLaunchInfo;
    m_ProgramIndex = 0;
    m_ProcessId = os::ProcessId::GetInvalidId();
    m_CanAcquire = false;
}
}}}
