﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Config.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ncm/ncm_StorageUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/nim/nim_Result.h>
#include "ns_InstallUtil.h"
#include "ns_SystemUpdateUtil.h"
#include "ns_TaskUtil.h"
#include "ns_ApplicationInstallTask.h"

namespace nn { namespace ns { namespace srv {

    namespace {
        TimeSpan CalculateRemainingTime(const nim::NetworkInstallTaskInfo& info) NN_NOEXCEPT
        {
            int64_t downloadPerMillisecond = CalculateThroughput(info.downloaded, info.elapsedTime);
            int64_t remainingSize = info.progress.totalSize - info.progress.installedSize;

            // 以下は計測中とする
            // - Running でない
            // - 計測を開始してから 1 秒経過していない
            // - 毎秒ダウンロード容量が 0 以下
            // - 残りダウンロード容量が 0 以下
            if (!info.isRunning ||
                downloadPerMillisecond <= 0 ||
                remainingSize <= 0)
            {
                return 0;
            }

            return TimeSpan::FromMilliSeconds(remainingSize / downloadPerMillisecond);
        }

    }

    ApplicationInstallTask::ApplicationInstallTask(ncm::ApplicationId appId) NN_NOEXCEPT
    {
        m_IsValid = nim::ListApplicationNetworkInstallTask(&m_Id, 1, appId) > 0;
    }

    Result ApplicationInstallTask::GetProgress(ApplicationDownloadProgress* outValue) NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));

        ApplicationDownloadProgress progress = {};
        progress.downloaded = info.progress.installedSize;
        progress.total = info.progress.totalSize;
        progress.state = ToState(info.attribute);
        progress.lastResult = util::Get(info.progress.lastResult);
        progress.isRunning = info.isRunning;
        progress.remainingTime = CalculateRemainingTime(info);

        *outValue = progress;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::IsDownloaded(bool* outValue) NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));
        if (info.progress.state == ncm::InstallProgressState::Downloaded)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::IsCommittable(bool* outValue, bool checkRequireSystemVersion) NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));
        if (info.progress.state != ncm::InstallProgressState::Downloaded)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (info.attribute == static_cast<Bit64>(ApplicationDownloadState::SystemUpdateRequired))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (checkRequireSystemVersion)
        {
            // ダウンロードが完了している状態なので、RequiredSystemVersion をチェックする
            // このチェックは、ダウンロード済みの ContentMeta を都度読み込むため、そこそこ負荷がかかる
            uint32_t requiredSystemVersion;
            NN_RESULT_DO(nim::FindMaxRequiredSystemVersionOfTask(&requiredSystemVersion, m_Id));
            uint32_t systemVersion = GetSystemUpdateVersion();
            NN_DETAIL_NS_TRACE("[ApplicationInstallTask] %u vs. %u\n", requiredSystemVersion, systemVersion);
            if (requiredSystemVersion > systemVersion)
            {
                NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(m_Id, static_cast<Bit64>(ApplicationDownloadState::SystemUpdateRequired)));
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::Commit() NN_NOEXCEPT
    {
        {
            auto result = nim::CommitNetworkInstallTask(m_Id);
            if (result.IsFailure())
            {
                NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(m_Id, static_cast<Bit64>(ApplicationDownloadState::Fatal)));
                NN_RESULT_THROW(result);
            }
        }
        nim::ApplyDeltaTaskId taskId;
        NN_RESULT_TRY(nim::CreateApplyDeltaTaskFromDownloadTask(&taskId, m_Id))
            // 適用タスクの作成が不要だったら何もしない
            NN_RESULT_CATCH(nim::ResultApplyDeltaNotNeeded) {}
            NN_RESULT_CATCH_ALL
            {
                // それ以外で失敗したら、delta を削除しておく
                // 現状 Commit のエラー = CommitNetworkInstallTask のエラーとしたいので、
                // こちらの Delta 適用タスクのエラーは返らないようにしている
                auto result = CleanupFragments();
                if (result.IsFailure())
                {
                    NN_DETAIL_NS_TRACE("[ApplicationInstallTask] Failed to cleanup delta 0x%08x\n", result.GetInnerValueForDebug());
                }
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::CommitPartially(const ncm::StorageContentMetaKey* keys, int count) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nim::CommitNetworkInstallTaskPartially(keys, count, m_Id))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(m_Id, static_cast<Bit64>(ApplicationDownloadState::Fatal)));
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::ListKey(int* outCount, ncm::StorageContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
    {
        return nim::ListNetworkInstallTaskContentMeta(outCount, outList, count, offset, m_Id);
    }

    Result ApplicationInstallTask::ListCommittedKey(int* outCount, ncm::StorageContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
    {
        return nim::ListNetworkInstallTaskCommittedContentMeta(outCount, outList, count, offset, m_Id);
    }

    Result ApplicationInstallTask::ListNotCommittedKey(int* outCount, ncm::StorageContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
    {
        return nim::ListNetworkInstallTaskNotCommittedContentMeta(outCount, outList, count, offset, m_Id);
    }

    Result ApplicationInstallTask::ListKeyFromInstallMeta(int* outCount, ncm::ContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
    {
        return nim::ListNetworkInstallTaskContentMetaFromInstallMeta(outCount, outList, count, offset, m_Id);
    }

    Result ApplicationInstallTask::NeedsCleanup(bool* outValue) NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));
        if (info.progress.state == ncm::InstallProgressState::Commited)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::Destroy() NN_NOEXCEPT
    {

        return nim::DestroyNetworkInstallTask(m_Id);
    }

    Result ApplicationInstallTask::Resume() NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));
        NN_RESULT_THROW_UNLESS(IsResumable(info.attribute), ResultApplicationDownloadNotResumable());
        NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(m_Id, ToAttribute(ApplicationDownloadState::Runnable)));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT
    {
        return nim::CalculateNetworkInstallTaskRequiredSize(outValue, m_Id);
    }
    Result ApplicationInstallTask::GetRequiredStorage(ncm::StorageId* outValue) NN_NOEXCEPT
    {
        *outValue = ncm::StorageId::Any;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::ResumeAll() NN_NOEXCEPT
    {
        // 16byte * 64 = 1KiByte
        nim::NetworkInstallTaskId ids[nim::MaxNetworkInstallTaskCount];
        auto count = nim::ListNetworkInstallTask(ids, nim::MaxNetworkInstallTaskCount);

        for (int i = 0; i < count; ++i)
        {
            nim::NetworkInstallTaskInfo info;
            NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, ids[i]));
            if (IsResumable(info.attribute))
            {
                NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(ids[i], ToAttribute(ApplicationDownloadState::Runnable)));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::CleanupFragments() NN_NOEXCEPT
    {
        // エラー時の処理なので都度 ContentStorage を開くような非効率なことを行っている
        const int CountStorageKeys = 16;
        ncm::StorageContentMetaKey skeys[CountStorageKeys];
        int offset = 0;
        for(;;)
        {
            int count;
            NN_RESULT_DO(nim::ListNetworkInstallTaskContentMeta(&count, skeys, CountStorageKeys, offset, m_Id));
            for (int i = 0; i < count; ++i)
            {
                if (skeys[i].key.type == ncm::ContentMetaType::Patch && skeys[i].key.installType == ncm::ContentInstallType::FragmentOnly)
                {
                    ncm::ContentMetaDatabase db;
                    ncm::ContentStorage storage;
                    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, skeys[i].storageId));
                    NN_RESULT_DO(ncm::OpenContentStorage(&storage, skeys[i].storageId));

                    ncm::ContentManagerAccessor accessor(&db, &storage);
                    NN_RESULT_DO(accessor.DeleteRedundant(skeys[i].key, nullptr));
                }
            }
            offset += count;
            if (count != CountStorageKeys)
            {
                break;
            }
        }
        NN_RESULT_SUCCESS;
    }
    Result ApplicationInstallTask::CalculateContentsSize(int64_t* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
    {
        return nim::CalculateNetworkInstallTaskContentsSize(outValue, m_Id, key, storageId);
    }

    Result ApplicationInstallTask::ListOccupiedSize(int* outCount, ncm::InstallTaskOccupiedSize* outList, int numList, int offset) NN_NOEXCEPT
    {
        return nim::ListNetworkInstallTaskOccupiedSize(outCount, outList, numList, offset, m_Id);
    }

    Result ApplicationInstallTask::GetThroughput(int64_t* outValue) NN_NOEXCEPT
    {
        nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, m_Id));
        *outValue = CalculateThroughput(info.downloaded, info.elapsedTime);

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::FindMaxRequiredApplicationVersion(uint32_t* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::FindMaxRequiredApplicationVersionOfTask(outValue, m_Id));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallTask::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nim::GetNetworkInstallTaskErrorContext(outValue, m_Id))
            NN_RESULT_CATCH_CONVERT(nim::ResultTaskStillRunning, ResultAlreadyDownloading())
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }
}}}
