﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_Api.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/ovln/format/ovln_DownloadMessage.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nim/nim_Config.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/ns_TicketSystemApi.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationInstallRequest.h>
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_ApplicationControlFileUtil.h"
#include "ns_InstallUtil.h"
#include "ns_LogUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        const int MaxRequiredTask = 8;
        Bit64 ToAttribute(Result result) NN_NOEXCEPT
        {
            if (result.IsSuccess()) return static_cast<Bit64>(ApplicationDownloadState::Finished);
            if (IsRetryable(result)) return static_cast<Bit64>(ApplicationDownloadState::Runnable);
            if (IsSuspendable(result)) return static_cast<Bit64>(ApplicationDownloadState::Suspended);
            if (IsNotEnoughSpace(result))  return static_cast<Bit64>(ApplicationDownloadState::NotEnoughSpace);
            return static_cast<Bit64>(ApplicationDownloadState::Fatal);
        }


    template <typename ListType, typename IdType> bool HasRequest(const ListType* list, size_t n, IdType id) NN_NOEXCEPT
    {
        for (int i = 0; i < static_cast<int>(n); ++i)
        {
            const auto& request = list[i];
            if (request && request->first == id)
            {
                return true;
            }
        }
        return false;
    }
    template <typename ListType> bool IsFull(const ListType* list, size_t n) NN_NOEXCEPT
    {
        for (int i = 0; i < static_cast<int>(n); ++i)
        {
            const auto& request = list[i];
            if (!request)
            {
                return false;
            }
        }

        return true;
    }
    }

    bool ApplicationInstallRequestList::NeedsProcess(ProcessType type) NN_NOEXCEPT
    {
        int count;
        RequiredTaskAttribute attributes[MaxRequiredTask];
        GetRequiredTaskAttributes(&count, attributes, MaxRequiredTask);

        for (int i = 0; i < count; ++i)
        {
            if (type == ProcessType::All)
            {
                return true;
            }
            else if (type == ProcessType::Lightweight && !(attributes[i].isHeavy))
            {
                return true;
            }
            else if (type == ProcessType::Offline && attributes[i].canProcessOffline)
            {
                return true;
            }
        }
        return false;
    }

    Result ApplicationInstallRequestList::Prepare(ProcessType type) NN_NOEXCEPT
    {
        {
            nim::NetworkInstallTaskId idList[nim::MaxNetworkInstallTaskCount];
            auto count = nim::ListNetworkInstallTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < count; i++)
            {
                if (type == ProcessType::All)
                {
                    NN_RESULT_DO(PrepareApplicationInstallRequest(idList[i]));
                }
                if (type == ProcessType::All || type == ProcessType::Lightweight)
                {
                    NN_RESULT_DO(PrepareApplicationControlDataRequest(idList[i]));
                }
            }
        }
        if (type == ProcessType::All || type == ProcessType::Offline)
        {
            nim::ApplyDeltaTaskId idList[nim::MaxApplyDeltaTaskCount];
            auto count = nim::ListApplyDeltaTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < count; i++)
            {
                NN_RESULT_DO(PrepareApplyDeltaRequest(idList[i]));
            }
        }

        if (type == ProcessType::All || type == ProcessType::Lightweight)
        {
            NN_RESULT_DO(PrepareDownloadTaskListRequest());
            NN_RESULT_DO(PrepareVersionListRequest());
            NN_RESULT_DO(PrepareDownloadTicketRequest());
        }

        NN_RESULT_SUCCESS;
    }

    bool ApplicationInstallRequestList::HasPreparedRequest() NN_NOEXCEPT
    {
        for (auto& request : m_ApplyDeltaRequestList)
        {
            if (request)
            {
                return true;
            }
        }
        for (auto& request : m_ApplicationInstallRequestList)
        {
            if (request)
            {
                return true;
            }
        }
        if (m_ApplicationControlRequest)
        {
            return true;
        }
        if (m_DownloadTaskListRequest)
        {
            return true;
        }
        if (m_VersionListRequest)
        {
            return true;
        }
        if (m_DownloadTicketRequest)
        {
            return true;
        }
        return false;
    }

    bool ApplicationInstallRequestList::HandleDone() NN_NOEXCEPT
    {
        // 終わっているタスクの後処理をする。どれかでエラーが返ったら、false を返す
        bool allSuccess = true;
        for (auto& request : m_ApplyDeltaRequestList)
        {
            auto result = HandleApplyDeltaRequestDone(&request);
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] ApplyDeltaRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }
        for (auto& request : m_ApplicationInstallRequestList)
        {
            auto result = HandleApplicationInstallRequestDone(&request, true);
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] ApplicationInstallRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (m_ApplicationControlRequest && m_ApplicationControlRequest->second.TryWait())
        {
            NN_UTIL_SCOPE_EXIT{ m_ApplicationControlRequest = util::nullopt; };
            auto result = ReadApplicationControlDataRequest();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] ApplicationControlRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (m_DownloadTaskListRequest && m_DownloadTaskListRequest->TryWait())
        {
            auto result = HandleDownloadTaskListRequestDone();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] DownloadTaskListRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (m_VersionListRequest && m_VersionListRequest->TryWait())
        {
            auto result = HandleVersionListRequestDone();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] VersionListRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }

        if (m_DownloadTicketRequest && m_DownloadTicketRequest->second.TryWait())
        {
            auto result = HandleDownloadTicketRequestDone();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] DownloadTicketRequest failed as %08x\n", result.GetInnerValueForDebug());
            }
        }

        return allSuccess;
    }

    void ApplicationInstallRequestList::WaitAlongWith(os::SystemEvent** events, int numEvents) NN_NOEXCEPT
    {
        MultiWaitSystemEvent multiWait;
        int count = 0;
        for (auto& request : m_ApplyDeltaRequestList)
        {
            if (request && count < MaxRequiredTask)
            {
                multiWait.Link(&(request->second.GetEvent()));
                count++;
            }
        }
        for (auto& request : m_ApplicationInstallRequestList)
        {
            if (request && count < MaxRequiredTask)
            {
                multiWait.Link(&(request->second.GetEvent()));
                count++;
            }
        }
        if (m_ApplicationControlRequest && count < MaxRequiredTask)
        {
            multiWait.Link(&(m_ApplicationControlRequest->second.GetEvent()));
            count++;
        }
        if (m_DownloadTaskListRequest && count < MaxRequiredTask)
        {
            multiWait.Link(&(m_DownloadTaskListRequest->GetEvent()));
            count++;
        }
        if (m_VersionListRequest && count < MaxRequiredTask)
        {
            multiWait.Link(&(m_VersionListRequest->GetEvent()));
            count++;
        }
        if (m_DownloadTicketRequest && count < MaxRequiredTask)
        {
            multiWait.Link(&(m_DownloadTicketRequest->second.GetEvent()));
            count++;
        }
        for (int i = 0; i < numEvents; ++i)
        {
            multiWait.Link(events[i]);
        }
        auto signaled = multiWait.WaitAny();
        NN_UNUSED(signaled);
    }

    void ApplicationInstallRequestList::Cleanup() NN_NOEXCEPT
    {
        CleanupApplicationInstallRequest();
        CleanupApplyDeltaRequest();
        m_ApplicationControlRequest = util::nullopt;
        m_DownloadTaskListRequest = util::nullopt;
        m_VersionListRequest = util::nullopt;
        CleanupDownloadTicketRequest();
    }

    void ApplicationInstallRequestList::RequestDownloadTaskList(const nn::nim::ETag& eTag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT
    {
        if (eTag.IsEmpty() || m_LatestDownloadTaskListETag != eTag)
        {
            m_IsDownloadTaskListRequested = true;
            m_IsSystemUpdateNotificationRequiredByDownloadTaskList = notifiesRequiredSystemUpdate;
        }
    }

    void ApplicationInstallRequestList::RequestVersionList(const nn::nim::ETag& eTag) NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] RequestVersionList\n");

        auto isEmpty = eTag.IsEmpty();
        auto isUpdated = m_LatestVersionListETag != eTag;
        auto doRequest = isEmpty || isUpdated;
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList]   Empty  : %s\n", isEmpty ? "true" : "false");
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList]   Updated: %s\n", isUpdated ? "true" : "false");
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList]   => DoRequest: %s\n", doRequest ? "true" : "false");

        if (doRequest)
        {
            m_IsVersionListRequested = true;
        }
    }

    void ApplicationInstallRequestList::RequestDownloadTicket(ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] RequestDownloadTicket\n");

        for (auto& item : m_DownloadTicketApplicationIdList)
        {
            if (item != ncm::ApplicationId::GetInvalidId())
            {
                continue;
            }
            else
            {
                item = id;
                return;
            }
        }
    }

    Result ApplicationInstallRequestList::PrepareApplicationInstallRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT
    {
        if (!HasApplicationInstallRequest(id) && !IsFullOfApplicationInstallRequest())
        {
            nim::NetworkInstallTaskInfo info;
            NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, id));

            if (ToState(info.attribute) == ApplicationDownloadState::Runnable)
            {
                for (auto& request : m_ApplicationInstallRequestList)
                {
                    if (request)
                    {
                        continue;
                    }
                    else
                    {
                        request.emplace();
                        request->first = id;
                        auto result = nim::RequestNetworkInstallTaskRun(&request->second, id);
                        if (result.IsFailure())
                        {
                            request = nullptr;
                        }

                        NN_RESULT_SUCCESS;
                    }
                }
            }
        }

        NN_RESULT_SUCCESS;
    }
    Result ApplicationInstallRequestList::PrepareApplyDeltaRequest(nim::ApplyDeltaTaskId id) NN_NOEXCEPT
    {
        if (!HasApplyDeltaRequest(id) && !IsFullOfApplyDeltaRequest())
        {
            ApplicationApplyDeltaTask task(id);
            bool needsRun;
            auto result = task.NeedsRequestToRun(&needsRun);

            if (result.IsSuccess() && needsRun)
            {
                for (auto& request : m_ApplyDeltaRequestList)
                {
                    if (request)
                    {
                        continue;
                    }
                    else
                    {
                        request.emplace();
                        request->first = id;
                        result = nim::RequestApplyDeltaTaskRun(&request->second, id);
                        if (result.IsFailure())
                        {
                            request = nullptr;
                        }

                        NN_RESULT_SUCCESS;
                    }
                }
            }
        }

        NN_RESULT_SUCCESS;
    }


    Result ApplicationInstallRequestList::PrepareApplicationControlDataRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT
    {
        if (!m_ApplicationControlRequest)
        {
            nim::NetworkInstallTaskInfo info;
            NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, id));
            if (NeedsDownloadApplicationControl(info))
            {
                m_ApplicationControlRequest.emplace();
                m_ApplicationControlRequest->first = info.applicationId;
                auto result = nim::RequestLatestApplicationControl(&m_ApplicationControlRequest->second, info.applicationId);
                if (result.IsFailure())
                {
                    m_ApplicationControlRequest = util::nullopt;
                }
                NN_RESULT_DO(result);
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::ReadApplicationControlDataRequest() NN_NOEXCEPT
    {
        nim::ApplicationControlInfo info;
        NN_RESULT_DO(m_ApplicationControlRequest->second.Get(&info));
        // TORIAEZU: ProgramIndex == 0 のアプリケーション管理データを落としてくる想定
        NN_RESULT_DO(m_ControlDataManager->PutPath(m_ApplicationControlRequest->first, info.version, 0, info.path));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::PrepareDownloadTaskListRequest() NN_NOEXCEPT
    {
        if (m_IsDownloadTaskListRequested)
        {
            NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Prepare download task list since it is requested\n");

            m_DownloadTaskListRequest.emplace();
            NN_RESULT_TRY(nim::RequestDownloadTaskList(&(*m_DownloadTaskListRequest), m_LatestDownloadTaskListETag))
                NN_RESULT_CATCH(nim::ResultDeviceAccountNotRegistered)
                {
                    NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Ignore download task list request since device account not registered\n");

                    m_DownloadTaskListRequest = util::nullopt;
                    m_IsDownloadTaskListRequested = false;
                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    m_DownloadTaskListRequest = util::nullopt;
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::PrepareVersionListRequest() NN_NOEXCEPT
    {
        if (NeedsProcessVersionListRequest())
        {
            NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Prepare version list since it is requested\n");

            m_VersionListRequest.emplace();
            NN_RESULT_TRY(nim::RequestVersionList(&(*m_VersionListRequest), m_LatestVersionListETag))
                NN_RESULT_CATCH_ALL
                {
                    m_VersionListRequest = util::nullopt;
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::PrepareDownloadTicketRequest() NN_NOEXCEPT
    {
        if (!m_DownloadTicketRequest)
        {
            for (auto& item : m_DownloadTicketApplicationIdList)
            {
                if (item != ncm::ApplicationId::GetInvalidId())
                {
                    NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Prepare download ticket since it is requested\n");

                    es::RightsIdIncludingKeyId rightsId;
                    // 現時点で来る通知は必ずアプリケーションの利用可能通知であり、かつ、アプリケーションは共通の外部鍵で暗号化されているため
                    // コンテンツを利用するためには Program のチケットを取得すればいいので、Program のチケットを取得する
                    // この前提が変わるときには他の ContentType のチケットの取得も必要となる
                    Result result = m_TicketManager->GetRightsId(&rightsId, item.value, ncm::ContentType::Program);
                    if (result.IsFailure())
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Getting rights_id for download ticket failed as %08x\n", result.GetInnerValueForDebug());
                        item = ncm::ApplicationId::GetInvalidId();

                        NN_RESULT_SUCCESS;
                    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
                    // 同じ RightsId のチケットを既に保有していた場合はチケットのダウンロードをスキップする
                    bool hasTicket;
                    es::OwnTicket(&hasTicket, &rightsId, 1);
                    if (hasTicket)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Skipping download ticket because ticket is already downloaded\n");
                        item = ncm::ApplicationId::GetInvalidId();

                        NN_RESULT_SUCCESS;
                    }
#endif

                    m_DownloadTicketRequest.emplace();
                    m_DownloadTicketRequest->first = item;
                    result = ec::system::RequestDownloadTicketForPrepurchasedContents(&m_DownloadTicketRequest->second, rightsId);
                    if (result.IsFailure())
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Starting download ticket failed as %08x\n", result.GetInnerValueForDebug());
                        m_DownloadTicketRequest = util::nullopt;
                        item = ncm::ApplicationId::GetInvalidId();

                        NN_RESULT_SUCCESS;
                    }

                    NN_RESULT_SUCCESS;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    void ApplicationInstallRequestList::CleanupApplicationInstallRequest() NN_NOEXCEPT
    {
        for (auto& request : m_ApplicationInstallRequestList)
        {
            // クリーンナップは、タスクの状態変化ではなく上位の状態変化を契機として走るので、
            // 仮に Suspended な Result になっていたとしても Suspend はさせない
            NN_NS_TRACE_RESULT_IF_FAILED(HandleApplicationInstallRequestDone(&request, false), "[ApplicationInstallRequestList] ApplicationInstallRequest failed in cleanup\n");
            request = util::nullopt;
        }
    }

    void ApplicationInstallRequestList::CleanupApplyDeltaRequest() NN_NOEXCEPT
    {
        for (auto& request : m_ApplyDeltaRequestList)
        {
            request = util::nullopt;
        }
    }

    void ApplicationInstallRequestList::CleanupDownloadTicketRequest() NN_NOEXCEPT
    {
        m_DownloadTicketRequest = util::nullopt;
    }

    Result ApplicationInstallRequestList::HandleApplyDeltaRequestDone(util::optional<ApplyDeltaRequest>* request) NN_NOEXCEPT
    {
        if ((*request) && (*request)->second.TryWait())
        {
            auto result = (*request)->second.Get();
            auto id = (*request)->first;
            (*request) = util::nullopt;

            if (result.IsSuccess())
            {
                NN_NS_TRACE_RESULT_IF_FAILED(m_CommitManager->NotifyApplyDeltaTaskCompleted(id), "[ApplicationInstallRequestList] failed to notify apply delta completed\n");
            }

            NN_RESULT_THROW(result);
        }
        NN_RESULT_SUCCESS;
    }
    Result ApplicationInstallRequestList::HandleApplicationInstallRequestDone(util::optional<ApplicationInstallRequest>* request, bool willSuspend) NN_NOEXCEPT
    {
        if (*request && (*request)->second.GetEvent().TryWait())
        {
            auto result = (*request)->second.Get();
            auto attribute = ToAttribute(result);
            auto id = (*request)->first;
            (*request) = util::nullopt;

            // workaround for SIGLO-48620
            // コンテンツ配信サーバから CURLE_PARTIAL_FILE が返ってきたらリトライ待ちする
            if (result <= nim::ResultHttpConnectionContentPartialFile())
            {
                NN_RESULT_THROW(result);
            }

            // タスクの状態変化を契機とした attribute == Suspended は、Suspend させる
            // 一方、上位の状態変化に伴う attribute == Suspended は Suspend 状態にはしない (再開可能にする)
            // タスクの状態変化契機の場合は willSuspend == true となる
            if ((willSuspend && attribute == static_cast<Bit64>(ApplicationDownloadState::Suspended)) ||
                attribute != static_cast<Bit64>(ApplicationDownloadState::Suspended))
            {
                NN_NS_TRACE_RESULT_IF_FAILED(nim::SetNetworkInstallTaskAttribute(id, attribute), "[ApplicationInstallRequestList] Failed to set attribute\n");
                // 状態を設定した場合は、もう上位層でハンドルする必要もないので success を返す

                if (result.IsSuccess())
                {
                    NN_NS_TRACE_RESULT_IF_FAILED(m_CommitManager->NotifyDownloadTaskCompleted(id), "[ApplicationInstallRequestList] failed to notify download completed\n");
                }

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW(result);
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::HandleDownloadTaskListRequestDone() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT{ m_DownloadTaskListRequest = util::nullopt; };

        auto requestResult = m_DownloadTaskListRequest->Get();
        if (requestResult.IsFailure())
        {
            NN_RESULT_DO([this](Result result) NN_NOEXCEPT -> Result
            {
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(nim::ResultDownloadTaskListNotFound)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Download task list not found\n");
                        NN_RESULT_SUCCESS;
                    }
                    NN_RESULT_CATCH(nim::ResultDownloadTaskListIsLatest)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Download task list is latest\n");
                        NN_RESULT_SUCCESS;
                    }
                    NN_RESULT_CATCH(nim::ResultSystemUpdateRequiredForContentDelivery)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] System update required to get download task list\n");
                        if (m_IsSystemUpdateNotificationRequiredByDownloadTaskList)
                        {
                            NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] System update notified\n");
                            ns::NotifySystemUpdateForContentDelivery();
                        }
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            } (requestResult));

            m_IsDownloadTaskListRequested = false;
            NN_RESULT_SUCCESS;
        }

        // タスクにコンテンツを追加する可能性があるので、ダウンロード要求を一度撤回する
        // 次回 Prepare 時に必要な要求は再度行われる
        CleanupApplicationInstallRequest();

        NN_RESULT_DO(m_DtlManager->Push(&(*m_DownloadTaskListRequest)));
        NN_RESULT_DO(m_DownloadTaskListRequest->GetETag(&m_LatestDownloadTaskListETag));

        m_IsDownloadTaskListRequested = false;
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Download task list updated\n");
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::HandleVersionListRequestDone() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT{ m_VersionListRequest = util::nullopt; };

        NN_SDK_ASSERT(m_VersionListRequest);

        auto requestResult = m_VersionListRequest->Get();

        if (requestResult.IsFailure())
        {
            NN_RESULT_DO([](Result result) NN_NOEXCEPT -> Result
            {
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(nim::ResultVersionListNotFound)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Version list not found\n");
                        NN_RESULT_SUCCESS;
                    }
                    NN_RESULT_CATCH(nim::ResultVersionListIsLatest)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Version list is latest\n");
                        NN_RESULT_SUCCESS;
                    }
                    NN_RESULT_CATCH(nim::ResultSystemUpdateRequiredForContentDelivery)
                    {
                        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] System update required to get version list\n");
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            } (requestResult));

            m_IsVersionListRequested = false;
            NN_RESULT_SUCCESS;
        }

        // 現状でバージョンリストの更新時に既存タスクへのコンテンツ追加は行われないので
        // ダウンロード要求の撤回はしない

        // TODO: エラー発生時の片付けなど
        // TODO: バージョンリストマネージャでやる
        static char s_Buffer[16 * 1024];
        NN_RESULT_DO(m_VersionManager->UpdateVersionList(&m_VersionListRequest.value(), s_Buffer, sizeof(s_Buffer)));
        NN_RESULT_DO(m_VersionListRequest->GetETag(&m_LatestVersionListETag));

        m_IsVersionListRequested = false;
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Version list updated\n");

        NotifyVersionListImported();
        NN_RESULT_SUCCESS;
    }

    Result ApplicationInstallRequestList::HandleDownloadTicketRequestDone() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT{ m_DownloadTicketRequest = util::nullopt; };

        for (auto& item : m_DownloadTicketApplicationIdList)
        {
            if (item == m_DownloadTicketRequest->first)
            {
                item = ncm::ApplicationId::GetInvalidId();
            }
        }

        ec::system::TicketDownloadStatusForPrepurchasedContents value;
        NN_RESULT_DO(m_DownloadTicketRequest->second.Get(&value));

        if (!value.isPrepurchaseRecordDownloaded)
        {
            NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Ticket downloaded\n");
            NN_NS_TRACE_RESULT_IF_FAILED(m_RecordDb->UpdateEvent(m_DownloadTicketRequest->first, ApplicationEvent::ContentAvailable, true), "[ApplicationInstallRequestList] Update event failed 0x%016llx\n", m_DownloadTicketRequest->first);
            NN_NS_TRACE_RESULT_IF_FAILED(SendContentAvailableOverlayNotification(m_DownloadTicketRequest->first), "[ApplicationInstallRequestList] Send overlay notification failed 0x%016llx\n", m_DownloadTicketRequest->first);
        }
        else
        {
            // チケットがまだ予約状態で取得できなかった場合
            // 起こらないはずだが念のためログだけ出しておく
            NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Ticket is still unavailable\n");
        }

        NN_RESULT_SUCCESS;
    }

    void ApplicationInstallRequestList::NotifyVersionListImported() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[ApplicationInstallRequestList] Notify version list imported\n");

        m_VersionListImportedEvent.Signal();
    }

    bool ApplicationInstallRequestList::HasApplicationInstallRequest(nim::NetworkInstallTaskId id) NN_NOEXCEPT
    {
        return HasRequest<util::optional<ApplicationInstallRequest>, nim::NetworkInstallTaskId>(m_ApplicationInstallRequestList, MaxRequestCount, id);
    }

    bool ApplicationInstallRequestList::IsFullOfApplicationInstallRequest() NN_NOEXCEPT
    {
        return IsFull<util::optional<ApplicationInstallRequest>>(m_ApplicationInstallRequestList, MaxRequestCount);
    }

    bool ApplicationInstallRequestList::HasApplyDeltaRequest(nim::ApplyDeltaTaskId id) NN_NOEXCEPT
    {
        return HasRequest<util::optional<ApplyDeltaRequest>, nim::ApplyDeltaTaskId>(m_ApplyDeltaRequestList, MaxApplyDeltaRequestCount, id);
    }

    bool ApplicationInstallRequestList::IsFullOfApplyDeltaRequest() NN_NOEXCEPT
    {
        return IsFull<util::optional<ApplyDeltaRequest>>(m_ApplyDeltaRequestList, MaxApplyDeltaRequestCount);
    }

    bool ApplicationInstallRequestList::NeedsDownloadApplicationControl(const nim::NetworkInstallTaskInfo& info) NN_NOEXCEPT
    {
        return !IsFatal(info.progress.GetLastResult()) && !m_ControlDataManager->HasAny(info.applicationId);
    }

    void ApplicationInstallRequestList::GetRequiredTaskAttributes(int* outCount, RequiredTaskAttribute* attributes, int count) NN_NOEXCEPT
    {
        int index = 0;
        bool hasNetworkInstall = false;
        bool hasControl = false;
        bool hasApplyDelta = false;
        {
            nim::NetworkInstallTaskId idList[nim::MaxNetworkInstallTaskCount];
            auto c = nim::ListNetworkInstallTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < c; i++)
            {
                nim::NetworkInstallTaskInfo info;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nim::GetNetworkInstallTaskInfo(&info, idList[i]));
                if (ToState(info.attribute) == ApplicationDownloadState::Runnable)
                {
                    hasNetworkInstall = true;
                }
                if (NeedsDownloadApplicationControl(info))
                {
                    hasControl = true;
                }
            }
        }
        {
            nim::ApplyDeltaTaskId idList[nim::MaxApplyDeltaTaskCount];
            auto c = nim::ListApplyDeltaTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < c; i++)
            {
                ApplicationApplyDeltaTask task(idList[i]);
                bool needsProcess;
                auto result = task.NeedsProcess(&needsProcess);
                if (result.IsSuccess() && needsProcess)
                {
                    hasApplyDelta = true;
                }
            }
        }
        if (hasNetworkInstall && index < count)
        {
            attributes[index] = {RequestType::NetworkInstall, true, false};
            index++;
        }
        if ((hasControl || m_ApplicationControlRequest) && index < count)
        {
            attributes[index] = {RequestType::Control, false, false};
            index++;
        }
        if (hasApplyDelta && index < count)
        {
            attributes[index] = {RequestType::ApplyDelta, true, true};
            index++;
        }
        if (m_IsDownloadTaskListRequested && index < count)
        {
            attributes[index] = {RequestType::DownloadTaskList, false, false};
            index++;
        }
        if (NeedsProcessVersionListRequest() && index < count)
        {
            attributes[index] = {RequestType::VersionList, false, false};
            index++;
        }
        if (NeedsDownloadTicketRequest() && index < count)
        {
            attributes[index] = {RequestType::DownloadTicket, false, false};
            index++;
        }
        *outCount = index;
    }

    bool ApplicationInstallRequestList::NeedsProcessVersionListRequest() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        return m_IsVersionListRequested;
#else
        return false;
#endif
    }

    bool ApplicationInstallRequestList::NeedsDownloadTicketRequest() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        for (auto& item : m_DownloadTicketApplicationIdList)
        {
            if (item != ncm::ApplicationId::GetInvalidId())
            {
                return true;
            }
        }
        return false;
#else
        return false;
#endif
    }

    Result ApplicationInstallRequestList::SendContentAvailableOverlayNotification(ncm::ApplicationId id) NN_NOEXCEPT
    {
        ovln::format::ContentAvailableData data = { id, static_cast<Bit32>(ovln::format::AvailableDataType::Application) };

        ovln::Message message;
        message.tag = ovln::format::ContentAvailableDataTag;
        message.dataSize = sizeof(data);
        std::memcpy(&message.data, &data, sizeof(data));

        ovln::Send(m_Sender, message);

        NN_RESULT_SUCCESS;
    }

}}}
