﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_GameCardApi.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationEntityManager.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_RequestServer.h>

#include "ns_DebugUtil.h"
#include "ns_ShellUtil.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        bool IsApplicationContent(ncm::ContentMetaType type) NN_NOEXCEPT
        {
            switch(type)
            {
            case ncm::ContentMetaType::Application:
            case ncm::ContentMetaType::Patch:
            case ncm::ContentMetaType::AddOnContent:
                return true;
            default:
                return false;
            }
        }

        bool IsTargetContentMetaType(ncm::ContentMetaType type, ApplicationEntityFlag flags) NN_NOEXCEPT
        {
            if (!IsApplicationContent(type))
            {
                return false;
            }
            if ((flags & ApplicationEntityFlag_All) == ApplicationEntityFlag_All)
            {
                return true;
            }
            else if (type == ncm::ContentMetaType::Application && flags.Test<ApplicationEntityFlag_Application>())
            {
                return true;
            }
            else if (type == ncm::ContentMetaType::AddOnContent && flags.Test<ApplicationEntityFlag_AddOnContent>())
            {
                return true;
            }
            else if (type == ncm::ContentMetaType::Patch && flags.Test<ApplicationEntityFlag_Patch>())
            {
                return true;
            }
            else
            {
                return false;
            }
        }
    }

void ApplicationEntityManager::Initialize(ApplicationRecordDatabase* record, IntegratedContentManager* integrated, RequestServer* requestServer, SystemReportManager* reportManager, TicketManager* ticketManager) NN_NOEXCEPT
{
    m_pRecordDb = record;
    m_pIntegrated = integrated;
    m_pRequestServer = requestServer;
    m_SystemReportManager = reportManager;
    m_pTicketManager = ticketManager;
}

Result ApplicationEntityManager::DeleteApplicationEntityImpl(bool* outIsApplicationDeleted, ncm::ApplicationId id, ApplicationEntityFlag flags, ncm::StorageId storageId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outIsApplicationDeleted);
    NN_RESULT_THROW_UNLESS(flags.Test<ApplicationEntityFlag_SkipRunningCheck>() || !IsRunningApplication(id), ResultApplicationIsRunning());
    // アプリの実体が削除されたら、ユーザにとって自動更新不要なはずなので、無効化する
    NN_RESULT_DO(m_pRecordDb->DisableAutoUpdate(id));
    NN_RESULT_DO(m_pRecordDb->UpdateEvent(id, ApplicationEvent::Deleted, false));

    NN_RESULT_DO(CleanupTask(id, m_pIntegrated, m_pRequestServer->Stop(), TaskType::All));

    {
        auto buffer = GetContentMetaKeyBuffer();
        auto count = m_pIntegrated->ListInstalledApplicationContent(buffer, MaxContentMetaKeyCount, id,
            ncm::ContentMetaType::Unknown, ncm::ContentInstallType::Unknown);
        bool needsCommit = false;
        *outIsApplicationDeleted = false;
        for (int i = 0; i < count; i++)
        {
            if (IsTargetContentMetaType(buffer[i].type, flags))
            {
                NN_RESULT_DO(m_pIntegrated->DeleteContentMeta(buffer[i], nullptr, storageId));
                if (buffer[i].type == ncm::ContentMetaType::Application)
                {
                    *outIsApplicationDeleted = true;
                }
                needsCommit = true;
            }
        }
        if (needsCommit)
        {
            NN_RESULT_DO(m_pIntegrated->Commit());
        }
    }

    // コンテンツ記録がない記録があったらその時点で消す
    {
        int count;
        NN_RESULT_DO(m_pRecordDb->CountContentMeta(&count, id));
        if (count == 0)
        {
            NN_RESULT_DO(m_pRecordDb->Delete(id));
        }
    }

    m_SystemReportManager->ReportSdCardFreeSpace();
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationEntity(ncm::ApplicationId id, ApplicationEntityFlag flags) NN_NOEXCEPT
{
    bool isDeleted;
    NN_RESULT_DO(DeleteApplicationEntityImpl(&isDeleted, id, flags, ncm::StorageId::Any));

    if (isDeleted)
    {
        m_SystemReportManager->ReportApplicationInstall(ApplicationInstallReportType::UninstalledTemporarily, id);
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationCompletely(ncm::ApplicationId id, ApplicationEntityFlag flags) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS((ApplicationEntityFlag_All & flags) == ApplicationEntityFlag_All, ResultInvalidFlagArgument());

    if (m_pIntegrated->IsRegisteredStorage(ncm::StorageId::GameCard) && !flags.Test<ApplicationEntityFlag_SkipGameCardCheck>())
    {
        ncm::ContentMetaKey keyList[MaxApplicationCountOnGameCard];
        int appCount = m_pIntegrated->ListContent(keyList, static_cast<int>(NN_ARRAY_SIZE(keyList)), ncm::StorageId::Card, ncm::ContentMetaType::Application);
        NN_RESULT_THROW_UNLESS(!std::any_of(keyList , keyList + appCount, [&id](const ncm::ContentMetaKey& key) NN_NOEXCEPT { return id.value == key.id; }), ResultGameCardIsInserted());
    }

    bool isDeleted;
    NN_RESULT_DO(DeleteApplicationEntityImpl(&isDeleted, id, flags, ncm::StorageId::Any));
    if (m_pRecordDb->Has(id))
    {
        NN_RESULT_DO(m_pRecordDb->Delete(id));
    }

    if (isDeleted)
    {
        m_SystemReportManager->ReportApplicationInstall(ApplicationInstallReportType::UninstalledCompletely, id);
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationContentEntities(ncm::ApplicationId id, uint32_t innerFlagsValue, ncm::StorageId storageId) NN_NOEXCEPT
{
    bool isDeleted;
    ApplicationEntityFlag flags;
    flags._storage[0] = innerFlagsValue;
    NN_RESULT_DO(DeleteApplicationEntityImpl(&isDeleted, id, flags, storageId));
    NN_UNUSED(isDeleted);

    // この API は実際に使われる想定がないのでプレイレポートは送らない
    // もし使われることになった場合は、プレイレポートを送るための修正が必要

    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationEntity(ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_RESULT_DO(DeleteApplicationEntity(id, ApplicationEntityFlag_All));
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationCompletely(ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_RESULT_DO(DeleteApplicationCompletely(id, ApplicationEntityFlag_All));
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationContentEntity(const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pIntegrated->DeleteContentMeta(key));
    NN_RESULT_DO(m_pIntegrated->Commit());

    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteApplicationCompletelyForDebug(ncm::ApplicationId id, uint32_t innerFlagsValue) NN_NOEXCEPT
{
    ApplicationEntityFlag flags;
    flags._storage[0] = innerFlagsValue;

    NN_RESULT_DO(DeleteApplicationCompletely(id, flags));
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::IsAnyApplicationEntityRedundant(bool* outValue) NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> guard(m_ContentMetaKeyMutex);

    bool hasOrphanContents;
    NN_RESULT_DO(m_pIntegrated->HasOrphanContents(&hasOrphanContents));
    if (hasOrphanContents)
    {
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    auto buffer = GetApplicationContentMetaKeyBuffer();
    auto count = m_pIntegrated->ListInstalledApplication(buffer, MaxContentMetaKeyCount);
    for (int i = 0; i < count; i++)
    {
        const auto& appKey = buffer[i];

        if (appKey.key.type == ncm::ContentMetaType::Patch && appKey.key.installType == ncm::ContentInstallType::FragmentOnly)
        {
            // fragment-only のパッチは削除しない
            continue;
        }

        // アプリケーション記録にないコンテンツを検索
        bool hasRecord;
        NN_RESULT_DO(HasRecordOf(&hasRecord, appKey.applicationId, appKey.key));
        if (!hasRecord)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        for (auto j = i + 1; j < count; j++)
        {
            const auto& cmpKey = buffer[j];

            // 重複しているコンテンツを検索
            if (appKey.key == cmpKey.key)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }
    }

    *outValue = false;
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteRedundantApplicationEntity() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pIntegrated->CleanupOrphanContents());

    std::lock_guard<NonRecursiveMutex> guard(m_ContentMetaKeyMutex);

    const auto invalidId = ncm::ApplicationId::GetInvalidId();
    auto buffer = GetApplicationContentMetaKeyBuffer();
    auto count = m_pIntegrated->ListInstalledApplication(buffer, MaxContentMetaKeyCount);
    for (auto i = 0; i < count; i++)
    {
        auto& appKey = buffer[i];

        if (appKey.key.type == ncm::ContentMetaType::Patch && appKey.key.installType == ncm::ContentInstallType::FragmentOnly)
        {
            // fragment-only のパッチは削除しない
            continue;
        }
        else if (appKey.applicationId == invalidId)
        {
            continue;
        }

        // 記録にないコンテンツを削除
        if (m_pRecordDb->Has(appKey.applicationId))
        {
            ApplicationRecordAccessor list;
            NN_RESULT_DO(m_pRecordDb->Open(&list, appKey.applicationId));
            NN_RESULT_DO(DeleteUnrecordedContentMeta(appKey.key, &list));
        }
        else
        {
            NN_RESULT_DO(m_pIntegrated->DeleteContentMeta(appKey.key));
        }

        for (auto j = i + 1; j < count; j++)
        {
            auto& cmpKey = buffer[j];
            if (cmpKey.applicationId == invalidId)
            {
                continue;
            }

            // 重複しているコンテンツを削除
            if (appKey.key == cmpKey.key)
            {
                // DeleteDuplicatedContentMeta はゲームカードが検索対象外なので、
                // ゲームカードとコンテンツが重複しても消されない
                NN_RESULT_DO(m_pIntegrated->DeleteDuplicatedContentMeta(appKey.key));

                // 削除済みのコンテンツを検索対象から外す
                // アプリケーション記録にないコンテンツを削除したときに削除されたものも同様
                cmpKey.applicationId = invalidId;
            }
        }
    }
    NN_RESULT_DO(m_pIntegrated->Commit());

    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::IsApplicationEntityMovable(bool* outValue, ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
{
    NN_UNUSED(outValue);
    NN_UNUSED(id);
    NN_UNUSED(storage);
    NN_RESULT_THROW(ResultNotImplemented());
}

Result ApplicationEntityManager::MoveApplicationEntity(ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
{
    NN_UNUSED(id);
    NN_UNUSED(storage);
    NN_RESULT_THROW(ResultNotImplemented());
}

Result ApplicationEntityManager::IsAnyApplicationEntityInstalled(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> guard(m_ContentMetaKeyMutex);

    if (!m_pRecordDb->Has(id))
    {
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto buffer = GetContentMetaKeyBuffer();
    auto count = m_pIntegrated->ListInstalledApplicationContent(buffer, MaxContentMetaKeyCount, id);

    *outValue = count > 0;
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::CleanupUnrecordedApplicationEntity(ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<NonRecursiveMutex> guard(m_ContentMetaKeyMutex);

    ApplicationRecordAccessor accessor;
    bool hasRecord = m_pRecordDb->Has(id);
    if (hasRecord)
    {
        NN_RESULT_THROW_UNLESS(!m_pRecordDb->IsLocked(), ResultApplicationRecordDataOccupied());
        NN_RESULT_DO(m_pRecordDb->Open(&accessor, id));
    }

    auto list = GetContentMetaKeyBuffer();
    // FragmentOnly のコンテンツは消さないようにする必要があるが、当該API ではリストに入ってこないので、対応する必要がない
    auto count = m_pIntegrated->ListInstalledApplicationContent(list, MaxContentMetaKeyCount, id);

    for (int i = 0; i < count; i++)
    {
        auto& key = list[i];

        auto result = hasRecord ? DeleteUnrecordedContentMeta(key, &accessor) : m_pIntegrated->DeleteContentMeta(key);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[ApplicationEntityManager] Failed to delete content meta as 0x%08x\n", result.GetInnerValueForDebug());
        }
    }

    NN_RESULT_DO(m_pIntegrated->Commit());
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::HasRecordOf(bool* outValue, ncm::ApplicationId id, const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    if (!m_pRecordDb->Has(id))
    {
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_pRecordDb->Open(&list, id));
    if (!list.HasInstalled(key))
    {
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::CleanupAddOnContentsWithNoRights(ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetCleanupAddOnContentWithNoRightsResultForDebug());

    std::lock_guard<NonRecursiveMutex> guard(m_ContentMetaKeyMutex);

    auto list = GetContentMetaKeyBuffer();
    auto count = m_pIntegrated->ListInstalledApplicationContent(list, MaxContentMetaKeyCount, id);

    bool needsCommit = false;
    for (int i = 0; i < count; i++)
    {
        auto& key = list[i];
        if (key.type == ncm::ContentMetaType::AddOnContent)
        {
            uint8_t programIndex;
            NN_RESULT_DO(m_pIntegrated->GetMinimumProgramIndex(&programIndex, key));
            bool hasRights;
            NN_RESULT_TRY(m_pTicketManager->HasContentRights(&hasRights, key, ncm::ContentType::Data, programIndex))
                NN_RESULT_CATCH(fs::ResultDataCorrupted)
                {
                    m_pRecordDb->SetApplicationTerminateResult(id, NN_RESULT_CURRENT_RESULT);
                    continue;
                }
            NN_RESULT_END_TRY;

            if (!hasRights)
            {
                NN_RESULT_DO(m_pIntegrated->DeleteContentMeta(key));
                // ここで記録を消してしまうと、記録にないけど ContentMetaDatabase には残っているというケースが出てきてしまうが、
                // Cleanup される際に消えるので、それでよしとする
                NN_RESULT_DO(m_pRecordDb->DeleteKey(id, key));
                needsCommit = true;
            }
        }
    }
    if (needsCommit)
    {
        NN_RESULT_DO(m_pIntegrated->Commit());
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationEntityManager::DeleteUnrecordedContentMeta(const ncm::ContentMetaKey& key, const ApplicationRecordAccessor* pAccessor) NN_NOEXCEPT
{
    auto recordedKey = pAccessor->GetByIdFromData(key.id);

    if (!recordedKey || recordedKey->key != key)
    {
        NN_RESULT_DO(m_pIntegrated->DeleteContentMeta(key, recordedKey ? &(recordedKey->key) : nullptr));
    }
    NN_RESULT_SUCCESS;
}

}}}
