﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nsd/detail/nsd_Shim.h>
#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/nsd/detail/nsd_IManager.sfdl.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/detail/nsd_ServiceObjectDfc.h>
#else
#include <nn/nsd/detail/nsd_ServiceObjectHipc.h>
#endif

// Windows 環境では Initialize でサーバ起動も行うため。
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/server/nsd_ApiForServer.h>
#endif

#include <mutex>
#include <nn/nsd/detail/nsd_StaticMutex.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace nsd { namespace detail {

namespace
{
    // 初期化周りの Mutex
    nn::nsd::detail::StaticMutex g_Mutex = NN_NSD_DETAIL_STATIC_MUTEX_INITIALIZER(false);

    // Initialize 参照カウント
    uint32_t g_InitializeCount = 0;

    // サービスオブジェクトへの共有ポインタ
    nn::sf::SharedPointer<nn::nsd::detail::IManager> g_NsdManager = nullptr;

    // 使用しているサービスポート
    nn::nsd::ServiceType g_UsingServiceType;

Result InitializeCore() NN_NOEXCEPT
{
    if ( g_InitializeCount == 0 )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        // Windows 環境では IPCサーバ起動もここで行う
        nn::nsd::server::StartServer();
#endif

        NN_SDK_ASSERT( !g_NsdManager );

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // Windows は権限チェックがないので常に Admin として全 API を利用可能にする
        g_UsingServiceType = nn::nsd::ServiceType_Admin;
        auto resultForGetServiceObject = nn::nsd::detail::GetServiceObjectByDfc( &g_NsdManager, nn::nsd::ServiceType_Admin );
#else

        // User で service object を取得してみる
        g_UsingServiceType = nn::nsd::ServiceType_User;
        auto resultForGetServiceObject = nn::nsd::detail::GetServiceObjectByHipc( &g_NsdManager, nn::nsd::ServiceType_User );

        if ( ! resultForGetServiceObject.IsSuccess() )
        {
            // 失敗なら Admin で service object を取得してみる
            g_UsingServiceType = nn::nsd::ServiceType_Admin;
            resultForGetServiceObject = nn::nsd::detail::GetServiceObjectByHipc( &g_NsdManager, nn::nsd::ServiceType_Admin );
        }

#endif

        // Initialize 出来なかった場合
        NN_RESULT_THROW_UNLESS( resultForGetServiceObject.IsSuccess(), nn::nsd::ResultInitializeFailed() );
    }

    ++ g_InitializeCount;

    NN_RESULT_SUCCESS;
}

bool IsInitializedCore() NN_NOEXCEPT
{
    return (g_InitializeCount > 0);
}

Result FinalizeCore() NN_NOEXCEPT
{
    if ( g_InitializeCount > 0 )
    {
        -- g_InitializeCount;

        if ( g_InitializeCount == 0 )
        {
            NN_SDK_ASSERT( g_NsdManager );

            // nullptr を代入して解放
            g_NsdManager = nullptr;

#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::nsd::server::StopServer();
#endif
        }
    }

    NN_RESULT_SUCCESS;
}
} // namespace

//----------------------------------------------------------------
nn::Result Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    return InitializeCore();
}
bool IsInitialized() NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    return IsInitializedCore();
}
nn::Result Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    return FinalizeCore();
}
namespace ipc {
nn::Result GetSettingName(nn::nsd::SettingName* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetSettingName( pOut );
}
nn::Result GetEnvironmentIdentifier(nn::nsd::EnvironmentIdentifier* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetEnvironmentIdentifier( pOut );
}
nn::Result GetDeviceId(nn::nsd::DeviceId* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetDeviceId( pOut );
}
nn::Result DeleteSettings(const nn::nsd::DeleteMode deleteMode) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    int32_t deleteModeValue = deleteMode;
    return g_NsdManager->DeleteSettings(deleteModeValue);
}
nn::Result ImportSettings(const void* pBuffer, size_t bufferSize, void* pWorkBuffer, size_t workBufferSize, const ImportMode saveMode) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pBuffer), bufferSize);
    nn::sf::OutBuffer pOutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize);
    int32_t saveModeValue = saveMode;
    return g_NsdManager->ImportSettings(pInBuffer, pOutBuffer, saveModeValue);
}
nn::Result Resolve(nn::nsd::Fqdn* pOut, const nn::nsd::Fqdn& fqdn) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->Resolve(pOut, fqdn);
}
nn::Result ResolveEx(nn::Result* pInnerResult, nn::nsd::Fqdn* pOut, const nn::nsd::Fqdn& fqdn) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    nn::nsd::InnerResult c;
    Result res = g_NsdManager->ResolveEx(&c, pOut, fqdn);
    *pInnerResult = c.result;
    return res;
}
nn::Result GetNasServiceSetting(nn::nsd::NasServiceSetting* pOutNasServiceSetting, const nn::nsd::NasServiceName& nasServiceName) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetNasServiceSetting(pOutNasServiceSetting, nasServiceName);
}
nn::Result GetNasServiceSettingEx(nn::Result* pInnerResult, nn::nsd::NasServiceSetting* pOutNasServiceSetting, const nn::nsd::NasServiceName& nasServiceName) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    nn::nsd::InnerResult c;
    Result res = g_NsdManager->GetNasServiceSettingEx(&c, pOutNasServiceSetting, nasServiceName);
    *pInnerResult = c.result;
    return res;
}
nn::Result GetNasRequestFqdn(nn::nsd::Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetNasRequestFqdn(pOut);
}
nn::Result GetNasRequestFqdnEx(nn::Result* pInnerResult, nn::nsd::Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    nn::nsd::InnerResult c;
    Result res = g_NsdManager->GetNasRequestFqdnEx(&c, pOut);
    *pInnerResult = c.result;
    return res;
}
nn::Result GetNasApiFqdn(nn::nsd::Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetNasApiFqdn(pOut);
}
nn::Result GetNasApiFqdnEx(nn::Result* pInnerResult, nn::nsd::Fqdn* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    nn::nsd::InnerResult c;
    Result res = g_NsdManager->GetNasApiFqdnEx(&c, pOut);
    *pInnerResult = c.result;
    return res;
}
nn::Result GetCurrentSetting(nn::nsd::SaveData* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }
    return g_NsdManager->GetCurrentSetting(pOut);
}


nn::Result WriteSaveDataToFsForTest(const nn::nsd::SaveData& saveData) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }

    return g_NsdManager->WriteSaveDataToFsForTest(saveData);
}

nn::Result ReadSaveDataFromFsForTest(nn::nsd::SaveData* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }

    return g_NsdManager->ReadSaveDataFromFsForTest(pOut);
}

nn::Result DeleteSaveDataOfFsForTest() NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }

    return g_NsdManager->DeleteSaveDataOfFsForTest();
}

nn::Result SetChangeEnvironmentIdentifierDisabled(bool isDisabled) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }

    return g_NsdManager->SetChangeEnvironmentIdentifierDisabled(isDisabled);
}

nn::Result IsChangeEnvironmentIdentifierDisabled(bool* pOutIsDisabled) NN_NOEXCEPT
{
    std::lock_guard<nn::nsd::detail::StaticMutex> lock(g_Mutex);
    if ( ! IsInitializedCore() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( InitializeCore() );
    }

    return g_NsdManager->IsChangeEnvironmentIdentifierDisabled(pOutIsDisabled);
}

} // nn::nsd::detail::ipc

}}} // nn::nsd::detail
