﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/fs/nsd_Fs.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/device/nsd_Device.h>
#include <nn/nsd/detail/nsd_BuiltinBackboneSettings.h>
#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/detail/nsd_Log.h>

#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#define NN_DETAIL_NSD_RESULT_DO(r) \
    do \
    { \
        const auto& _nn_detail_nsd_result_do_temporary((r)); \
        if(_nn_detail_nsd_result_do_temporary.IsFailure()) \
        { \
            g_ErrorLogger.SetIfFailed(_nn_detail_nsd_result_do_temporary, #r); \
        } \
        NN_RESULT_THROW_UNLESS(_nn_detail_nsd_result_do_temporary.IsSuccess(), _nn_detail_nsd_result_do_temporary); \
    } while (NN_STATIC_CONDITION(0))


namespace nn { namespace nsd { namespace detail { namespace fs {

    class ErrorLogger
    {
    public:
        ErrorLogger() NN_NOEXCEPT:
            m_Msg(""),
            m_Result(nn::ResultSuccess())
        {}

        void SetIfFailed(nn::Result result, const char* msg) NN_NOEXCEPT
        {
            if(result.IsFailure())
            {
                // パスが見つからないというnsdエラーはここでは何もしない
                if ( nn::nsd::ResultNotFound().Includes(result) )
                {
                    return;
                }

                if(this->m_Result.IsSuccess()) // 一度だけセットできる
                {
                    this->m_Result = result;
                    this->m_Msg = msg;
                }

                NN_DETAIL_NSD_ERROR("[NSD] %s failed. (%08x, %03d-%04d)\n",
                    msg,
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());
            }
        }

        void DumpIfFailed() const NN_NOEXCEPT
        {
            if(m_Result.IsSuccess())
            {
                return;
            }

            NN_DETAIL_NSD_ERROR("[NSD] nn::nsd::detail::fs first error / %s failed. (%08x, %03d-%04d)\n",
                m_Msg,
                m_Result.GetInnerValueForDebug(),
                m_Result.GetModule(), m_Result.GetDescription());
        }

    private:
        const char* m_Msg;
        nn::Result m_Result;
    };

    namespace
    {
        const uint64_t SystemDataId = 0x80000000000000b0ull; // システムセーブデータID
        const char NsdMountName[] = "nsdsave";
        const char NsdFileName[] = "nsdsave:/file";

        const size_t SystemDataUsingSize = 0x20000; // 利用サイズ (0x20000 = 131072byte)
        const size_t SystemDataJournalSize = 0x20000 + 0x4000; // ジャーナル領域のサイズ (0x24000 = 147456byte)

        ErrorLogger g_ErrorLogger;
    }

    //----------------------------------------------------------------
    // システムセーブデータのマウント
    // 存在しなかった場合に isToCreateFile が true なら作成して再マウントする
    // isToCreateFile が false なら nn::nsd::ResultNotFound() で戻る
    nn::Result MountNsdSystemSaveData( bool isToCreateFile )
    {
        // 移行期間は暫定フォーマット版のセーブデータが作成される。
        // それを無効化する。(移行期間後削除されるか、空実装となる予定)
        nn::fs::DisableAutoSaveDataCreation();

        // マウント
        NN_RESULT_TRY(nn::fs::MountSystemSaveData( NsdMountName, SystemDataId ))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            // 存在しなかったらファイルを作成せずにエラーで戻る場合
            NN_RESULT_THROW_UNLESS( isToCreateFile , nn::nsd::ResultNotFound() );

            // セーブデータ作成・再マウント
            NN_RESULT_DO(nn::fs::CreateSystemSaveData( SystemDataId, SystemDataUsingSize, SystemDataJournalSize, nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData)); // ユーザセーブデータを残す初期化で削除しない指定
            NN_RESULT_DO(nn::fs::MountSystemSaveData( NsdMountName, SystemDataId ));
        }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------
    // セーブファイル存在確認
    nn::Result ConfirmSaveDataExistence()
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        nn::Result result = nn::fs::GetEntryType( &directoryEntryType, NsdFileName );
        if ( nn::fs::ResultPathNotFound().Includes(result) )
        {
            NN_DETAIL_NSD_INFO("nsd savedata not found.\n");
            return nn::nsd::ResultNotFound();
        }

        g_ErrorLogger.SetIfFailed(result, "nn::fs::GetEntryType");

        return result;
    }

    //----------------------------------------------------------------
    // デバイスに保存
    nn::Result WriteSaveDataWithOffset( const void* in, int64_t offset, size_t size ) NN_NOEXCEPT
    {
        // マウント
        NN_DETAIL_NSD_RESULT_DO( MountNsdSystemSaveData( true ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( NsdMountName );
        };

        // ファイル存在確認・作成
        nn::Result result = ConfirmSaveDataExistence();
        if ( nn::nsd::ResultNotFound().Includes(result) )
        {
            NN_DETAIL_NSD_RESULT_DO( nn::fs::CreateFile( NsdFileName, 0 ) );
        }
        else if ( result.IsFailure() )
        {
            NN_RESULT_THROW( result );
        }

        {
            // オープン
            nn::fs::FileHandle fileHandle;
            NN_DETAIL_NSD_RESULT_DO( nn::fs::OpenFile( &fileHandle, NsdFileName, nn::fs::OpenMode_Write|nn::fs::OpenMode_AllowAppend ));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile( fileHandle );
            };

            // 書き込み
            NN_DETAIL_NSD_RESULT_DO( nn::fs::WriteFile( fileHandle, offset, in, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush) ));
        }

        NN_DETAIL_NSD_RESULT_DO( nn::fs::CommitSaveData( NsdMountName ) );
        NN_RESULT_SUCCESS;

    }

    //----------------------------------------------------------------
    // デバイスに保存
    nn::Result WriteSaveData(const SaveData& in) NN_NOEXCEPT
    {
        return WriteSaveDataWithOffset( reinterpret_cast<const void*>(&in), 0, sizeof(SaveData) );
    }

    //----------------------------------------------------------------
    // Expire時間の保存
    nn::Result WriteExpireUnixTime(const SaveData& in) NN_NOEXCEPT
    {
        return WriteSaveDataWithOffset( reinterpret_cast<const void*>(&in.expireUnixTime), offsetof(SaveData, expireUnixTime), sizeof(in.expireUnixTime) );
    }

    //----------------------------------------------------------------
    // isAvailableフラグの保存
    nn::Result WriteIsAvailable(const SaveData& in) NN_NOEXCEPT
    {
        return WriteSaveDataWithOffset( reinterpret_cast<const void*>(&in.isAvailable), offsetof(SaveData, isAvailable), sizeof(in.isAvailable) );
    }

    //----------------------------------------------------------------
    // アプリケーション設定部分を 0 埋め
    nn::Result FillEmptyApplicationSetting() NN_NOEXCEPT
    {
        NN_DETAIL_NSD_RESULT_DO( MountNsdSystemSaveData( false ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( NsdMountName );
        };

        NN_RESULT_DO(ConfirmSaveDataExistence());

        {
            // オープン
            nn::fs::FileHandle fileHandle;
            NN_RESULT_DO( nn::fs::OpenFile( &fileHandle, NsdFileName, nn::fs::OpenMode_Write|nn::fs::OpenMode_AllowAppend ));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::FlushFile( fileHandle );
                nn::fs::CloseFile( fileHandle );
            };

            const size_t fillBufferSize = 512;
            char fillBuffer[ fillBufferSize ];
            std::memset(fillBuffer, 0, fillBufferSize);

            size_t fillSize = sizeof(nn::nsd::ApplicationSettings);
            for( int position=0; position < static_cast<int>(fillSize); position += fillBufferSize )
            {
                NN_RESULT_DO( nn::fs::WriteFile(
                                                fileHandle,
                                                offsetof(nn::nsd::SaveData, applicationSettings) + position,
                                                &fillBuffer,
                                                std::min( fillSize - position, fillBufferSize ),
                                                nn::fs::WriteOption::MakeValue(0) ));
            }
        }

        NN_RESULT_DO( nn::fs::CommitSaveData( NsdMountName ) );
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------
    // デバイスから読み込み (オフセット・サイズ指定)
    nn::Result ReadSaveDataWithOffset( void* buffer, int64_t offset, size_t size ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        nn::fs::FileHandle fileHandle = {nullptr};
        nn::Result result;

        // マウント
        NN_DETAIL_NSD_RESULT_DO( MountNsdSystemSaveData( false ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( NsdMountName );
        };

        // ファイル存在確認
        NN_RESULT_DO( ConfirmSaveDataExistence());

        // オープン
        NN_DETAIL_NSD_RESULT_DO( nn::fs::OpenFile( &fileHandle, NsdFileName, nn::fs::OpenMode_Read ));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( fileHandle );
        };

        // 読み込み
        size_t readSize;
        NN_DETAIL_NSD_RESULT_DO( nn::fs::ReadFile( &readSize, fileHandle, offset, buffer, size ));
        // TODO: readSize のチェックが必要か？

        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------
    // デバイスから読み込み
    nn::Result ReadSaveData(SaveData* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        return ReadSaveDataWithOffset( pOut, 0, sizeof(nn::nsd::SaveData) );
    }

    //----------------------------------------------------------------
    // 設定名を取得 (セーブデータの設定名の部分だけを読み込み)
    /*
    nn::Result GetSettingName(SettingName* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        return ReadSaveDataWithOffset( pOut, offsetof(nn::nsd::SaveData, settingName), sizeof(nn::nsd::SettingName) );
    }
    */

    //----------------------------------------------------------------
    // 環境識別子を取得 (セーブデータの環境識別子の部分だけを読み込み)
    nn::Result GetEnvironmentIdentifier(EnvironmentIdentifier* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        return ReadSaveDataWithOffset( pOut, offsetof(nn::nsd::SaveData, environmentIdentifier), sizeof(nn::nsd::EnvironmentIdentifier) );
    }

    //----------------------------------------------------------------
    // デバイスから削除
    // もとから存在しなかった場合は nn::fs::ResultTargetNotFound が返る
    nn::Result DeleteSaveData() NN_NOEXCEPT
    {
        auto result = nn::fs::DeleteSaveData( SystemDataId );
        if( result.IsFailure() )
        {
            if ( !nn::fs::ResultTargetNotFound().Includes(result) )
            {
                g_ErrorLogger.SetIfFailed(result, "nn::fs::DeleteSystemFile"); // nn::fs::ResultTargetNotFound 以外は想定外
            }
            return result;
        }

        NN_RESULT_SUCCESS;
    }


    void DumpFirstFsErrorIfFailed() NN_NOEXCEPT
    {
        g_ErrorLogger.DumpIfFailed();
    }

}}}} // nn::nsd::detail::fs
