﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_RegisteredUpdatePartition.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/ns/detail/ns_ISystemUpdateInterface.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/util/util_StringUtil.h>

#include "ns_Initialize.h"

namespace nn { namespace ns {

    namespace
    {

        void MakeEulaDataPath(detail::EulaDataPath* outValue, const char* path)
        {
            util::Strlcpy(outValue->str, path, static_cast<int>(sizeof(*outValue)));
        }

    }

    void InitializeForSystemUpdate() NN_NOEXCEPT
    {
        InitializeServiceObjects(false);
    }

    void FinalizeForSystemUpdate() NN_NOEXCEPT
    {
        FinalizeServiceObjects();
    }


    BackgroundNetworkUpdateState GetBackgroundNetworkUpdateState() NN_NOEXCEPT
    {
        BackgroundNetworkUpdateState state;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemUpdateInterfaceInterface()->GetBackgroundNetworkUpdateState(&state));

        return state;
    }

    Result NotifyExFatDriverRequired() NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->NotifyExFatDriverRequired();
    }

    Result NotifyExFatDriverDownloaded() NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->NotifyExFatDriverDownloadedForDebug();
    }

    Result ClearExFatDriverStatus() NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->ClearExFatDriverStatusForDebug();
    }

    Result RequestBackgroundNetworkUpdate() NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->RequestBackgroundNetworkUpdate();
    }

    Result NotifyBackgroundNetworkUpdate(const nn::ncm::ContentMetaKey& systemUpdateMetaKey) NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->NotifyBackgroundNetworkUpdate(systemUpdateMetaKey);
    }

    void GetSystemUpdateNotificationEventForContentDelivery(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemUpdateInterfaceInterface()->GetSystemUpdateNotificationEventForContentDelivery(&nativeHandle));
        outValue->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_ManualClear);
        nativeHandle.Detach();
    }

    void NotifySystemUpdateForContentDelivery() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemUpdateInterfaceInterface()->NotifySystemUpdateForContentDelivery());
    }

    Result DestroySystemUpdateTask() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetSystemUpdateInterfaceInterface()->DestroySystemUpdateTask());
        NN_RESULT_SUCCESS;
    }

    Result RequestSendSystemUpdate(AsyncResult* outValue, uint32_t ipv4, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetSystemUpdateInterfaceInterface()->RequestSendSystemUpdate(&nativeHandle, &sp, ipv4, port, info));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    SystemUpdateProgress GetSendSystemUpdateProgress() NN_NOEXCEPT
    {
        SystemUpdateProgress progress;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemUpdateInterfaceInterface()->GetSendSystemUpdateProgress(&progress));
        return progress;
    }

    SystemUpdateControl::SystemUpdateControl() NN_NOEXCEPT{}
    SystemUpdateControl::~SystemUpdateControl() NN_NOEXCEPT
    {
        if (m_Interface)
        {
            Relieve();
        }
    }

    SystemUpdateControl::SystemUpdateControl(sf::SharedPointer<detail::ISystemUpdateControl> interfac) NN_NOEXCEPT : m_Interface(interfac) {}

    SystemUpdateControl::SystemUpdateControl(SystemUpdateControl&& rvalue) NN_NOEXCEPT
    {
        m_Interface = std::move(rvalue.m_Interface);
    }

    SystemUpdateControl& SystemUpdateControl::operator=(SystemUpdateControl&& rvalue) NN_NOEXCEPT
    {
        SystemUpdateControl(std::move(rvalue)).swap(*this);

        return *this;
    }

    void SystemUpdateControl::swap(SystemUpdateControl& other) NN_NOEXCEPT
    {
        m_Interface = std::move(other.m_Interface);
    }

    Result SystemUpdateControl::Occupy() NN_NOEXCEPT
    {
        return GetSystemUpdateInterfaceInterface()->OpenSystemUpdateControl(&m_Interface);
    }

    void SystemUpdateControl::Relieve() NN_NOEXCEPT
    {
        m_Interface = nullptr;
    }

    bool SystemUpdateControl::HasDownloaded() NN_NOEXCEPT
    {
        bool has;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->HasDownloaded(&has));

        return has;
    }

    Result SystemUpdateControl::RequestCheckLatestUpdate(AsyncLatestSystemUpdate* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        return RequestAsyncValue<AsyncLatestSystemUpdate, detail::IAsyncValue>(outValue,
            [this](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncValue>* outResult)->Result
            {
                return this->m_Interface->RequestCheckLatestUpdate(outHandle, outResult);
            });
    }

    Result SystemUpdateControl::RequestDownloadLatestUpdate(AsyncResult* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [this](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return this->m_Interface->RequestDownloadLatestUpdate(outHandle, outResult);
            });
    }

    SystemUpdateProgress SystemUpdateControl::GetDownloadProgress() NN_NOEXCEPT
    {
        SystemUpdateProgress progress;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetDownloadProgress(&progress));

        return progress;
    }

    Result SystemUpdateControl::GetDownloadedEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetDownloadedEulaDataSize(&size, path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateControl::GetDownloadedEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetDownloadedEulaData(&size, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    void SystemUpdateControl::ApplyDownloadedUpdate() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ApplyDownloadedUpdate());
    }

    Result SystemUpdateControl::SetupCardUpdate(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        os::TransferMemory transferMemory(buffer, bufferSize, os::MemoryPermission_None);
        sf::NativeHandle nativeHandle(transferMemory.Detach(), true);

        return this->m_Interface->SetupCardUpdate(std::move(nativeHandle), bufferSize);
    }

    Result SystemUpdateControl::RequestPrepareCardUpdate(AsyncResult* outValue) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [this](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
        {
            return this->m_Interface->RequestPrepareCardUpdate(outHandle, outResult);
        });
    }

    SystemUpdateProgress SystemUpdateControl::GetPrepareCardUpdateProgress() NN_NOEXCEPT
    {
        SystemUpdateProgress progress;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetPrepareCardUpdateProgress(&progress));

        return progress;
    }

    bool SystemUpdateControl::HasPreparedCardUpdate() NN_NOEXCEPT
    {
        bool has;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->HasPreparedCardUpdate(&has));

        return has;
    }

    Result SystemUpdateControl::GetPreparedCardUpdateEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetPreparedCardUpdateEulaDataSize(&size, path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateControl::GetPreparedCardUpdateEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetPreparedCardUpdateEulaData(&size, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    void SystemUpdateControl::ApplyCardUpdate() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ApplyCardUpdate());
    }

    Result SystemUpdateControl::ApplyCardUpdateForDebug() NN_NOEXCEPT
    {
        return m_Interface->ApplyCardUpdate();
    }

    Result SystemUpdateControl::SetupCardUpdateViaSystemUpdater(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::RegisterUpdatePartition());
        os::TransferMemory transferMemory(buffer, bufferSize, os::MemoryPermission_None);
        sf::NativeHandle nativeHandle(transferMemory.Detach(), true);
        return this->m_Interface->SetupCardUpdateViaSystemUpdater(std::move(nativeHandle), bufferSize);
    }

    bool SystemUpdateControl::HasReceived() NN_NOEXCEPT
    {
        bool has;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->HasReceived(&has));

        return has;
    }

    Result SystemUpdateControl::SetupToReceiveSystemUpdate() NN_NOEXCEPT
    {
        return m_Interface->SetupToReceiveSystemUpdate();
    }

    Result SystemUpdateControl::RequestReceiveSystemUpdate(AsyncResult* outValue, uint32_t ipv4, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT
    {
        return RequestAsyncValue<AsyncResult, detail::IAsyncResult>(outValue,
            [this, &ipv4, &port, &info](sf::NativeHandle* outHandle, sf::SharedPointer<detail::IAsyncResult>* outResult)->Result
            {
                return this->m_Interface->RequestReceiveSystemUpdate(outHandle, outResult, ipv4, port, info);
            });
    }

    SystemUpdateProgress SystemUpdateControl::GetReceiveProgress() NN_NOEXCEPT
    {
        SystemUpdateProgress progress;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetReceiveProgress(&progress));

        return progress;
    }

    Result SystemUpdateControl::GetReceivedEulaDataSize(size_t* outValue, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetReceivedEulaDataSize(&size, path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateControl::GetReceivedEulaData(size_t* outValue, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        detail::EulaDataPath path;
        MakeEulaDataPath(&path, eulaDataPath);

        uint64_t size;
        NN_RESULT_DO(m_Interface->GetReceivedEulaData(&size, sf::OutBuffer(reinterpret_cast<char*>(buffer), bufferSize), path));
        *outValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    void SystemUpdateControl::ApplyReceivedUpdate() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ApplyReceivedUpdate());
    }

    template <typename AsyncT, typename IAsyncT, typename RequestFuncT>
    Result SystemUpdateControl::RequestAsyncValue(AsyncT* outValue, RequestFuncT func) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<IAsyncT> sp;
        NN_RESULT_DO(func(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }
}}
