﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_DefaultAllocationPolicy.h>

namespace nn { namespace ns {
    // このクラスは nnMain 以前で使用することはできません。
    template<typename InterfaceType>
    class ServiceObjectHolder
    {
    public:
        NN_IMPLICIT ServiceObjectHolder() NN_NOEXCEPT : m_InterfaceObject(nullptr), m_InitializeCountLock(false), m_InitializedCount(0)
        {
        }

        void InitializeDirectly(sf::SharedPointer<InterfaceType> interfaceObject) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> lock(m_InitializeCountLock);

            if (m_InitializedCount == 0)
            {
                m_InterfaceObject = interfaceObject.Detach();
            }
            m_InitializedCount++;
        }

        Result InitializeWithProxyObject(const char* portName, MemoryResource* memoryResource) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> lock(m_InitializeCountLock);

            if (m_InitializedCount == 0)
            {
                sf::SharedPointer<InterfaceType> p;
                NN_RESULT_DO((sf::CreateHipcProxyByName(&p, memoryResource, portName)));
                m_InterfaceObject = p.Detach();
            }
            m_InitializedCount++;
            NN_RESULT_SUCCESS;
        }

        template<typename ImplType>
        void InitializeWithImplObject(ImplType* implObject, MemoryResource* memoryResource) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> lock(m_InitializeCountLock);

            if (m_InitializedCount == 0)
            {
                sf::ScopedCurrentMemoryResourceSetter setter(memoryResource);
                m_InterfaceObject = sf::CreateSharedObjectWithoutManagement<InterfaceType, ImplType>(implObject).Detach();
            }
            m_InitializedCount++;
        }

        void Finalize() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> lock(m_InitializeCountLock);

            if (m_InitializedCount > 0)
            {
                m_InitializedCount--;
                if (m_InitializedCount == 0 && m_InterfaceObject)
                {
                    m_InterfaceObject->Release();
                    m_InterfaceObject = nullptr;
                }
            }
        }

        bool IsInitialized() NN_NOEXCEPT
        {
            return m_InitializedCount > 0;
        }

        InterfaceType* GetInterfaceObject() NN_NOEXCEPT
        {
            return m_InterfaceObject;
        }

    private:
        InterfaceType* m_InterfaceObject;
        os::Mutex m_InitializeCountLock;
        int m_InitializedCount;
    };

}}

