﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/srv/ns_DevelopInterfaceServerFactory.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/arp/arp_Types.h>

namespace nn { namespace ns {

    namespace
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        struct AllocatorTag{};
        typedef nn::sf::ExpHeapStaticAllocator<1024, AllocatorTag> MyAllocator;
#endif

        nn::ns::detail::IDevelopInterface* g_RefDevelopInterface;
    }

    Result InitializeForDevelop() NN_NOEXCEPT
    {
        NN_SDK_ASSERT( ! g_RefDevelopInterface );

        nn::sf::SharedPointer<nn::ns::detail::IDevelopInterface> shared;

#if defined( NN_BUILD_CONFIG_OS_WIN )
        shared = nn::ns::srv::GetSharedDevelopInterfaceServer();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
        NN_RESULT_DO((sf::CreateHipcProxyByName<detail::IDevelopInterface, MyAllocator::Policy>(&shared, detail::PortNameForDevelop)));
#else
    #error "unsupported os"
#endif

        g_RefDevelopInterface = shared.Detach();

        NN_RESULT_SUCCESS;
    }

    Result FinalizeForDevelop() NN_NOEXCEPT
    {
        nn::sf::ReleaseSharedObject(g_RefDevelopInterface);
        g_RefDevelopInterface = nullptr;

        NN_RESULT_SUCCESS;
    }

    Result LaunchProgram(os::ProcessId* pOut, const ProgramLaunchProperty& launchProperty, int flags) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->LaunchProgram(pOut, launchProperty, static_cast<int32_t>(flags));
    }

    Result GetShellEvent(os::SystemEvent* pOut) NN_NOEXCEPT
    {
        sf::NativeHandle temp;
        auto ret = g_RefDevelopInterface->GetShellEventHandle(&temp);
        pOut->Attach(temp.GetOsHandle(), true, os::InvalidNativeHandle, false, os::EventClearMode_AutoClear);
        temp.Detach();
        return ret;
    }

    Result GetShellEventInfo(ShellEventInfo* pOut) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->GetShellEventInfo(pOut);
    }

    Result TerminateProcess(os::ProcessId processId) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->TerminateProcess(processId);
    }

    Result TerminateProgram(ncm::ProgramId programId) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->TerminateProgram(programId);
    }

    Result TerminateApplication() NN_NOEXCEPT
    {
        NN_RESULT_DO(g_RefDevelopInterface->TerminateApplication());

        // SIGLO-68960
        //  アプリ動画撮影用のメモリが am->ns->pm というパスで、
        //  本来のアプリメモリ予算に返却し終わるまで待機する。
        //  念のため、10 秒程度であきらめるようにしておく。
        for (int i=0; i<100; ++i)
        {
            bool isBoosted = false;
            g_RefDevelopInterface->IsSystemMemoryResourceLimitBoosted(&isBoosted);
            if (!isBoosted)
            {
                break;
            }
            os::SleepThread( TimeSpan::FromMilliSeconds(100) );
        }
        NN_RESULT_SUCCESS;
    }

    Result PrepareLaunchProgramFromHost(ProgramLaunchProperty* pOut, const char* programPath) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->PrepareLaunchProgramFromHost(pOut, sf::InBuffer(programPath, programPath ? std::strlen(programPath) + 1 : 0));
    }

    Result LaunchApplicationForDevelop(os::ProcessId* pOut, ncm::ApplicationId applicationId, int32_t flags) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->LaunchApplication(pOut, applicationId, flags);
    }

    Result LaunchApplicationWithStorageIdForDevelop(
        os::ProcessId* pOut,
        ncm::ApplicationId applicationId,
        int32_t flags,
        ncm::StorageId appStorageId,
        ncm::StorageId patchStorageId) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->LaunchApplicationWithStorageId(
            pOut, applicationId, flags, appStorageId, patchStorageId);
    }

    Result GetRunningApplicationProcessIdForDevelop(os::ProcessId* pOut) NN_NOEXCEPT
    {
        return g_RefDevelopInterface->GetRunningApplicationProcessId(pOut);
    }
}}
