﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/npns/npns_INpns.sfdl.h>
#include <nn/npns/detail/npns_HipcPorts.h>
#include <nn/npns/detail/npns_Utility.h>

#include "npns_Config.h"
#include "npns_ServiceProxy.h"
#include "npns_ServiceImpl.h"

namespace nn { namespace npns {

namespace {
    nn::sf::SharedPointer<INpnsUser>   s_pProxyUser   = nullptr;
    nn::sf::SharedPointer<INpnsSystem> s_pProxySystem = nullptr;

    nn::sf::ShimLibraryObjectHolder<INpnsUser> g_UserHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
    nn::sf::ShimLibraryObjectHolder<INpnsSystem> g_SystemHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
}   // namespace

#if !NN_NPNS_ENABLE_HIPC
Result CreateDfcProxy(PortIndex portIndex) NN_NOEXCEPT
{
    // 1つのみ存在するのでStaticOneAllocationPolicyを利用
    typedef nn::sf::ObjectFactory<nn::sf::impl::StaticOneAllocationPolicy> Factory;

    switch (portIndex)
    {
    case PortIndex_User:
        NN_SDK_REQUIRES(s_pProxyUser == nullptr);
        s_pProxyUser   = Factory::CreateSharedEmplaced<INpnsUser,  ServiceImpl>();
        if (!s_pProxyUser)
        {
            return ResultOutOfMemory();
        }
        g_UserHolder.InitializeHolderDirectly(s_pProxyUser);
        break;

    case PortIndex_System:
        NN_SDK_REQUIRES(s_pProxySystem == nullptr);
        s_pProxySystem = Factory::CreateSharedEmplaced<INpnsSystem, ServiceImpl>();
        if (!s_pProxySystem)
        {
            return ResultOutOfMemory();
        }
        g_SystemHolder.InitializeHolderDirectly(s_pProxySystem);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return ResultSuccess();
}
#else
Result CreateHipcProxy(PortIndex portIndex) NN_NOEXCEPT
{
    const char* pPortName = PortNames[portIndex];

    // 1つのみ生成するので ObjectCountMax=1 で.
    NN_FUNCTION_LOCAL_STATIC(nn::sf::SimpleAllInOneHipcClientManager<1>, g_Manager, = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER);

    switch (portIndex)
    {
    case PortIndex_User:
        {
            NN_SDK_REQUIRES(s_pProxyUser == nullptr);
            Result result = g_Manager.InitializeShimLibraryHolder(&g_UserHolder, pPortName);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            s_pProxyUser = g_UserHolder.GetObject();
        }
        return ResultSuccess();

    case PortIndex_System:
        {
            NN_SDK_REQUIRES(s_pProxySystem == nullptr);
            Result result = g_Manager.InitializeShimLibraryHolder(&g_SystemHolder, pPortName);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            s_pProxySystem = g_SystemHolder.GetObject();
        }
        return ResultSuccess();

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
#endif

void DestroyServiceProxy(PortIndex portIndex) NN_NOEXCEPT
{
    switch (portIndex)
    {
    case PortIndex_User:
        NN_SDK_REQUIRES_NOT_NULL(s_pProxyUser);
        s_pProxyUser = nullptr;
        g_UserHolder.FinalizeHolder();
        break;

    case PortIndex_System:
        NN_SDK_REQUIRES_NOT_NULL(s_pProxySystem);
        s_pProxySystem = nullptr;
        g_SystemHolder.FinalizeHolder();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

INpnsUser& GetServiceProxyUser() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(s_pProxyUser != nullptr || s_pProxySystem != nullptr);
    if (s_pProxySystem)
    {
        return *s_pProxySystem.Get();
    }
    else
    {
        return *s_pProxyUser.Get();
    }
}

INpnsSystem& GetServiceProxySystem() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(s_pProxySystem);
    return *s_pProxySystem.Get();
}

//-----------------------------------------------------------------------------

}}  // namespace nn::npns
