﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/nim/detail/nim_Log.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_RapidJsonInputStream.h>

#include <nn/account/account_ApiForSystemServices.h>
#include "nim_HeapUtil.h"
#include "nim_TokenStore.h"

// 内部デバッグトレースを出力します。
#define ENABLE_DEBUG_TRACE
// ::nn::account::CachedNintendoAccountInfoForSystemService::GetLanguage() から ISO 639-X の言語コード抽出関数を有効化します。
//#define ENABLE_ISO639_LANGUAGE_CODE_EXTRACT_FROM_NA_REGISTORY

namespace nn { namespace nim { namespace srv {

#define PRIVATE_ALIGNED_SIZEOF(TypeName_) ((sizeof(TypeName_) + (std::alignment_of<std::max_align_t>::value - 1)) & (~std::alignment_of<std::max_align_t>::value))

struct ShopServiceAccess
{
    /**
     * @brief       各種共通定数。
     */
    struct Requirement
    {
        /**
         * @brief   アプリケーションID 16進数文字列表現時の文字列長の定義。( null終端文字は含みません )
         */
        static const size_t LengthMaxForStringApplicationId = sizeof(::nn::ncm::ApplicationId) * 2;
    };

    /**
     * @brief       アプリケーションID を16進数文字列に変換します。
     *
     * @param[out]  pOut        変換後文字列格納バッファ先頭要素アドレスへのポインタ。
     * @param[in]   pOutEnd     変換後文字列格納バッファ末尾要素アドレスへのポインタ。@n
     *                          このアドレスが示す領域には書き込まれません。
     *
     * @return      末尾位置(null終端子付与想定位置)を示すアドレスへのポインタを返します。
     *
     * @details     '0x' などのプレフィクスは付与されません。@n
     *              null終端は付与されません。@n
     *              指定アプリケーションIDを正しく16進数文字列化するために、@ref Requirement::LengthMaxForStringApplicationId 以上の出力領域容量が必要です。@n
     *              出力末尾に null終端文字の付与などを独自追加で行う場合は ( @ref Requirement::LengthMaxForStringApplicationId + 1 ) 以上の出力領域容量が必要になる点注意してください。
     */
    static char* ToHexStringFromApplicationId(char* pOut, const char* pOutEnd, const ::nn::ncm::ApplicationId& id) NN_NOEXCEPT;

    /**
     * @brief       指定の値を10進数文字列に変換します。
     *
     * @param[out]  pOut        変換後文字列格納バッファ先頭要素アドレスへのポインタ。
     * @param[in]   pOutEnd     変換後文字列格納バッファ末尾要素アドレスへのポインタ。@n
     *                          このアドレスが示す領域には書き込まれません。
     *
     * @return      末尾位置(null終端子付与位置)を示すアドレスへのポインタを返します。
     *
     * @details     null終端が付与されます。
     * @note        スタック消費したくないので再帰型ではなく逆順配置→反転の手法による実装です。
     */
    static char* ToDecimalStringFrom(char* pOut, const char* pOutEnd, size_t value) NN_NOEXCEPT;

    /**
     * @brief       指定のユーザアカウントIDに紐づくニンテンドーアカウントに登録された情報のシステムキャッシュを取得します。
     *
     * @param[out]  pOut        キャッシュ構造体の先頭アドレスを示すポインタ。
     * @param[in]   uid         取得対象ユーザID。
     *
     * @return      ニンテンドーアカウント情報参照結果。
     *
     * @details     ACCOUNT ライブラリが保持しているキャッシュ情報を参照します。@n
     *              キャッシュ情報の更新要求は行いません。@n
     *              内部で @ref nn::account::NetworkServiceAccountManager::LoadCachedNintendoAccountInfo() を呼び出すため、@n
     *              ::nn::account::RequiredBufferSizeForCachedNintendoAccountInfo サイズのスタックを消費します。
     */
    static Result GetCachedUserProperty(::nn::account::CachedNintendoAccountInfoForSystemService* pOut, const ::nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief       指定のアカウントキャッシュから居住国コード( ISO 3166-1 alpha-2 )を出力します。
     *
     * @param[out]  pOutBytes   居住国コード文字数( null終端文字を含まない )を返します。
     * @param[out]  pOut        居住国コード文字列格納バッファ先頭要素アドレスへのポインタ。
     * @param[in]   pOutEnd     居住国コード文字列格納バッファ末尾要素アドレスへのポインタ。@n
     *                          このアドレスが示す領域には書き込まれません。
     *
     * @return      ニンテンドーアカウント情報参照結果。
     * @retval      nn::ResultSuccess                   処理に成功しました。
     * @retval      nn:::nim::ResultBufferNotEnough     出力先バッファ容量が不足しています。
     *
     * @details     null 終端は付与されません。
     */
    static Result GetUserCountryCode(size_t* pOutBytes, char* pOut, const char* pOutEnd, const ::nn::account::CachedNintendoAccountInfoForSystemService& prop) NN_NOEXCEPT;

#if defined(ENABLE_ISO639_LANGUAGE_CODE_EXTRACT_FROM_NA_REGISTORY)

    /**
     * @brief       指定のアカウントキャッシュから使用言語コード( ISO 639-1 )を出力します。
     *
     * @param[out]  pOutBytes   使用言語コード文字数( null終端文字を含まない )を返します。
     * @param[out]  pOut        使用言語コード文字列格納バッファ先頭要素アドレスへのポインタ。
     * @param[in]   pOutEnd     使用言語コード文字列格納バッファ末尾要素アドレスへのポインタ。@n
     *                          このアドレスが示す領域には書き込まれません。
     *
     * @return      ニンテンドーアカウント情報参照結果。
     * @retval      nn::ResultSuccess                   処理に成功しました。
     * @retval      nn:::nim::ResultBufferNotEnough     出力先バッファ容量が不足しています。
     *
     * @details     null 終端は付与されません。
     */
    static Result GetUserLanguageCode(size_t* pOutBytes, char* pOut, const char* pOutEnd, const ::nn::account::CachedNintendoAccountInfoForSystemService& prop) NN_NOEXCEPT;

#endif  // defined(ENABLE_ISO639_LANGUAGE_CODE_EXTRACT_FROM_NA_REGISTORY)

    /**
     * @brief   ケベック13歳未満制限チェック。
     */
    static Result VerifyQuebecAgeRestriction(const ::nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   一時ヒープセッション。
     */
    typedef ::nn::nim::srv::HeapUtil::OnetimeHeapSession OnetimeHeapSession;

    /**
     * @brief   任意領域メモリアロケータ。
     */
    class HeapAllocator
    {
        NN_DISALLOW_COPY(HeapAllocator);
        NN_DISALLOW_MOVE(HeapAllocator);

    public:
        static const size_t MinimumRequiredCapacity = sizeof(::nn::lmem::HeapCommonHead) + ::nn::lmem::DefaultAlignment;

        explicit HeapAllocator() NN_NOEXCEPT;

        Result Initialize(void* pTop, size_t size) NN_NOEXCEPT;

        bool Finalize() NN_NOEXCEPT;

        void* Allocate(size_t size, int alignment) NN_NOEXCEPT;

        void Deallocate (void* pBlock) NN_NOEXCEPT;

    private:
        ::nn::lmem::HeapHandle m_Handle;
    };

    /**
     * @brief   クライアント譲渡トランスファーメモリストレージ管理
     */
    class TransferStorage
    {
        NN_DISALLOW_COPY(TransferStorage);
        NN_DISALLOW_MOVE(TransferStorage);

    public:
        NN_IMPLICIT TransferStorage() NN_NOEXCEPT : m_pMappedAddress(nullptr), m_MappedSize(0) {}

        Result Initialize(const ::nn::sf::NativeHandle& handle, uint64_t size) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        NN_FORCEINLINE HeapAllocator* GetAllocator() NN_NOEXCEPT
        {
            return &m_BaseAllocator;
        }

    private:
        void CleanMappedMemory() NN_NOEXCEPT;

        HeapAllocator                   m_BaseAllocator;
        ::nn::os::TransferMemoryType    m_TransferHandle;
        void*                           m_pMappedAddress;
        size_t                          m_MappedSize;
    };

    /**
     * @brief   文字列中エイリアスリプレースフレームワーク
     */
    struct Replacer
    {
        /**
         * @brief       リプレース要素検出時のコールバック。
         *
         * @param[out]  pOut        リプレース文字列の出力先先頭要素アドレスへのポインタ。
         * @param[in]   pOutEnd     リプレース文字列の出力先末尾要素アドレスへのポインタ。@n
         *                          このアドレスが示す領域には書き込めません。
         * @param[in]   pAlias      リプレース対象エイリアス文字列( null終端 )の先頭アドレスへのポインタ。
         * @param[in]   pAliasEnd   リプレース対象エイリアス文字列( null終端 )の末尾アドレスへのポインタ。( null終端を示す )
         *
         * @return      @a pOut に出力したリプレース文字列の末尾文字の次のアドレスを示すポインタ。
         */
        typedef char* OnDetectReplaceAliasCallback(char* pOut, const char* pOutEnd, char* pAlias, char* pAliasEnd);

        /**
         * @brief       リプレース実施。
         *
         * @tparam      Predicate           リプレース変換実施メソッド型。( OnDetectReplaceAliasCallback型の引数及び返値を保証するメソッド )
         * @tparam      AliasKeyCapacity    リプレースエイリアス文字の抽出用テンポラリバッファ容量。( バイトサイズ )@n
         *                                  null終端を含めた検出可能文字数を設定します。@n
         *                                  1文字のエイリアスを対象とする場合は、2 を指定します。@n
         *                                  容量を超えるエイリアス文字は捨てられます。最低 2以上を指定する必要があります。
         *
         * @param[out]  pOutBuffer          変換後文字列格納バッファ。
         * @param[in]   bufferCapacity      変換後文字列格納バッファ長。( バイトサイズ )@n
         *                                  null終端を含めたバッファ長を設定します。@n
         * @param[in]   pSource             変換前文字列。( null 終端 )
         * @param[in]   onReplacer          リプレース変換実施メソッド実体。
         *
         * @return      格納バッファに出力された変換後文字列の末尾文字の次のアドレスを示すポインタ。
         *
         * @details     リプレース対象エイリアスキーを見つけたら、@a onReplacer へ渡します。@n
         *              ブレース文字( '{', '}' )のエスケープは提供していません。
         */
        template<typename Predicate, size_t AliasKeyCapacity = 64>
        static const char* Execute(char* const pOutBuffer, const size_t bufferCapacity, const char* const pSource, Predicate onReplacer) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
            NN_SDK_REQUIRES_NOT_NULL(pSource);
            NN_SDK_REQUIRES(bufferCapacity > 0);
            NN_SDK_REQUIRES(AliasKeyCapacity > 1);

            char pAliasKey[AliasKeyCapacity];
            auto pEndAlias = &pAliasKey[AliasKeyCapacity - 1];
            auto pNowAlias = pAliasKey;

            auto pInNow = pSource;
            auto pNow = pOutBuffer;
            auto pEnd = &pOutBuffer[bufferCapacity - 1];

            while (pNow < pEnd)
            {
                auto data = *pInNow++;
                if ('\0' == data)
                {
                    break;
                }
                else if ('{' == data)
                {
                    data = *pInNow++;
                    if ('\0' == data)
                    {
                        break;
                    }
                    else if (pNowAlias == pAliasKey)
                    {
                        *pNowAlias++ = data;
                        continue;   // エイリアス開始を検出
                    }
                }
                else if ('}' == data)
                {
                    data = *pInNow++;
                    if (pNowAlias != pAliasKey)
                    {
                        // リプレース実施。
                        *pNowAlias = '\0';
                        if ((pNow = onReplacer(pNow, pEnd, pAliasKey, pNowAlias)) >= pEnd)
                        {
                            break;
                        }
                        pNowAlias = pAliasKey;
                    }
                    if ('\0' == data)
                    {
                        break;
                    }
                }

                // 抽出文字出力先判定
                if (pNowAlias != pAliasKey)
                {
                    if (pNowAlias < pEndAlias)
                    {
                        *pNowAlias++ = data;
                    }
                }
                else
                {
                    *pNow++ = data;
                }
            }
            *pNow = '\0';
            return pNow;
        }
    };

    /**
     * @brief   時間内要求回数制限管理クラス。
     *
     * @note    キャンセルハンドルは要求単位でインスタンスを保持してください。@n
     *          @ref AcquirePermission() で制限ストールが発生している際に、他スレッドからキャンセルする場合にキャンセルハンドルを利用します。
     */
    class RequestCountRestriction
    {
    private:
        ::nn::os::SdkMutex      m_Lock;
        ::nn::os::TimerEvent    m_Timer;
        ::nn::os::Event         m_Cancel;
        const ::nn::TimeSpan    m_Period;
        const int               m_Limit;
        int                     m_Counter;

    public:
        /**
         * @brief       要求制限解除待ち中断ハンドルコンテナ。
         */
        typedef TokenStore::SignalCanceler<::nn::os::Event> Canceler;

        /**
         * @brief       コンストラクタ。
         *
         * @param[in]   restrictCount   時間内有効要求回数制限最大値。
         * @param[in]   restrictPeriod  制限対象期間。
         */
        explicit RequestCountRestriction(int restrictCount, const ::nn::TimeSpan& restrictPeriod) NN_NOEXCEPT;

        /**
         * @brief       デストラクタ。
         */
        ~RequestCountRestriction() NN_NOEXCEPT;

        /**
         * @brief       制限許可申請。
         *
         * @param[out]  pCanceler   ロック中キャンセルイベントコンテナ。@n
         *                          制限オーバーに伴ってストールする場合、ストール期間中のみキャンセル用のイベントを格納します。
         *
         * @return      許可申請結果を返します。
         * @retval      true        許可申請が受理されました。
         * @retval      false       申請者によって申請がキャンセルされました。
         *
         * @pre
         *              - pCanceler != nullptr
         *
         * @details     制限オーバー時は時間経過まで呼び出し元スレッドをストールします。@n
         *              ストール中に他スレッドから呼び出した場合は、ストール解除まで待機させられます。@n
         *              キャンセルされた場合は時間制限中であってもメソッドから復帰しますが、制限対象処理を実施しない事を想定しています。
         */
        bool AcquirePermission(Canceler* pCanceler) NN_NOEXCEPT;
    };
};

/**
 * @brief   Json SAXパーサ。
 *
 * @details nn::http::json ライブラリを利用します。
 */
struct JsonParser
{
    /**
     * @brief   JSONパス文字列の検索用ユーティリティ
     * @details あるパス文字列で表されるJonPathがこれまで渡されたかどうかを保持します。
     */
    struct LookupEntry
    {
        const char* pPath;
        bool found;

        explicit LookupEntry(const char* p) NN_NOEXCEPT : pPath(p), found(false)
        {
            NN_SDK_ASSERT(p);
        }

        NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
        {
            return found;
        }

        template <typename JsonPathType>
        bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
        {
            return !(*this) && jsonPath.Match(pPath);
        }

        void MarkAccepted() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(!found);
            found = true;
        }
    };

    /**
     * @brief   D4C REST API エラーレスポンスJSONパースアダプタ。
     *
     * @details エラーコードだけ引っこ抜きます。
     */
    template<typename TJsonPath>
    class ErrorReponseAdaptor
    {
    private:
        JsonParser::LookupEntry m_EntryForCode;
        Bit32                   m_Code;

    public:
        /**
         * @brief   レスポンスJson文字列パス構成。
         */
        typedef TJsonPath JsonPathType;

        /**
         * @brief   コンストラクタ。
         */
        explicit ErrorReponseAdaptor(const char* const pErrorCodePath) NN_NOEXCEPT
            : m_EntryForCode(pErrorCodePath)
            , m_Code(0)
        {
        }

        /**
         * @brief   エラーコードの取得。
         */
        Bit32 GetErrorCode() const NN_NOEXCEPT
        {
            return m_Code;
        }

        /**
         * @brief   エラーコード文字列を 32bit整数値に変換します。
         */
        static bool ExtractErrorCodeFromString(Bit32* pOut, const char* pValue, int valueLength) NN_NOEXCEPT
        {
            if ('\0' != pValue[valueLength])
            {
                return false;
            }
            char* pEnd;
            *pOut = std::strtoul(pValue, &pEnd, 10);
            return ('\0' == *pEnd);
        }

        /**
         * @brief   文字列検出ハンドラ。
         *
         * @details パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
         */
        void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
        {
            if (m_EntryForCode.CanAccept(jsonPath))
            {
                if (valueLength <= 0 || !ExtractErrorCodeFromString(&m_Code, pValue, valueLength))
                {
                    return;
                }
                m_EntryForCode.MarkAccepted();
            }
        }

        // 以下検出対象外
        void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
        void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
        void Update(const JsonPathType&, double) NN_NOEXCEPT {}
        void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
        void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}
    };
};

}}}
